<?php
/**
 * Process functions
 *
 * @package WPVulnerability
 *
 * @since 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Load the settings to be available always.
 *
 * @since 2.0.0
 *
 * @return array|false An array containing the WPVulnerability settings if they exist, or false if they don't.
 */
$wpvulnerability_settings = get_site_option( 'wpvulnerability-config' );
$wpvulnerability_analyze  = get_site_option( 'wpvulnerability-analyze' );

/**
 * Enqueues the WPVulnerability admin CSS file on WPVulnerability admin pages.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_enqueue_scripts() {
		// Enqueue the admin stylesheet.
		wp_enqueue_style(
			'wpvulnerability-admin',
			WPVULNERABILITY_PLUGIN_URL . 'assets/admin.css',
			array(),
			WPVULNERABILITY_PLUGIN_VERSION
		);

		wp_enqueue_script(
			'wpvulnerability-admin-js',
			WPVULNERABILITY_PLUGIN_URL . 'assets/admin.js',
			array( 'jquery' ),
			WPVULNERABILITY_PLUGIN_VERSION,
			true
		);

	// Localize script with AJAX URL for multisite network admin.
	wp_localize_script(
		'wpvulnerability-admin-js',
		'wpvulnerabilityAjax',
		array(
			'ajaxurl' => admin_url( 'admin-ajax.php' ),
		)
	);
}
add_action( 'admin_enqueue_scripts', 'wpvulnerability_admin_enqueue_scripts' );


/**
 * Processes the form submission for the WPVulnerability plugin settings in a multisite network admin context.
 *
 * Checks if the current request is a submission from the WPVulnerability settings page.
 * Verifies the security nonce to prevent CSRF attacks. If the verification fails or if the request
 * is not from a network admin or the admin dashboard, the function will halt execution and display an error.
 * Otherwise, it sanitizes and updates the plugin settings, reschedules any relevant wp-cron events,
 * and registers a settings error to notify the user that the information has been updated.
 *
 * @since 3.0.0
 * @return void
 */
function wpvulnerability_process_network_config_forms() {
	// Only process config forms that have the wpvulnerability_submit button.
	// Other action forms (delete logs, reset, etc.) have their own handlers.
	if ( ! isset( $_POST['wpvulnerability_submit'] ) ) {
		return;
	}

	if ( isset( $_GET['page'] ) && 'wpvulnerability-options' === $_GET['page'] && ! empty( $_POST ) ) {

		if ( check_admin_referer( 'wpvulnerability_nonce', 'wpauto_nonce' ) ) {

			if ( isset( $_POST['wpvulnerability-config'] ) ) {

				$post_config = filter_input( INPUT_POST, 'wpvulnerability-config', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY );
				if ( ! is_array( $post_config ) ) {
					$post_config = array();
				}

				$wpvulnerability_sanitized_values = wpvulnerability_sanitize_config( $post_config );

				update_site_option( 'wpvulnerability-config', $wpvulnerability_sanitized_values );

				unset( $wpvulnerability_sanitized_values );

				add_settings_error(
					'wpvulnerability-messages',
					'wpvulnerability-updated',
					__( 'Settings saved.', 'wpvulnerability' ),
					'success'
				);

			}

			if ( isset( $_POST['wpvulnerability_submit'] ) && isset( $_POST['wpvulnerability-analyze'] ) ) {

				$wpvulnerability_sanitized_values = array(
					'core'        => 0,
					'plugins'     => 0,
					'themes'      => 0,
					'php'         => 0,
					'apache'      => 0,
					'nginx'       => 0,
					'mariadb'     => 0,
					'mysql'       => 0,
					'imagemagick' => 0,
					'curl'        => 0,
					'memcached'   => 0,
					'redis'       => 0,
					'sqlite'      => 0,
				);

				$wpvulnerability_values = array_map( 'sanitize_text_field', wp_unslash( $_POST['wpvulnerability-analyze'] ) );

				foreach ( $wpvulnerability_values as $data ) {
					switch ( $data ) {
						case 'core':
							$wpvulnerability_sanitized_values['core'] = 1;
							break;
						case 'plugins':
							$wpvulnerability_sanitized_values['plugins'] = 1;
							break;
						case 'themes':
							$wpvulnerability_sanitized_values['themes'] = 1;
							break;
						case 'php':
							$wpvulnerability_sanitized_values['php'] = 1;
							break;
						case 'apache':
							$wpvulnerability_sanitized_values['apache'] = 1;
							break;
						case 'nginx':
							$wpvulnerability_sanitized_values['nginx'] = 1;
							break;
						case 'mariadb':
							$wpvulnerability_sanitized_values['mariadb'] = 1;
							break;
						case 'mysql':
							$wpvulnerability_sanitized_values['mysql'] = 1;
							break;
						case 'imagemagick':
							$wpvulnerability_sanitized_values['imagemagick'] = 1;
							break;
						case 'curl':
							$wpvulnerability_sanitized_values['curl'] = 1;
							break;
						case 'memcached':
							$wpvulnerability_sanitized_values['memcached'] = 1;
							break;
						case 'redis':
							$wpvulnerability_sanitized_values['redis'] = 1;
							break;
						case 'sqlite':
							$wpvulnerability_sanitized_values['sqlite'] = 1;
							break;
					}
				}

				update_site_option(
					'wpvulnerability-analyze',
					array(
						'core'        => $wpvulnerability_sanitized_values['core'],
						'plugins'     => $wpvulnerability_sanitized_values['plugins'],
						'themes'      => $wpvulnerability_sanitized_values['themes'],
						'php'         => $wpvulnerability_sanitized_values['php'],
						'apache'      => $wpvulnerability_sanitized_values['apache'],
						'nginx'       => $wpvulnerability_sanitized_values['nginx'],
						'mariadb'     => $wpvulnerability_sanitized_values['mariadb'],
						'mysql'       => $wpvulnerability_sanitized_values['mysql'],
						'imagemagick' => $wpvulnerability_sanitized_values['imagemagick'],
						'curl'        => $wpvulnerability_sanitized_values['curl'],
						'memcached'   => $wpvulnerability_sanitized_values['memcached'],
						'redis'       => $wpvulnerability_sanitized_values['redis'],
						'sqlite'      => $wpvulnerability_sanitized_values['sqlite'],
					)
				);

				unset( $wpvulnerability_sanitized_values );

				add_settings_error(
					'wpvulnerability-messages',
					'wpvulnerability-updated',
					__( 'Settings saved.', 'wpvulnerability' ),
					'success'
				);

			}
		}
	}
}
add_action( 'admin_init', 'wpvulnerability_process_network_config_forms' );

/**
 * Process action form submissions (delete logs, reset, test email, etc.).
 *
 * @since 4.3.0
 * @return void
 */
function wpvulnerability_process_network_action_forms() {
	/**
	 * Reset the data
	 *
	 * @since 3.0.0
	 */
	if ( isset( $_POST['wpvulnerability_reset'] ) && check_admin_referer( 'wpvulnerability_reset_action', 'wpvulnerability_reset_nonce' ) ) {

		if ( current_user_can( 'manage_network_options' ) ) {
			// Calls the reset function.
			wpvulnerability_update_database_data();

			set_transient( 'wpvulnerability_message_manual_success', __( 'Data from source has been reloaded.', 'wpvulnerability' ), 10 );
		} else {
			set_transient( 'wpvulnerability_message_manual_error', __( 'You do not have permission to reload data.', 'wpvulnerability' ), 10 );
		}
	}

	/**
	 * Send an test email
	 *
	 * @since 3.0.0
	 */
	if ( isset( $_POST['wpvulnerability_email'] ) && check_admin_referer( 'wpvulnerability_email_action', 'wpvulnerability_email_nonce' ) ) {

		if ( ! function_exists( 'wpvulnerability_execute_notification' ) ) {
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-core.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-plugins.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-themes.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-software.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-process.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-notifications.php';
		}

		// Calls the notifications function, forced.
		if ( current_user_can( 'manage_network_options' ) ) {
			$wpmail = wpvulnerability_execute_notification( true );

			if ( $wpmail ) {

				set_transient( 'wpvulnerability_message_manual_success', __( 'Test email has been sent.', 'wpvulnerability' ), 10 );

			} else {

				set_transient( 'wpvulnerability_message_manual_error', __( 'Test email has failed. Please, check your email settings.', 'wpvulnerability' ), 10 );

			}
		} else {
			set_transient( 'wpvulnerability_message_manual_error', __( 'You do not have permission to send test emails.', 'wpvulnerability' ), 10 );
		}
	}

	/**
	 * Repairs scheduled cron events across the network.
	 *
	 * @since 4.3.0
	 */
	if ( isset( $_POST['wpvulnerability_repair_cron'] ) && check_admin_referer( 'wpvulnerability_cron_action', 'wpvulnerability_cron_nonce' ) ) {
		if ( current_user_can( 'manage_network_options' ) ) {
			$wpvulnerability_settings = get_site_option( 'wpvulnerability-config', array() );
			$cron_config              = is_array( $wpvulnerability_settings ) ? $wpvulnerability_settings : array();
			wpvulnerability_repair_network_cron_events( $cron_config );
			set_transient( 'wpvulnerability_message_manual_success', __( 'WPVulnerability cron events have been repaired across the network.', 'wpvulnerability' ), 10 );
		} else {
			set_transient( 'wpvulnerability_message_manual_error', __( 'You do not have permission to repair cron events.', 'wpvulnerability' ), 10 );
		}
	}

	/**
	 * Delete all stored API logs across the network.
	 *
	 * @since 4.3.0
	 */
	if ( isset( $_POST['wpvulnerability_delete_logs'] ) && check_admin_referer( 'wpvulnerability_delete_logs_action', 'wpvulnerability_delete_logs_nonce' ) ) {
		if ( current_user_can( 'manage_network_options' ) ) {
			wpvulnerability_delete_all_logs();
			set_transient( 'wpvulnerability_message_manual_success', __( 'All logs have been deleted.', 'wpvulnerability' ), 10 );
		} else {
			set_transient( 'wpvulnerability_message_manual_error', __( 'You do not have permission to delete logs.', 'wpvulnerability' ), 10 );
		}
	}

	/**
	 * Fully resets plugin data, settings, and cached API content across the network.
	 *
	 * @since 4.3.0
	 */
	if ( isset( $_POST['wpvulnerability_full_reset'] ) && check_admin_referer( 'wpvulnerability_full_reset_action', 'wpvulnerability_full_reset_nonce' ) ) {
		if ( current_user_can( 'manage_network_options' ) ) {
			wpvulnerability_reset_plugin_data();
			set_transient( 'wpvulnerability_message_manual_success', __( 'WPVulnerability has been reset to defaults and reloaded.', 'wpvulnerability' ), 10 );
		} else {
			set_transient( 'wpvulnerability_message_manual_error', __( 'You do not have permission to reset WPVulnerability.', 'wpvulnerability' ), 10 );
		}
	}

	/**
	 * Handles debug action: Clear all caches.
	 *
	 * @since 4.3.0
	 */
	if ( isset( $_POST['wpvulnerability_debug_clear_caches'] ) && check_admin_referer( 'wpvulnerability_debug_action', 'wpvulnerability_debug_nonce' ) ) {
		if ( current_user_can( 'manage_network_options' ) ) {
			if ( ! function_exists( 'wpvulnerability_debug_clear_all_caches' ) ) {
				require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-debug.php';
			}
			wpvulnerability_debug_clear_all_caches();
			set_transient( 'wpvulnerability_message_manual_success', __( 'All caches have been cleared.', 'wpvulnerability' ), 10 );
		} else {
			set_transient( 'wpvulnerability_message_manual_error', __( 'You do not have permission to clear caches.', 'wpvulnerability' ), 10 );
		}
	}

	/**
	 * Handles debug action: Reset signatures.
	 *
	 * @since 4.3.0
	 */
	if ( isset( $_POST['wpvulnerability_debug_reset_signatures'] ) && check_admin_referer( 'wpvulnerability_debug_action', 'wpvulnerability_debug_nonce' ) ) {
		if ( current_user_can( 'manage_network_options' ) ) {
			if ( ! function_exists( 'wpvulnerability_debug_reset_signatures' ) ) {
				require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-debug.php';
			}
			wpvulnerability_debug_reset_signatures();
			set_transient( 'wpvulnerability_message_manual_success', __( 'Plugin and theme signatures have been reset.', 'wpvulnerability' ), 10 );
		} else {
			set_transient( 'wpvulnerability_message_manual_error', __( 'You do not have permission to reset signatures.', 'wpvulnerability' ), 10 );
		}
	}

	/**
	 * Handles debug action: Export debug info.
	 *
	 * @since 4.3.0
	 */
	if ( isset( $_POST['wpvulnerability_debug_export'] ) && check_admin_referer( 'wpvulnerability_debug_action', 'wpvulnerability_debug_nonce' ) ) {
		if ( current_user_can( 'manage_network_options' ) ) {
			if ( ! function_exists( 'wpvulnerability_debug_export_info' ) ) {
				require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-debug.php';
			}
			$wpvulnerability_debug_info = wpvulnerability_debug_export_info();
			$wpvulnerability_filename   = 'wpvulnerability-debug-' . gmdate( 'Y-m-d-His' ) . '.json';

			header( 'Content-Type: application/json' );
			header( 'Content-Disposition: attachment; filename="' . $wpvulnerability_filename . '"' );
			header( 'Content-Length: ' . strlen( $wpvulnerability_debug_info ) );
			echo $wpvulnerability_debug_info; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			exit;
		} else {
			set_transient( 'wpvulnerability_message_manual_error', __( 'You do not have permission to export debug information.', 'wpvulnerability' ), 10 );
		}
	}

	/**
	 * Handles debug action: Run update database now.
	 *
	 * @since 4.3.0
	 */
	if ( isset( $_POST['wpvulnerability_run_update'] ) && check_admin_referer( 'wpvulnerability_cron_action', 'wpvulnerability_cron_nonce' ) ) {
		if ( current_user_can( 'manage_network_options' ) ) {
			wpvulnerability_update_database_data();
			set_transient( 'wpvulnerability_message_manual_success', __( 'Database update has been executed.', 'wpvulnerability' ), 10 );
		} else {
			set_transient( 'wpvulnerability_message_manual_error', __( 'You do not have permission to run database updates.', 'wpvulnerability' ), 10 );
		}
	}

	/**
	 * Handles debug action: Run notification now.
	 *
	 * @since 4.3.0
	 */
	if ( isset( $_POST['wpvulnerability_run_notification'] ) && check_admin_referer( 'wpvulnerability_cron_action', 'wpvulnerability_cron_nonce' ) ) {
		if ( current_user_can( 'manage_network_options' ) ) {
			if ( ! function_exists( 'wpvulnerability_execute_notification' ) ) {
				require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-core.php';
				require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-plugins.php';
				require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-themes.php';
				require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-software.php';
				require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-process.php';
				require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-notifications.php';
			}
			$wpvulnerability_result = wpvulnerability_execute_notification( true );
			if ( $wpvulnerability_result ) {
				set_transient( 'wpvulnerability_message_manual_success', __( 'Notification has been sent.', 'wpvulnerability' ), 10 );
			} else {
				set_transient( 'wpvulnerability_message_manual_error', __( 'Notification sending failed.', 'wpvulnerability' ), 10 );
			}
		} else {
			set_transient( 'wpvulnerability_message_manual_error', __( 'You do not have permission to send notifications.', 'wpvulnerability' ), 10 );
		}
	}
}
add_action( 'admin_init', 'wpvulnerability_process_network_action_forms' );

/**
 * Create the WP-Admin options page
 * This function generates the HTML output for the WPVulnerability settings page in the WP-Admin.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_create_admin_page() {
	?>
	<div class="header-wrap">
		<div class="wrapper">
			<div class="header wpvulnerability-header">
				<div class="logo">
								<img src="<?php echo esc_url( WPVULNERABILITY_PLUGIN_URL ); ?>assets/icon.svg" style="height: 64px; vertical-align: text-top; width: 64px;" alt="" title="WPVulnerability">
					<h2><?php esc_html_e( 'WPVulnerability settings', 'wpvulnerability' ); ?></h2>
				</div>
			</div>
		</div>
	</div>
	<?php
	$wpvulnerability_message_manual_success = get_transient( 'wpvulnerability_message_manual_success' );
	if ( $wpvulnerability_message_manual_success ) {
		echo '<div class="notice notice-success"><p>' . esc_html( (string) $wpvulnerability_message_manual_success ) . '</p></div>';
		delete_transient( 'wpvulnerability_message_manual_success' );
		unset( $wpvulnerability_message_manual_success );
	}
	$wpvulnerability_message_manual_error = get_transient( 'wpvulnerability_message_manual_error' );
	if ( $wpvulnerability_message_manual_error ) {
		echo '<div class="notice notice-error"><p>' . esc_html( (string) $wpvulnerability_message_manual_error ) . '</p></div>';
		delete_transient( 'wpvulnerability_message_manual_error' );
		unset( $wpvulnerability_message_manual_error );
	}
	?>
		<?php settings_errors( 'wpvulnerability-messages', true ); ?>

		<?php
		$tabs = wpvulnerability_get_network_admin_tabs();

		if ( empty( $tabs ) ) {
			return;
		}

		$current_tab = wpvulnerability_get_current_network_admin_tab( $tabs );

		if ( ! isset( $tabs[ $current_tab ] ) ) {
			$tab_keys    = array_keys( $tabs );
			$current_tab = reset( $tab_keys );
		}

		?>
	<div class="wrap">
		<div class="wpvulnerability-settings">
			<h2 class="nav-tab-wrapper wpvulnerability-tab-nav" role="tablist">
				<?php
				foreach ( $tabs as $tab_slug => $tab_data ) {
					$tab_label = isset( $tab_data['label'] ) ? $tab_data['label'] : '';
					$is_active = ( $tab_slug === $current_tab );
					$tab_url   = add_query_arg(
						array(
							'page' => 'wpvulnerability-options',
							'tab'  => $tab_slug,
						),
						network_admin_url( 'settings.php' )
					);
					$tab_class = 'nav-tab wpvulnerability-tab-link';
					if ( $is_active ) {
						$tab_class .= ' nav-tab-active';
					}
					?>
					<a
						href="<?php echo esc_url( $tab_url ); ?>"
						class="<?php echo esc_attr( $tab_class ); ?>"
						id="<?php echo esc_attr( 'wpvulnerability-network-tab-link-' . $tab_slug ); ?>"
						role="tab"
						aria-controls="<?php echo esc_attr( 'wpvulnerability-network-tab-panel-' . $tab_slug ); ?>"
						aria-selected="<?php echo $is_active ? 'true' : 'false'; ?>"
						<?php
						if ( ! $is_active ) :
							?>
							tabindex="-1"<?php endif; ?>
					>
						<?php echo esc_html( $tab_label ); ?>
					</a>
					<?php
				}
				?>
			</h2>
			<div
				id="<?php echo esc_attr( 'wpvulnerability-network-tab-panel-' . $current_tab ); ?>"
				class="wpvulnerability-tab-panel is-active"
				role="tabpanel"
				aria-labelledby="<?php echo esc_attr( 'wpvulnerability-network-tab-link-' . $current_tab ); ?>"
				tabindex="0"
			>
				<?php wpvulnerability_render_network_admin_tab( $current_tab ); ?>
			</div>
		</div>
	</div>
	<?php
}


/**
 * Retrieves the available tabs for the multisite admin settings page.
 *
 * @since 4.1.2
 *
 * @return array<string, array<string, string>> An associative array of tab slugs and their labels.
 */
function wpvulnerability_get_network_admin_tabs() {
	$tabs = array(
		'notifications' => array(
			'label' => __( 'Notifications', 'wpvulnerability' ),
		),
		'analysis'      => array(
			'label' => __( 'Analysis', 'wpvulnerability' ),
		),
		'logs'          => array(
			'label' => __( 'Logs', 'wpvulnerability' ),
		),
		'security'      => array(
			'label' => __( 'Security', 'wpvulnerability' ),
		),
		'tools'         => array(
			'label' => __( 'Tools', 'wpvulnerability' ),
		),
	);

	// Add Debug tab only if WP_DEBUG is enabled.
	if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		$tabs['debug'] = array(
			'label' => __( 'Debug', 'wpvulnerability' ),
		);
	}

	$tabs['about'] = array(
		'label' => __( 'About', 'wpvulnerability' ),
	);

	return $tabs;
}

/**
 * Determines the active multisite admin tab.
 *
 * @since 4.1.2
 *
 * @param array<string, array<string, string>> $tabs Registered multisite admin tabs.
 *
 * @return string The active tab slug.
 */
function wpvulnerability_get_current_network_admin_tab( $tabs ) {
	$tab_keys   = array_keys( $tabs );
	$default    = reset( $tab_keys );
	$tab_filter = filter_input( INPUT_GET, 'tab', FILTER_SANITIZE_SPECIAL_CHARS );

	if ( $tab_filter ) {
		$tab_filter = sanitize_key( $tab_filter );
	}

	if ( $tab_filter && isset( $tabs[ $tab_filter ] ) ) {
		return $tab_filter;
	}

	return $default ? $default : 'notifications';
}

/**
 * Renders the requested multisite admin tab content.
 *
 * @since 4.1.2
 *
 * @param string $tab Tab slug to render.
 *
 * @return void
 */
function wpvulnerability_render_network_admin_tab( $tab ) {
	switch ( $tab ) {
		case 'analysis':
			wpvulnerability_render_network_admin_tab_analysis();
			break;
		case 'logs':
			wpvulnerability_render_network_admin_tab_logs();
			break;
		case 'security':
			wpvulnerability_render_network_admin_tab_security();
			break;
		case 'tools':
			wpvulnerability_render_network_admin_tab_tools();
			break;
		case 'debug':
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				wpvulnerability_render_network_admin_tab_debug();
			}
			break;
		case 'about':
			wpvulnerability_render_network_admin_tab_about();
			break;
		case 'notifications':
		default:
			wpvulnerability_render_network_admin_tab_notifications();
			break;
	}
}

/**
 * Outputs the Notifications tab contents for the multisite settings screen.
 *
 * @since 4.1.2
 *
 * @return void
 */
function wpvulnerability_render_network_admin_tab_notifications() {
	$wpvulnerability_settings = get_site_option( 'wpvulnerability-config', array() );
	$defaults                 = array(
		'cache'              => 12,
		'period'             => 'weekly',
		'day'                => 'monday',
		'hour'               => 0,
		'minute'             => 0,
		'emails'             => '',
		'slack_webhook'      => '',
		'teams_webhook'      => '',
		'discord_webhook'    => '',
		'telegram_bot_token' => '',
		'telegram_chat_id'   => '',
		'notify'             => array(
			'email'    => 'y',
			'slack'    => 'n',
			'teams'    => 'n',
			'discord'  => 'n',
			'telegram' => 'n',
		),
	);
	$wpvulnerability_settings = wp_parse_args( $wpvulnerability_settings, $defaults );

	// Normalize notify settings.
	if ( ! is_array( $wpvulnerability_settings['notify'] ) ) {
		$wpvulnerability_settings['notify'] = $defaults['notify'];
	} else {
		$wpvulnerability_settings['notify'] = wp_parse_args( $wpvulnerability_settings['notify'], $defaults['notify'] );
	}
	$wpvulnerability_settings['notify'] = wpvulnerability_normalize_notify_settings( $wpvulnerability_settings['notify'] );

	$email_enabled    = wpvulnerability_is_yes( $wpvulnerability_settings['notify']['email'] );
	$slack_enabled    = wpvulnerability_is_yes( $wpvulnerability_settings['notify']['slack'] );
	$teams_enabled    = wpvulnerability_is_yes( $wpvulnerability_settings['notify']['teams'] );
	$discord_enabled  = wpvulnerability_is_yes( $wpvulnerability_settings['notify']['discord'] );
	$telegram_enabled = wpvulnerability_is_yes( $wpvulnerability_settings['notify']['telegram'] );

	// Check if cache is forced.
	$forced_cache  = null;
	$cache_options = array( 1, 6, 12, 24 );
	if ( defined( 'WPVULNERABILITY_CACHE_HOURS' ) ) {
		$forced_cache = (int) WPVULNERABILITY_CACHE_HOURS;
		if ( ! in_array( $forced_cache, $cache_options, true ) ) {
			$cache_options[] = $forced_cache;
			sort( $cache_options, SORT_NUMERIC );
		}
	}
	$current_cache = null !== $forced_cache ? $forced_cache : (int) $wpvulnerability_settings['cache'];

	$admin_email = get_site_option( 'admin_email' );
	?>
	<section class="section wpvulnerability-notifications-panel">

		<form method="post" action="">
			<?php wp_nonce_field( 'wpvulnerability_nonce', 'wpauto_nonce' ); ?>

			<div class="wpvulnerability-security-section">
				<h3><?php esc_html_e( 'Notification Settings', 'wpvulnerability' ); ?></h3>

				<div class="wpvulnerability-intro">
					<p><strong><?php esc_html_e( 'Configure how and when you want to receive vulnerability notifications.', 'wpvulnerability' ); ?></strong></p>
					<p><?php esc_html_e( 'Stay informed about security vulnerabilities in your WordPress installation, plugins, themes, and server software.', 'wpvulnerability' ); ?></p>
				</div>

				<!-- Cache Settings -->
				<div class="wpvulnerability-setting-group">
					<div class="wpvulnerability-setting-label">
						<span class="wpvulnerability-setting-icon">⏱️</span>
						<?php esc_html_e( 'Cache Expiration Time', 'wpvulnerability' ); ?>
						<?php if ( null !== $forced_cache ) : ?>
							<span class="wpvulnerability-forced-badge"><?php esc_html_e( 'FORCED', 'wpvulnerability' ); ?></span>
						<?php endif; ?>
					</div>
					<div class="wpvulnerability-setting-description">
						<?php esc_html_e( 'How long to cache vulnerability data before refreshing from the API.', 'wpvulnerability' ); ?>
					</div>
					<select name="wpvulnerability-config[cache]" id="wpvulnerability_cache" <?php disabled( null !== $forced_cache ); ?>>
						<?php foreach ( $cache_options as $hours ) : ?>
							<option value="<?php echo esc_attr( $hours ); ?>" <?php selected( $current_cache, $hours ); ?>>
								<?php
								echo esc_html(
									sprintf(
										/* translators: %d: number of hours */
										_n( '%d hour', '%d hours', $hours, 'wpvulnerability' ),
										$hours
									)
								);
								?>
							</option>
						<?php endforeach; ?>
					</select>
					<?php if ( null !== $forced_cache ) : ?>
						<input type="hidden" name="wpvulnerability-config[cache]" value="<?php echo esc_attr( $current_cache ); ?>" />
					<?php endif; ?>
					<div class="wpvulnerability-info-box">
						<p>
							<?php
							printf(
								/* translators: %s: documentation URL */
								wp_kses_post( __( 'You can force the cache time via wp-config.php constant. <a href="%s" target="_blank" rel="noopener noreferrer">Learn more →</a>', 'wpvulnerability' ) ),
								esc_url( 'https://www.wpvulnerability.com/plugin/#cache-duration' )
							);
							?>
						</p>
					</div>
				</div>

				<!-- Notification Frequency -->
				<div class="wpvulnerability-setting-group">
					<div class="wpvulnerability-setting-label">
						<span class="wpvulnerability-setting-icon">📅</span>
						<?php esc_html_e( 'Notification Frequency', 'wpvulnerability' ); ?>
					</div>
					<div class="wpvulnerability-setting-description">
						<?php esc_html_e( 'Choose how often you want to receive vulnerability notifications.', 'wpvulnerability' ); ?>
					</div>
					<div class="wpvulnerability-radio-group">
						<label>
							<input type="radio" name="wpvulnerability-config[period]" value="never" <?php checked( $wpvulnerability_settings['period'], 'never' ); ?> onchange="wpvUpdateScheduleVisibility()" />
							<?php esc_html_e( 'Never - Disable automatic notifications', 'wpvulnerability' ); ?>
						</label>
						<label>
							<input type="radio" name="wpvulnerability-config[period]" value="daily" <?php checked( $wpvulnerability_settings['period'], 'daily' ); ?> onchange="wpvUpdateScheduleVisibility()" />
							<?php esc_html_e( 'Daily - Receive notifications every day', 'wpvulnerability' ); ?>
						</label>
						<label>
							<input type="radio" name="wpvulnerability-config[period]" value="weekly" <?php checked( $wpvulnerability_settings['period'], 'weekly' ); ?> onchange="wpvUpdateScheduleVisibility()" />
							<?php esc_html_e( 'Weekly - Receive notifications once a week', 'wpvulnerability' ); ?>
						</label>
					</div>

					<div class="wpvulnerability-schedule-controls" id="wpvulnerability-schedule-controls">
						<div class="wpvulnerability-schedule-row" id="wpvulnerability-day-selector">
							<label for="wpvulnerability_day"><?php esc_html_e( 'Day:', 'wpvulnerability' ); ?></label>
							<select name="wpvulnerability-config[day]" id="wpvulnerability_day">
								<option value="monday" <?php selected( $wpvulnerability_settings['day'], 'monday' ); ?>><?php esc_html_e( 'Monday', 'wpvulnerability' ); ?></option>
								<option value="tuesday" <?php selected( $wpvulnerability_settings['day'], 'tuesday' ); ?>><?php esc_html_e( 'Tuesday', 'wpvulnerability' ); ?></option>
								<option value="wednesday" <?php selected( $wpvulnerability_settings['day'], 'wednesday' ); ?>><?php esc_html_e( 'Wednesday', 'wpvulnerability' ); ?></option>
								<option value="thursday" <?php selected( $wpvulnerability_settings['day'], 'thursday' ); ?>><?php esc_html_e( 'Thursday', 'wpvulnerability' ); ?></option>
								<option value="friday" <?php selected( $wpvulnerability_settings['day'], 'friday' ); ?>><?php esc_html_e( 'Friday', 'wpvulnerability' ); ?></option>
								<option value="saturday" <?php selected( $wpvulnerability_settings['day'], 'saturday' ); ?>><?php esc_html_e( 'Saturday', 'wpvulnerability' ); ?></option>
								<option value="sunday" <?php selected( $wpvulnerability_settings['day'], 'sunday' ); ?>><?php esc_html_e( 'Sunday', 'wpvulnerability' ); ?></option>
							</select>
						</div>
						<div class="wpvulnerability-schedule-row">
							<label for="wpvulnerability_hour"><?php esc_html_e( 'Time:', 'wpvulnerability' ); ?></label>
							<input type="number" min="0" max="23" name="wpvulnerability-config[hour]" id="wpvulnerability_hour" value="<?php echo esc_attr( (string) $wpvulnerability_settings['hour'] ); ?>" />
							<span>:</span>
							<input type="number" min="0" max="59" name="wpvulnerability-config[minute]" id="wpvulnerability_minute" value="<?php echo esc_attr( (string) $wpvulnerability_settings['minute'] ); ?>" />
							<span class="wpvulnerability-input-hint"><?php esc_html_e( '(24-hour format, server timezone)', 'wpvulnerability' ); ?></span>
						</div>
					</div>
				</div>

				<!-- Notification Channels -->
				<div class="wpvulnerability-setting-group">
					<div class="wpvulnerability-setting-label">
						<span class="wpvulnerability-setting-icon">📢</span>
						<?php esc_html_e( 'Notification Channels', 'wpvulnerability' ); ?>
					</div>
					<div class="wpvulnerability-setting-description">
						<?php esc_html_e( 'Select where you want to receive notifications.', 'wpvulnerability' ); ?>
					</div>
					<div class="wpvulnerability-checkbox-group">
						<label>
							<input type="checkbox" name="wpvulnerability-config[notify][email]" value="y" <?php checked( $email_enabled ); ?> onchange="wpvToggleChannelInput('email')" />
							<?php esc_html_e( 'Email', 'wpvulnerability' ); ?>
						</label>
						<div class="wpvulnerability-channel-inputs" id="wpvulnerability-email-inputs">
							<label for="wpvulnerability_emails"><?php esc_html_e( 'Email Addresses (separated by commas):', 'wpvulnerability' ); ?></label>
							<input class="wpvulnerability-input-full" type="text" name="wpvulnerability-config[emails]" id="wpvulnerability_emails" placeholder="<?php echo esc_attr( (string) $admin_email ); ?>" value="<?php echo esc_attr( (string) $wpvulnerability_settings['emails'] ); ?>" />
							<span class="wpvulnerability-input-hint"><?php esc_html_e( 'Default:', 'wpvulnerability' ); ?> <?php echo esc_html( (string) $admin_email ); ?></span>
						</div>

						<label>
							<input type="checkbox" name="wpvulnerability-config[notify][slack]" value="y" <?php checked( $slack_enabled ); ?> onchange="wpvToggleChannelInput('slack')" />
							<?php esc_html_e( 'Slack', 'wpvulnerability' ); ?>
						</label>
						<div class="wpvulnerability-channel-inputs" id="wpvulnerability-slack-inputs">
							<label for="wpvulnerability_slack_webhook"><?php esc_html_e( 'Slack Webhook URL:', 'wpvulnerability' ); ?></label>
							<input class="wpvulnerability-input-full" type="url" name="wpvulnerability-config[slack_webhook]" id="wpvulnerability_slack_webhook" placeholder="https://hooks.slack.com/services/..." value="<?php echo esc_attr( (string) $wpvulnerability_settings['slack_webhook'] ); ?>" />
							<span class="wpvulnerability-input-hint">
								<?php
								printf(
									/* translators: %s: Slack documentation URL */
									wp_kses_post( __( '<a href="%s" target="_blank" rel="noopener noreferrer">Learn how to create a Slack incoming webhook →</a>', 'wpvulnerability' ) ),
									esc_url( 'https://docs.slack.dev/messaging/sending-messages-using-incoming-webhooks/' )
								);
								?>
							</span>
						</div>

						<label>
							<input type="checkbox" name="wpvulnerability-config[notify][teams]" value="y" <?php checked( $teams_enabled ); ?> onchange="wpvToggleChannelInput('teams')" />
							<?php esc_html_e( 'Microsoft Teams', 'wpvulnerability' ); ?>
						</label>
						<div class="wpvulnerability-channel-inputs" id="wpvulnerability-teams-inputs">
							<label for="wpvulnerability_teams_webhook"><?php esc_html_e( 'Teams Webhook URL:', 'wpvulnerability' ); ?></label>
							<input class="wpvulnerability-input-full" type="url" name="wpvulnerability-config[teams_webhook]" id="wpvulnerability_teams_webhook" placeholder="https://outlook.office.com/webhook/..." value="<?php echo esc_attr( (string) $wpvulnerability_settings['teams_webhook'] ); ?>" />
							<span class="wpvulnerability-input-hint">
								<?php
								printf(
									/* translators: %s: Microsoft Teams documentation URL */
									wp_kses_post( __( '<a href="%s" target="_blank" rel="noopener noreferrer">Learn how to create a Teams incoming webhook →</a>', 'wpvulnerability' ) ),
									esc_url( 'https://learn.microsoft.com/microsoftteams/platform/webhooks-and-connectors/how-to/add-incoming-webhook' )
								);
								?>
							</span>
						</div>

					<label>
						<input type="checkbox" name="wpvulnerability-config[notify][discord]" value="y" <?php checked( $discord_enabled ); ?> onchange="wpvToggleChannelInput('discord')" />
						<?php esc_html_e( 'Discord', 'wpvulnerability' ); ?>
					</label>
					<div class="wpvulnerability-channel-inputs" id="wpvulnerability-discord-inputs">
						<label for="wpvulnerability_discord_webhook"><?php esc_html_e( 'Discord Webhook URL:', 'wpvulnerability' ); ?></label>
						<input class="wpvulnerability-input-full" type="url" name="wpvulnerability-config[discord_webhook]" id="wpvulnerability_discord_webhook" placeholder="https://discord.com/api/webhooks/..." value="<?php echo esc_attr( (string) $wpvulnerability_settings['discord_webhook'] ); ?>" />
						<span class="wpvulnerability-input-hint">
							<?php
							printf(
								/* translators: %s: Discord documentation URL */
								wp_kses_post( __( '<a href="%s" target="_blank" rel="noopener noreferrer">Learn how to create a Discord webhook →</a>', 'wpvulnerability' ) ),
								esc_url( 'https://support.discord.com/hc/articles/228383668' )
							);
							?>
						</span>
					</div>

					<label>
						<input type="checkbox" name="wpvulnerability-config[notify][telegram]" value="y" <?php checked( $telegram_enabled ); ?> onchange="wpvToggleChannelInput('telegram')" />
						<?php esc_html_e( 'Telegram', 'wpvulnerability' ); ?>
					</label>
					<div class="wpvulnerability-channel-inputs" id="wpvulnerability-telegram-inputs">
						<label for="wpvulnerability_telegram_bot_token"><?php esc_html_e( 'Telegram Bot Token:', 'wpvulnerability' ); ?></label>
						<input class="wpvulnerability-input-full" type="text" name="wpvulnerability-config[telegram_bot_token]" id="wpvulnerability_telegram_bot_token" placeholder="123456:ABC-DEF1234ghIkl-zyx57W2v1u123ew11" value="<?php echo esc_attr( (string) $wpvulnerability_settings['telegram_bot_token'] ); ?>" />
						<span class="wpvulnerability-input-hint"><?php esc_html_e( 'Format: 123456:ABC-DEF1234ghIkl-zyx57W2v1u123ew11', 'wpvulnerability' ); ?></span>

						<label for="wpvulnerability_telegram_chat_id" style="margin-top: 12px;"><?php esc_html_e( 'Telegram Chat ID:', 'wpvulnerability' ); ?></label>
						<input class="wpvulnerability-input-full" type="text" name="wpvulnerability-config[telegram_chat_id]" id="wpvulnerability_telegram_chat_id" placeholder="-1001234567890" value="<?php echo esc_attr( (string) $wpvulnerability_settings['telegram_chat_id'] ); ?>" />
						<span class="wpvulnerability-input-hint">
							<?php
							printf(
								/* translators: %s: Telegram documentation URL */
								wp_kses_post( __( '<a href="%s" target="_blank" rel="noopener noreferrer">Learn how to create a Telegram bot and get Chat ID →</a>', 'wpvulnerability' ) ),
								esc_url( 'https://core.telegram.org/bots' )
							);
							?>
						</span>
					</div>
					</div>
				</div>

				<div class="wpvulnerability-save-section">
					<?php submit_button( __( 'Save Notification Settings', 'wpvulnerability' ), 'primary', 'wpvulnerability_submit', false ); ?>
				</div>
			</div>
		</form>

		<script>
		function wpvUpdateScheduleVisibility() {
			var period = document.querySelector('input[name="wpvulnerability-config[period]"]:checked').value;
			var scheduleControls = document.getElementById('wpvulnerability-schedule-controls');
			var daySelector = document.getElementById('wpvulnerability-day-selector');

			if (period === 'never') {
				scheduleControls.classList.add('wpvulnerability-hidden');
			} else {
				scheduleControls.classList.remove('wpvulnerability-hidden');
				if (period === 'weekly') {
					daySelector.style.display = 'block';
				} else {
					daySelector.style.display = 'none';
				}
			}
		}

		function wpvToggleChannelInput(channel) {
			var checkbox = document.querySelector('input[name="wpvulnerability-config[notify][' + channel + ']"]');
			var inputs = document.getElementById('wpvulnerability-' + channel + '-inputs');

			if (checkbox.checked) {
				inputs.classList.remove('wpvulnerability-hidden');
			} else {
				inputs.classList.add('wpvulnerability-hidden');
			}
		}

		// Initialize visibility on page load.
		document.addEventListener('DOMContentLoaded', function() {
			wpvUpdateScheduleVisibility();
			wpvToggleChannelInput('email');
			wpvToggleChannelInput('slack');
			wpvToggleChannelInput('teams');
			wpvToggleChannelInput('discord');
			wpvToggleChannelInput('telegram');
		});
		</script>
	</section>
	<?php
}

/**
 * Outputs the Analysis tab contents for the multisite settings screen.
 *
 * @since 4.1.2
 *
 * @return void
 */
function wpvulnerability_render_network_admin_tab_analysis() {
	$wpvulnerability_analyze = get_site_option( 'wpvulnerability-analyze', array() );
	$components              = array( 'core', 'plugins', 'themes', 'php', 'apache', 'nginx', 'mariadb', 'mysql', 'imagemagick', 'curl', 'memcached', 'redis', 'sqlite' );
	$forced                  = array();

	foreach ( $components as $component ) {
		if ( ! isset( $wpvulnerability_analyze[ $component ] ) ) {
			$wpvulnerability_analyze[ $component ] = 0;
		}
		$constant             = 'WPVULNERABILITY_HIDE_' . strtoupper( (string) $component );
		$forced[ $component ] = defined( $constant ) && constant( $constant );
		if ( $forced[ $component ] ) {
			$wpvulnerability_analyze[ $component ] = 1;
		}
	}

	// Component configuration with labels and icons.
	$component_config = array(
		'wordpress'  => array(
			'label' => __( 'WordPress Components', 'wpvulnerability' ),
			'items' => array(
				'core'    => array(
					'label' => __( 'WordPress Core', 'wpvulnerability' ),
					'icon'  => '🌐',
				),
				'plugins' => array(
					'label' => __( 'Plugins', 'wpvulnerability' ),
					'icon'  => '🧩',
				),
				'themes'  => array(
					'label' => __( 'Themes', 'wpvulnerability' ),
					'icon'  => '🎨',
				),
			),
		),
		'software'   => array(
			'label' => __( 'Software & Languages', 'wpvulnerability' ),
			'items' => array(
				'php' => array(
					'label' => __( 'PHP', 'wpvulnerability' ),
					'icon'  => '🐘',
				),
			),
		),
		'webservers' => array(
			'label' => __( 'Web Servers', 'wpvulnerability' ),
			'items' => array(
				'apache' => array(
					'label' => __( 'Apache HTTPD', 'wpvulnerability' ),
					'icon'  => '🪶',
				),
				'nginx'  => array(
					'label' => __( 'nginx', 'wpvulnerability' ),
					'icon'  => '🟩',
				),
			),
		),
		'databases'  => array(
			'label' => __( 'Databases', 'wpvulnerability' ),
			'items' => array(
				'mariadb' => array(
					'label' => __( 'MariaDB', 'wpvulnerability' ),
					'icon'  => '🐬',
				),
				'mysql'   => array(
					'label' => __( 'MySQL', 'wpvulnerability' ),
					'icon'  => '🐬',
				),
				'sqlite'  => array(
					'label' => __( 'SQLite', 'wpvulnerability' ),
					'icon'  => '💾',
				),
			),
		),
		'tools'      => array(
			'label' => __( 'Additional Tools', 'wpvulnerability' ),
			'items' => array(
				'imagemagick' => array(
					'label' => __( 'ImageMagick', 'wpvulnerability' ),
					'icon'  => '🖼️',
				),
				'curl'        => array(
					'label' => __( 'curl', 'wpvulnerability' ),
					'icon'  => '🌐',
				),
				'memcached'   => array(
					'label' => __( 'memcached', 'wpvulnerability' ),
					'icon'  => '⚡',
				),
				'redis'       => array(
					'label' => __( 'Redis', 'wpvulnerability' ),
					'icon'  => '🔴',
				),
			),
		),
	);
	?>
	<section class="section wpvulnerability-analysis-panel">

		<form method="post" action="">
			<?php wp_nonce_field( 'wpvulnerability_nonce', 'wpauto_nonce' ); ?>

			<div class="wpvulnerability-security-section">
				<h3><?php esc_html_e( 'Component Analysis Settings', 'wpvulnerability' ); ?></h3>

				<div class="wpvulnerability-intro">
					<p><strong><?php esc_html_e( 'Enable or disable vulnerability analysis for specific components.', 'wpvulnerability' ); ?></strong></p>
					<p><?php esc_html_e( 'Check the components you want to HIDE from vulnerability scans and reports. Unchecked components will be actively analyzed. Components with active analysis are shown in green, hidden components in red.', 'wpvulnerability' ); ?></p>
				</div>

				<?php foreach ( $component_config as $category_key => $category_data ) : ?>
					<div class="wpvulnerability-category">
						<div class="wpvulnerability-category-title"><?php echo esc_html( $category_data['label'] ); ?></div>
						<div class="wpvulnerability-components-grid">
							<?php foreach ( $category_data['items'] as $component => $component_data ) : ?>
								<?php
								$is_checked = ! empty( $wpvulnerability_analyze[ $component ] );
								$is_forced  = ! empty( $forced[ $component ] );
								$card_class = 'wpvulnerability-component-card';
								if ( $is_checked ) {
									$card_class .= ' wpvulnerability-checked';
								}
								if ( $is_forced ) {
									$card_class .= ' wpvulnerability-disabled';
								}
								?>
								<div class="<?php echo esc_attr( $card_class ); ?>" onclick="if (!this.classList.contains('wpvulnerability-disabled')) { var cb = this.querySelector('input[type=checkbox]'); cb.checked = !cb.checked; this.classList.toggle('wpvulnerability-checked'); }">
									<label>
										<span class="wpvulnerability-component-icon"><?php echo esc_html( $component_data['icon'] ); ?></span>
										<span class="wpvulnerability-component-content">
											<span class="wpvulnerability-component-label">
												<?php echo esc_html( $component_data['label'] ); ?>
												<?php if ( $is_forced ) : ?>
													<span class="wpvulnerability-forced-badge"><?php esc_html_e( 'FORCED', 'wpvulnerability' ); ?></span>
												<?php endif; ?>
											</span>
											<span class="wpvulnerability-component-status">
												<?php
												if ( $is_forced ) {
													echo '<span class="wpvulnerability-status-badge wpvulnerability-status-hidden">' . esc_html__( 'Hidden (forced by constant)', 'wpvulnerability' ) . '</span>';
												} elseif ( $is_checked ) {
													// Checked = value is 1 = analysis is DISABLED (hidden).
													echo '<span class="wpvulnerability-status-badge wpvulnerability-status-hidden">' . esc_html__( '🔴 Analysis disabled', 'wpvulnerability' ) . '</span>';
												} else {
													// Unchecked = value is 0 or not set = analysis is ENABLED (active).
													echo '<span class="wpvulnerability-status-badge wpvulnerability-status-active">' . esc_html__( '🟢 Analysis active', 'wpvulnerability' ) . '</span>';
												}
												?>
											</span>
										</span>
										<input
											type="checkbox"
											name="wpvulnerability-analyze[<?php echo esc_attr( $component ); ?>]"
											value="<?php echo esc_attr( $component ); ?>"
											<?php checked( $is_checked ); ?>
											<?php disabled( $is_forced ); ?>
											onclick="event.stopPropagation();"
										/>
									</label>
								</div>
							<?php endforeach; ?>
						</div>
					</div>
				<?php endforeach; ?>

				<div class="wpvulnerability-info-box">
					<p>
						<strong><?php esc_html_e( 'Tip:', 'wpvulnerability' ); ?></strong>
						<?php
						printf(
							/* translators: %s: documentation URL */
							wp_kses_post( __( 'You can force-disable components via wp-config.php constants. <a href="%s" target="_blank" rel="noopener noreferrer">Learn more about force-hiding checks →</a>', 'wpvulnerability' ) ),
							esc_url( 'https://www.wpvulnerability.com/plugin/#force-hiding-checks' )
						);
						?>
					</p>
				</div>

				<div class="wpvulnerability-save-section">
					<?php submit_button( __( 'Save Analysis Settings', 'wpvulnerability' ), 'primary', 'wpvulnerability_submit', false ); ?>
				</div>
			</div>
		</form>
	</section>
	<?php
}

/**
 * Outputs the Logs tab contents for the multisite settings screen.
 *
 * @since 4.2.0
 *
 * @return void
 */
function wpvulnerability_render_network_admin_tab_logs() {
	$choices          = wpvulnerability_get_log_retention_values();
	$current          = wpvulnerability_log_retention_days();
	$forced           = wpvulnerability_forced_log_retention();
	$per_page_options = wpvulnerability_get_log_per_page_options();
	$logs_per_page    = wpvulnerability_get_default_log_per_page();
	$per_page_request = isset( $_GET['logs_per_page'] ) ? absint( wp_unslash( $_GET['logs_per_page'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	if ( in_array( $per_page_request, $per_page_options, true ) ) {
		$logs_per_page = $per_page_request;
	}
	$current_page = isset( $_GET['log_page'] ) ? absint( wp_unslash( $_GET['log_page'] ) ) : 1; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	if ( $current_page < 1 ) {
		$current_page = 1;
	}
	$total_logs  = wpvulnerability_count_api_logs();
	$total_pages = max( 1, (int) ceil( $total_logs / $logs_per_page ) );
	if ( $current_page > $total_pages ) {
		$current_page = $total_pages;
	}
	$logs          = wpvulnerability_get_api_logs( $logs_per_page, $current_page );
	$logs_page_url = add_query_arg(
		array(
			'page'          => 'wpvulnerability-options',
			'tab'           => 'logs',
			'logs_per_page' => $logs_per_page,
			'log_page'      => $current_page,
		),
		network_admin_url( 'settings.php' )
	);
	$requested_log = isset( $_GET['log'] ) ? absint( wp_unslash( $_GET['log'] ) ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	$view_log      = null;
	$log_error     = '';
	$pagination    = '';

	if ( $total_pages > 1 ) {
		$pagination_base = remove_query_arg(
			array( 'log', 'log_page' ),
			$logs_page_url
		);
		$pagination      = paginate_links(
			array(
				'base'      => add_query_arg( 'log_page', '%#%', $pagination_base ),
				'format'    => '',
				'current'   => $current_page,
				'total'     => $total_pages,
				'prev_text' => __( '&laquo; Previous', 'wpvulnerability' ),
				'next_text' => __( 'Next &raquo;', 'wpvulnerability' ),
				'type'      => 'list',
			)
		);
	}

	if ( $requested_log > 0 ) {
		$nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( $nonce && wp_verify_nonce( $nonce, 'wpvulnerability_view_log_' . $requested_log ) ) {
			$view_log = wpvulnerability_get_api_log( $requested_log );
			if ( ! $view_log ) {
				$log_error = __( 'Log entry not found or has been removed.', 'wpvulnerability' );
			}
		} else {
			$log_error = __( 'Unable to load the requested log entry.', 'wpvulnerability' );
		}
	}

	?>
	<section class="section wpvulnerability-logs-panel">

		<?php if ( ! $view_log ) : ?>
			<!-- Configuration Section -->
			<div class="wpvulnerability-security-section">
				<h3><?php esc_html_e( 'Log Configuration', 'wpvulnerability' ); ?></h3>

				<!-- Stats Row -->
				<div class="wpvulnerability-stats-row">
					<div class="wpvulnerability-stat-card">
						<div class="wpvulnerability-stat-label"><?php esc_html_e( 'Total Logs', 'wpvulnerability' ); ?></div>
						<div class="wpvulnerability-stat-value"><?php echo esc_html( number_format_i18n( $total_logs ) ); ?></div>
					</div>
					<div class="wpvulnerability-stat-card">
						<div class="wpvulnerability-stat-label"><?php esc_html_e( 'Retention Period', 'wpvulnerability' ); ?></div>
						<div class="wpvulnerability-stat-value">
							<?php
							echo esc_html(
								0 === $current
									? __( 'Disabled', 'wpvulnerability' )
									: sprintf(
										/* translators: %d: Number of days. */
										_n( '%d day', '%d days', $current, 'wpvulnerability' ),
										$current
									)
							);
							?>
						</div>
					</div>
				</div>

				<!-- Config Form -->
				<form method="post" action="" class="wpvulnerability-config-form">
					<?php wp_nonce_field( 'wpvulnerability_nonce', 'wpauto_nonce' ); ?>
					<label for="wpvulnerability_log_retention">
						<?php esc_html_e( 'Log Retention Period', 'wpvulnerability' ); ?>
					</label>
					<select name="wpvulnerability-config[log_retention]" id="wpvulnerability_log_retention"<?php disabled( null !== $forced ); ?>>
						<?php
						foreach ( $choices as $days ) {
							$label = 0 === $days
								? __( 'None (Disabled)', 'wpvulnerability' )
								: sprintf(
									/* translators: %d: Number of days. */
									_n( '%d day', '%d days', $days, 'wpvulnerability' ),
									$days
								);
							printf(
								'<option value="%1$s"%2$s>%3$s</option>',
								esc_attr( $days ),
								selected( $current, $days, false ),
								esc_html( $label )
							);
						}
						?>
					</select>
					<?php
					if ( null !== $forced ) {
						printf(
							'<input type="hidden" name="wpvulnerability-config[log_retention]" value="%s" />',
							esc_attr( $forced )
						);
						printf(
							'<p class="description">%s</p>',
							esc_html__( 'This value is enforced by the WPVULNERABILITY_LOG_RETENTION_DAYS constant.', 'wpvulnerability' )
						);
					} else {
						printf(
							'<p class="description">%s</p>',
							esc_html__( 'Choose how long WPVulnerability should keep API response logs. Older logs are automatically deleted.', 'wpvulnerability' )
						);
					}
					?>
					<?php submit_button( __( 'Save Configuration', 'wpvulnerability' ), 'primary', 'wpvulnerability_submit', false ); ?>
				</form>

				<?php if ( 0 === $current ) : ?>
					<div class="wpvulnerability-info-box" style="margin-top: 20px;">
						<p><strong><?php esc_html_e( 'Log retention is currently disabled.', 'wpvulnerability' ); ?></strong> <?php esc_html_e( 'API responses are not being saved.', 'wpvulnerability' ); ?></p>
					</div>
				<?php endif; ?>
			</div>

			<?php if ( $log_error ) : ?>
				<div class="notice notice-error"><p><?php echo esc_html( $log_error ); ?></p></div>
			<?php endif; ?>
		<?php endif; ?>

		<?php if ( $view_log ) : ?>
			<!-- Log Detail View -->
			<div class="wpvulnerability-log-detail">
				<h3><?php esc_html_e( 'Log Details', 'wpvulnerability' ); ?></h3>

				<div class="wpvulnerability-log-meta">
					<div class="wpvulnerability-log-meta-item">
						<div class="wpvulnerability-log-meta-label"><?php esc_html_e( 'Date', 'wpvulnerability' ); ?></div>
						<div class="wpvulnerability-log-meta-value"><?php echo esc_html( wpvulnerability_format_log_date( $view_log ) ); ?></div>
					</div>
					<div class="wpvulnerability-log-meta-item">
						<div class="wpvulnerability-log-meta-label"><?php esc_html_e( 'API Endpoint', 'wpvulnerability' ); ?></div>
						<div class="wpvulnerability-log-meta-value"><?php echo esc_html( get_the_title( $view_log ) ); ?></div>
					</div>
				</div>

				<h4 style="margin: 20px 0 12px 0; color: #1d2327; font-size: 15px;"><?php esc_html_e( 'API Response', 'wpvulnerability' ); ?></h4>
				<pre><code><?php echo esc_html( wpvulnerability_format_log_content( $view_log->post_content ) ); ?></code></pre>

				<p style="margin-top: 20px;">
					<a class="button button-primary" href="<?php echo esc_url( $logs_page_url ); ?>">
						&larr; <?php esc_html_e( 'Back to Logs', 'wpvulnerability' ); ?>
					</a>
				</p>
			</div>
		<?php else : ?>
			<!-- Logs List Section -->
			<?php if ( empty( $logs ) ) : ?>
				<div class="wpvulnerability-empty-state">
					<div class="wpvulnerability-empty-state-icon">📋</div>
					<h3><?php esc_html_e( 'No logs available', 'wpvulnerability' ); ?></h3>
					<p><?php esc_html_e( 'API response logs will appear here once the vulnerability scanner runs.', 'wpvulnerability' ); ?></p>
				</div>
			<?php else : ?>
				<div class="wpvulnerability-security-section">
					<h3><?php esc_html_e( 'API Response Logs', 'wpvulnerability' ); ?></h3>

					<!-- Toolbar -->
					<div class="wpvulnerability-toolbar">
						<div class="wpvulnerability-toolbar-left">
							<form method="get" action="<?php echo esc_url( network_admin_url( 'settings.php' ) ); ?>" style="display: flex; align-items: center; gap: 8px; margin: 0;">
								<input type="hidden" name="page" value="wpvulnerability-options" />
								<input type="hidden" name="tab" value="logs" />
								<input type="hidden" name="log_page" value="1" />
								<label for="wpvulnerability_logs_per_page" style="margin: 0; font-size: 13px; color: #646970;">
									<?php esc_html_e( 'Show:', 'wpvulnerability' ); ?>
								</label>
								<select name="logs_per_page" id="wpvulnerability_logs_per_page" style="min-width: 80px;">
									<?php foreach ( $per_page_options as $per_page_option ) : ?>
										<option value="<?php echo esc_attr( $per_page_option ); ?>"<?php selected( $logs_per_page, $per_page_option ); ?>>
											<?php echo esc_html( number_format_i18n( $per_page_option ) ); ?>
										</option>
									<?php endforeach; ?>
								</select>
								<?php submit_button( __( 'Apply', 'wpvulnerability' ), 'secondary', 'submit', false, array( 'style' => 'margin: 0; padding: 4px 12px; height: auto;' ) ); ?>
							</form>
						</div>
						<div class="wpvulnerability-toolbar-right">
							<span style="font-size: 13px; color: #646970;">
								<?php
								printf(
									/* translators: 1: first item, 2: last item, 3: total items */
									esc_html__( 'Showing %1$s–%2$s of %3$s', 'wpvulnerability' ),
									esc_html( number_format_i18n( ( ( $current_page - 1 ) * $logs_per_page ) + 1 ) ),
									esc_html( number_format_i18n( min( $current_page * $logs_per_page, $total_logs ) ) ),
									esc_html( number_format_i18n( $total_logs ) )
								);
								?>
							</span>
						</div>
					</div>

					<!-- Logs Table -->
					<table class="wpvulnerability-logs-table">
						<thead>
							<tr>
								<th style="width: 200px;"><?php esc_html_e( 'Date', 'wpvulnerability' ); ?></th>
								<th><?php esc_html_e( 'API Endpoint', 'wpvulnerability' ); ?></th>
								<th style="width: 100px; text-align: center;"><?php esc_html_e( 'Actions', 'wpvulnerability' ); ?></th>
							</tr>
						</thead>
						<tbody>
							<?php foreach ( $logs as $log ) : ?>
								<tr>
									<td>
										<span class="wpvulnerability-log-date"><?php echo esc_html( wpvulnerability_format_log_date( $log ) ); ?></span>
									</td>
									<td>
										<span class="wpvulnerability-log-url"><?php echo esc_html( get_the_title( $log ) ); ?></span>
									</td>
									<td style="text-align: center;">
										<?php
										$view_url = add_query_arg(
											array(
												'page'     => 'wpvulnerability-options',
												'tab'      => 'logs',
												'log'      => $log->ID,
												'logs_per_page' => $logs_per_page,
												'log_page' => $current_page,
											),
											network_admin_url( 'settings.php' )
										);
										$view_url = wp_nonce_url( $view_url, 'wpvulnerability_view_log_' . $log->ID );
										?>
										<a class="button button-small" href="<?php echo esc_url( $view_url ); ?>">
											<?php esc_html_e( 'View', 'wpvulnerability' ); ?>
										</a>
									</td>
								</tr>
							<?php endforeach; ?>
						</tbody>
					</table>

					<!-- Pagination -->
					<?php if ( $pagination ) : ?>
						<div class="wpvulnerability-pagination">
							<?php echo wp_kses_post( $pagination ); ?>
						</div>
					<?php endif; ?>

					<!-- Danger Zone -->
					<div class="wpvulnerability-danger-zone">
						<h4><?php esc_html_e( 'Danger Zone', 'wpvulnerability' ); ?></h4>
						<p style="margin: 0 0 12px 0; font-size: 13px; color: #646970;">
							<?php esc_html_e( 'Permanently delete all API response logs. This action cannot be undone.', 'wpvulnerability' ); ?>
						</p>
						<form method="post" action="" onsubmit="return confirm('<?php echo esc_js( __( 'Are you sure you want to delete all logs? This action cannot be undone.', 'wpvulnerability' ) ); ?>');">
							<?php wp_nonce_field( 'wpvulnerability_delete_logs_action', 'wpvulnerability_delete_logs_nonce' ); ?>
							<input type="hidden" name="wpvulnerability_delete_logs" value="1" />
							<?php submit_button( __( 'Delete All Logs', 'wpvulnerability' ), 'delete', 'submit', false ); ?>
						</form>
					</div>
				</div>
			<?php endif; ?>
		<?php endif; ?>
	</section>
	<?php
}

/**
 * Outputs the Tools tab contents for the multisite settings screen.
 *
 * @since 4.1.2
 *
 * @return void
 */
function wpvulnerability_render_network_admin_tab_tools() {
	global $wpvulnerability_settings;

	$tools_action = add_query_arg(
		array(
			'page' => 'wpvulnerability-options',
			'tab'  => 'tools',
		),
		network_admin_url( 'settings.php' )
	);

	$wpvulnerability_cron_config = is_multisite() ? get_site_option( 'wpvulnerability-config', array() ) : get_option( 'wpvulnerability-config', array() );
	$is_main_site                = ( ! is_multisite() || is_main_site() );
	$cron_status                 = wpvulnerability_get_cron_status( $wpvulnerability_cron_config, $is_main_site );
	$cron_schedules              = function_exists( 'wp_get_schedules' ) ? wp_get_schedules() : array();
	$date_format                 = get_option( 'date_format' ) . ' ' . get_option( 'time_format' );
	?>
	<section class="section wpvulnerability-tools-panel">

		<!-- Reload Data Tool -->
		<div class="wpvulnerability-tool-card">
			<h3>
				<span class="wpvulnerability-tool-icon">🔄</span>
				<?php esc_html_e( 'Reload Data from API', 'wpvulnerability' ); ?>
			</h3>
			<p class="wpvulnerability-tool-description">
				<?php esc_html_e( 'Reload all Core, Plugins, Themes and other components information directly from the API to have updated data. This will clear the cache and fetch fresh vulnerability information.', 'wpvulnerability' ); ?>
			</p>
			<div class="wpvulnerability-tool-actions">
				<form method="post" action="" style="display: inline;">
					<?php wp_nonce_field( 'wpvulnerability_reset_action', 'wpvulnerability_reset_nonce' ); ?>
					<input type="submit" name="wpvulnerability_reset" value="<?php esc_attr_e( 'Reload Data', 'wpvulnerability' ); ?>" class="button button-primary">
				</form>
			</div>
		</div>

		<!-- Email Test Tool -->
		<div class="wpvulnerability-tool-card">
			<h3>
				<span class="wpvulnerability-tool-icon">📧</span>
				<?php esc_html_e( 'Test Email Notifications', 'wpvulnerability' ); ?>
			</h3>
			<p class="wpvulnerability-tool-description">
				<?php esc_html_e( 'Send a test email notification with current vulnerability data to verify your email configuration is working correctly.', 'wpvulnerability' ); ?>
			</p>
			<?php
			$from_email = null;
			if ( defined( 'WPVULNERABILITY_MAIL' ) ) {
				$from_email = sanitize_email( trim( (string) WPVULNERABILITY_MAIL ) );
				if ( is_email( $from_email ) ) {
					?>
					<div class="wpvulnerability-info-box">
						<p>
							<strong><?php esc_html_e( 'From address (configured via constant):', 'wpvulnerability' ); ?></strong>
							<code><?php echo esc_html( $from_email ); ?></code>
						</p>
					</div>
					<?php
				}
			}
			if ( ! $from_email ) {
				$from_email = get_bloginfo( 'admin_email' );
				?>
				<div class="wpvulnerability-info-box">
					<p>
						<strong><?php esc_html_e( 'From address (default):', 'wpvulnerability' ); ?></strong>
						<code><?php echo esc_html( $from_email ); ?></code>
					</p>
				</div>
				<?php
			}
			?>
			<p style="margin: 8px 0; font-size: 13px;">
				<a href="https://www.wpvulnerability.com/plugin/#from-mail" target="_blank" rel="noopener noreferrer">
					<?php esc_html_e( 'Learn how to customize the email sender address →', 'wpvulnerability' ); ?>
				</a>
			</p>
			<div class="wpvulnerability-tool-actions">
				<form method="post" action="" style="display: inline;">
					<?php wp_nonce_field( 'wpvulnerability_email_action', 'wpvulnerability_email_nonce' ); ?>
					<input type="submit" name="wpvulnerability_email" value="<?php esc_attr_e( 'Send Test Email', 'wpvulnerability' ); ?>" class="button button-primary">
				</form>
			</div>
		</div>

		<!-- WP-Cron Status Tool -->
		<div class="wpvulnerability-tool-card">
			<h3>
				<span class="wpvulnerability-tool-icon">⏰</span>
				<?php esc_html_e( 'WP-Cron Status', 'wpvulnerability' ); ?>
			</h3>
			<p class="wpvulnerability-tool-description">
				<?php esc_html_e( 'View and manage WP-Cron scheduled events. Compare expected schedules with actual cron jobs to ensure automated tasks are running correctly.', 'wpvulnerability' ); ?>
			</p>

			<?php if ( empty( $cron_status['expected'] ) ) : ?>
				<div class="wpvulnerability-info-box">
					<p><?php esc_html_e( 'No cron data available.', 'wpvulnerability' ); ?></p>
				</div>
			<?php else : ?>
				<table class="wpvulnerability-cron-table">
					<thead>
						<tr>
							<th><?php esc_html_e( 'Hook', 'wpvulnerability' ); ?></th>
							<th><?php esc_html_e( 'Expected', 'wpvulnerability' ); ?></th>
							<th><?php esc_html_e( 'Found', 'wpvulnerability' ); ?></th>
							<th><?php esc_html_e( 'Next Run', 'wpvulnerability' ); ?></th>
							<th><?php esc_html_e( 'Status', 'wpvulnerability' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php foreach ( $cron_status['expected'] as $row ) : ?>
							<?php
							$expected_schedule = isset( $row['schedule'] ) ? (string) $row['schedule'] : '';
							$schedule_label    = '';
							if ( ! $row['should_exist'] ) {
								$schedule_label = __( 'Not expected', 'wpvulnerability' );
							} elseif ( '' === $expected_schedule ) {
								$schedule_label = __( 'Disabled', 'wpvulnerability' );
							} elseif ( isset( $cron_schedules[ $expected_schedule ]['display'] ) ) {
								$schedule_label = $cron_schedules[ $expected_schedule ]['display'];
							} else {
								$schedule_label = $expected_schedule;
							}

							$found_schedules = array();
							if ( ! empty( $row['schedules_found'] ) ) {
								foreach ( $row['schedules_found'] as $schedule_id ) {
									if ( isset( $cron_schedules[ $schedule_id ]['display'] ) ) {
										$found_schedules[] = $cron_schedules[ $schedule_id ]['display'];
									} else {
										$found_schedules[] = $schedule_id;
									}
								}
							}

							$found_label = $row['count'] > 0 ? implode( ', ', $found_schedules ) : __( 'None', 'wpvulnerability' );
							$next_run    = ( isset( $row['next_run'] ) && $row['next_run'] ) ? date_i18n( $date_format, (int) $row['next_run'] ) : '—';
							$status_text = isset( $row['messages'] ) && is_array( $row['messages'] ) ? implode( ' ', $row['messages'] ) : '';

							// Determine status badge class.
							$status_class = 'success';
							if ( strpos( $status_text, 'Missing' ) !== false || strpos( $status_text, 'Duplicate' ) !== false ) {
								$status_class = 'error';
							} elseif ( strpos( $status_text, 'legacy' ) !== false ) {
								$status_class = 'warning';
							}
							?>
							<tr>
								<td>
									<strong><?php echo esc_html( isset( $row['label'] ) ? $row['label'] : $row['hook'] ); ?></strong><br>
									<code><?php echo esc_html( $row['hook'] ); ?></code>
								</td>
								<td><?php echo esc_html( $schedule_label ); ?></td>
								<td><?php echo esc_html( $found_label ); ?></td>
								<td><?php echo esc_html( $next_run ); ?></td>
								<td>
									<?php if ( $status_text ) : ?>
										<span class="wpvulnerability-status-badge <?php echo esc_attr( $status_class ); ?>">
											<?php echo esc_html( $status_text ); ?>
										</span>
									<?php else : ?>
										<span class="wpvulnerability-status-badge success">✓ <?php esc_html_e( 'OK', 'wpvulnerability' ); ?></span>
									<?php endif; ?>
								</td>
							</tr>
						<?php endforeach; ?>
					</tbody>
				</table>

				<?php if ( ! empty( $cron_status['unexpected'] ) ) : ?>
					<div class="wpvulnerability-unexpected-list">
						<h4><?php esc_html_e( '⚠️ Unexpected WPVulnerability Events', 'wpvulnerability' ); ?></h4>
						<ul>
							<?php foreach ( $cron_status['unexpected'] as $unexpected ) : ?>
								<?php
								$unexpected_schedules = array();
								if ( ! empty( $unexpected['schedules'] ) ) {
									foreach ( $unexpected['schedules'] as $schedule_id ) {
										if ( isset( $cron_schedules[ $schedule_id ]['display'] ) ) {
											$unexpected_schedules[] = $cron_schedules[ $schedule_id ]['display'];
										} else {
											$unexpected_schedules[] = $schedule_id;
										}
									}
								}
								$unexpected_label = empty( $unexpected_schedules ) ? __( 'No interval', 'wpvulnerability' ) : implode( ', ', $unexpected_schedules );
								$unexpected_next  = ( isset( $unexpected['next_run'] ) && $unexpected['next_run'] ) ? date_i18n( $date_format, (int) $unexpected['next_run'] ) : '—';
								?>
								<li>
									<code><?php echo esc_html( $unexpected['hook'] ); ?></code>
									– <?php echo esc_html( $unexpected_label ); ?>
									– <?php esc_html_e( 'Next:', 'wpvulnerability' ); ?> <?php echo esc_html( $unexpected_next ); ?>
								</li>
							<?php endforeach; ?>
						</ul>
					</div>
				<?php endif; ?>

				<div class="wpvulnerability-tool-actions">
					<form method="post" action="" style="display: inline;">
						<?php wp_nonce_field( 'wpvulnerability_cron_action', 'wpvulnerability_cron_nonce' ); ?>
						<input type="submit" name="wpvulnerability_repair_cron" value="<?php esc_attr_e( 'Repair Cron Events', 'wpvulnerability' ); ?>" class="button button-secondary">
					</form>
				</div>
			<?php endif; ?>
		</div>

		<!-- Reset Plugin Tool -->
		<div class="wpvulnerability-tool-card">
			<div class="wpvulnerability-danger-zone">
				<h4>⚠️ <?php esc_html_e( 'Reset WPVulnerability', 'wpvulnerability' ); ?></h4>
				<p style="margin: 0 0 16px 0; color: #646970; font-size: 13px; line-height: 1.6;">
					<?php esc_html_e( 'Delete all WPVulnerability settings, cached data, logs, and scheduled events, then restore defaults and reload data from the API.', 'wpvulnerability' ); ?>
					<br>
					<strong style="color: #b32d2e;"><?php esc_html_e( 'This action cannot be undone.', 'wpvulnerability' ); ?></strong>
				</p>
				<form method="post" action="" onsubmit="return confirm('<?php echo esc_js( __( 'Are you sure you want to reset all WPVulnerability data? This action cannot be undone.', 'wpvulnerability' ) ); ?>');">
					<?php wp_nonce_field( 'wpvulnerability_full_reset_action', 'wpvulnerability_full_reset_nonce' ); ?>
					<input type="submit" name="wpvulnerability_full_reset" value="<?php esc_attr_e( 'Reset Plugin', 'wpvulnerability' ); ?>" class="button button-delete">
				</form>
			</div>
		</div>
	</section>
	<?php
}

/**
 * Outputs the About tab contents for the multisite settings screen.
 *
 * @since 4.1.2
 *
 * @return void
 */
function wpvulnerability_render_network_admin_tab_about() {
	$wpvulnerability_statistics = json_decode( get_site_option( 'wpvulnerability-statistics' ), true );

	if ( ! is_array( $wpvulnerability_statistics ) ) {
		$wpvulnerability_statistics = array();
	}

	?>
	<section class="section wpvulnerability-about-panel">

		<!-- Database Statistics Section -->
		<div class="wpvulnerability-security-section">
			<h3><?php esc_html_e( 'Vulnerability Database Statistics', 'wpvulnerability' ); ?></h3>

			<?php
			$components = array(
				'plugins'     => __( 'Plugins', 'wpvulnerability' ),
				'themes'      => __( 'Themes', 'wpvulnerability' ),
				'php'         => __( 'PHP', 'wpvulnerability' ),
				'apache'      => __( 'Apache HTTPD', 'wpvulnerability' ),
				'nginx'       => __( 'nginx', 'wpvulnerability' ),
				'mariadb'     => __( 'MariaDB', 'wpvulnerability' ),
				'mysql'       => __( 'MySQL', 'wpvulnerability' ),
				'imagemagick' => __( 'ImageMagick', 'wpvulnerability' ),
				'curl'        => __( 'curl', 'wpvulnerability' ),
				'memcached'   => __( 'memcached', 'wpvulnerability' ),
				'redis'       => __( 'Redis', 'wpvulnerability' ),
				'sqlite'      => __( 'SQLite', 'wpvulnerability' ),
			);
			?>

			<div class="wpvulnerability-stats-grid">
				<?php foreach ( $components as $component => $label ) : ?>
					<?php
					$has_data   = isset( $wpvulnerability_statistics[ $component ] );
					$card_class = 'wpvulnerability-stat-card';
					if ( ! $has_data ) {
						$card_class .= ' wpvulnerability-no-data';
					}
					?>
					<div class="<?php echo esc_attr( $card_class ); ?>">
						<div class="wpvulnerability-stat-label"><?php echo esc_html( $label ); ?></div>
						<?php if ( $has_data ) : ?>
							<div class="wpvulnerability-stat-value">
								<?php echo esc_html( number_format_i18n( absint( $wpvulnerability_statistics[ $component ]['vulnerabilities'] ) ) ); ?>
							</div>
							<div class="wpvulnerability-stat-meta">
								<?php
								printf(
									// translators: number of vulnerabilities.
									esc_html( _n( '%s vulnerability', '%s vulnerabilities', absint( $wpvulnerability_statistics[ $component ]['vulnerabilities'] ), 'wpvulnerability' ) ),
									''
								);
								?>
								<?php if ( isset( $wpvulnerability_statistics[ $component ]['products'] ) ) : ?>
									<br>
									<?php
									printf(
										// translators: number of products.
										esc_html( _n( '(%s product)', '(%s products)', absint( $wpvulnerability_statistics[ $component ]['products'] ), 'wpvulnerability' ) ),
										esc_html( number_format_i18n( absint( $wpvulnerability_statistics[ $component ]['products'] ) ) )
									);
									?>
								<?php endif; ?>
							</div>
						<?php else : ?>
							<div class="wpvulnerability-stat-value"><?php esc_html_e( 'No data', 'wpvulnerability' ); ?></div>
						<?php endif; ?>
					</div>
				<?php endforeach; ?>
			</div>

			<?php if ( isset( $wpvulnerability_statistics['updated'] ) ) : ?>
				<div class="wpvulnerability-info-box">
					<p>
						<?php
						if ( version_compare( $GLOBALS['wp_version'], '5.0', '>=' ) ) {
							switch_to_locale( determine_locale() );
						} elseif ( version_compare( $GLOBALS['wp_version'], '4.7', '>=' ) ) {
							switch_to_locale( get_locale() );
						}
						$formatted_datetime = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), (int) $wpvulnerability_statistics['updated']['unixepoch'] );
						// translators: date of last update.
						printf( esc_html__( 'Database last updated: %s', 'wpvulnerability' ), '<strong>' . esc_html( $formatted_datetime ) . '</strong>' );
						if ( version_compare( $GLOBALS['wp_version'], '4.7', '>=' ) ) {
							restore_previous_locale();
						}
						?>
					</p>
				</div>
			<?php endif; ?>
		</div>

		<!-- Sponsors Section -->
		<div class="wpvulnerability-security-section">
			<h3><?php esc_html_e( 'Sponsors', 'wpvulnerability' ); ?></h3>

			<?php if ( isset( $wpvulnerability_statistics['sponsors'] ) && ! empty( $wpvulnerability_statistics['sponsors'] ) ) : ?>
				<div class="wpvulnerability-people-grid">
					<?php foreach ( $wpvulnerability_statistics['sponsors'] as $sponsor ) : ?>
						<div class="wpvulnerability-person-card">
							<img src="<?php echo esc_url( $sponsor['image'] ); ?>" alt="<?php echo esc_attr( $sponsor['name'] ); ?>">
							<a href="<?php echo esc_url( $sponsor['url'] ); ?>" target="_blank" rel="noreferrer noopener">
								<?php echo esc_html( $sponsor['name'] ); ?>
							</a>
						</div>
					<?php endforeach; ?>
				</div>
			<?php else : ?>
				<div class="wpvulnerability-empty-state">
					<div class="wpvulnerability-empty-state-icon">💎</div>
					<p><strong><?php esc_html_e( 'No sponsor data available.', 'wpvulnerability' ); ?></strong></p>
				</div>
			<?php endif; ?>
		</div>

		<!-- Contributors Section -->
		<div class="wpvulnerability-security-section">
			<h3><?php esc_html_e( 'Contributors', 'wpvulnerability' ); ?></h3>

			<?php if ( isset( $wpvulnerability_statistics['contributors'] ) && ! empty( $wpvulnerability_statistics['contributors'] ) ) : ?>
				<div class="wpvulnerability-people-grid">
					<?php foreach ( $wpvulnerability_statistics['contributors'] as $contributor ) : ?>
						<div class="wpvulnerability-person-card">
							<img src="<?php echo esc_url( $contributor['image'] ); ?>" alt="<?php echo esc_attr( $contributor['name'] ); ?>">
							<a href="<?php echo esc_url( $contributor['url'] ); ?>" target="_blank" rel="noreferrer noopener">
								<?php echo esc_html( $contributor['name'] ); ?>
							</a>
						</div>
					<?php endforeach; ?>
				</div>
			<?php else : ?>
				<div class="wpvulnerability-empty-state">
					<div class="wpvulnerability-empty-state-icon">👥</div>
					<p><strong><?php esc_html_e( 'No contributor data available.', 'wpvulnerability' ); ?></strong></p>
				</div>
			<?php endif; ?>
		</div>
	</section>
	<?php
}

/**
 * Adds a WP-Admin menu option for the WPVulnerability plugin.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_menu() {
	// Adds a submenu page under the Settings menu.
	add_submenu_page(
		'settings.php',
		__( 'WPVulnerability', 'wpvulnerability' ),
		__( 'WPVulnerability', 'wpvulnerability' ),
		'manage_network_options',
		'wpvulnerability-options',
		'wpvulnerability_create_admin_page'
	);
}
add_action( 'network_admin_menu', 'wpvulnerability_admin_menu' );

/**
 * Print the settings header information for the notifications section.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_section_notifications() {
		// Output the header information for the notifications section.
		esc_html_e( 'Configure and save these settings to receive notifications.', 'wpvulnerability' );
}

/**
 * Callback function to display the email input field in the admin settings page.
 * This function retrieves the current WPVulnerability plugin settings and displays the email input field
 * for users to enter their email addresses. If no email is saved in the settings, the admin email is displayed.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_emails_callback() {
	// Retrieve the WPVulnerability plugin settings.
	$wpvulnerability_settings = get_site_option( 'wpvulnerability-config' );

	// Set a default value for the email input field if no email is saved in the settings.
	if ( ! isset( $wpvulnerability_settings['emails'] ) ) {
		$wpvulnerability_settings['emails'] = '';
	}

	// Output the email input field. Use the network admin email as a placeholder in a multisite environment.
	$admin_email = get_site_option( 'admin_email' );

	// Output the email input field and display the admin email as a hint.
	?>
	<input class="regular-text" type="text" name="wpvulnerability-config[emails]" id="wpvulnerability_emails" placeholder="<?php echo esc_attr( (string) $admin_email ); ?>" value="<?php echo esc_attr( (string) $wpvulnerability_settings['emails'] ); ?>">
	<br><small><?php esc_html_e( 'Default administrator email', 'wpvulnerability' ); ?>: <?php echo esc_attr( (string) $admin_email ); ?></small>
	<?php

	unset( $admin_email );
}

/**
 * Print the cache expiration selector.
 *
 * @since 4.1.0
 *
 * @return void
 */
function wpvulnerability_admin_cache_callback() {

	$wpvulnerability_settings = get_site_option( 'wpvulnerability-config', array() );
	$options                  = array( 1, 6, 12, 24 );
	$forced_cache             = null;

	if ( defined( 'WPVULNERABILITY_CACHE_HOURS' ) ) {
		$forced_cache = (int) WPVULNERABILITY_CACHE_HOURS;
		if ( ! in_array( $forced_cache, $options, true ) ) {
			$options[] = $forced_cache;
			sort( $options, SORT_NUMERIC );
		}
	}

	$current = isset( $wpvulnerability_settings['cache'] ) ? (int) $wpvulnerability_settings['cache'] : 12;
	if ( null !== $forced_cache ) {
		$current = $forced_cache;
	}

	echo '<select name="wpvulnerability-config[cache]" id="wpvulnerability_cache"';
	disabled( null !== $forced_cache );
	echo '>';
	foreach ( $options as $hours ) {
		printf(
			'<option value="%1$s"%2$s>%3$s</option>',
			esc_attr( $hours ),
			selected( $current, $hours, false ),
			esc_html(
				sprintf(
						/* translators: %d: number of hours */
					_n( '%d hour', '%d hours', $hours, 'wpvulnerability' ),
					$hours
				)
			)
		);
	}
		echo '</select>';

	if ( null !== $forced_cache ) {
			printf(
				'<input type="hidden" name="wpvulnerability-config[cache]" value="%s" />',
				esc_attr( $current )
			);
	}

		printf(
			'<p class="description"><a href="%1$s" target="_blank"><small><i>%2$s</i></small></a></p>',
			esc_url( 'https://www.wpvulnerability.com/plugin/#cache-duration' ),
			esc_html__( 'Read more if you want to force the cache time.', 'wpvulnerability' )
		);
}


/**
 * Print the settings header information for the analyze section.
 *
 * @since 3.3.0
 *
 * @return void
 */
function wpvulnerability_admin_section_analyze() {
	// Output the header information for the analyze section.
	esc_html_e( 'Configure and save these settings to hide vulnerabilities.', 'wpvulnerability' );
}

/**
 * Print when to send the vulnerability scan emails.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_period_callback() {
		// Get the saved plugin settings.
		$wpvulnerability_settings = get_site_option( 'wpvulnerability-config', array() );
		$defaults                 = array(
			'period' => 'weekly',
			'day'    => 'monday',
			'hour'   => 0,
			'minute' => 0,
		);
		$wpvulnerability_settings = wp_parse_args( $wpvulnerability_settings, $defaults );

		?>
		<div id="wpvulnerability_period">
		<label>
				<input type="radio" name="wpvulnerability-config[period]" value="never" <?php checked( $wpvulnerability_settings['period'], 'never' ); ?> />
				<?php esc_html_e( 'Never', 'wpvulnerability' ); ?>
		</label>
		<br/>
		<label>
				<input type="radio" name="wpvulnerability-config[period]" value="daily" <?php checked( $wpvulnerability_settings['period'], 'daily' ); ?> />
				<?php esc_html_e( 'Daily', 'wpvulnerability' ); ?>
		</label>
		<br/>
		<label>
				<input type="radio" name="wpvulnerability-config[period]" value="weekly" <?php checked( $wpvulnerability_settings['period'], 'weekly' ); ?> />
				<?php esc_html_e( 'Weekly', 'wpvulnerability' ); ?>
		</label>
		<div id="wpvulnerability_day_wrap">
				<br/>
				<label for="wpvulnerability_day"><?php esc_html_e( 'Day', 'wpvulnerability' ); ?></label>
				<select name="wpvulnerability-config[day]" id="wpvulnerability_day">
						<option value="monday" <?php selected( $wpvulnerability_settings['day'], 'monday' ); ?>><?php esc_html_e( 'Monday', 'wpvulnerability' ); ?></option>
						<option value="tuesday" <?php selected( $wpvulnerability_settings['day'], 'tuesday' ); ?>><?php esc_html_e( 'Tuesday', 'wpvulnerability' ); ?></option>
						<option value="wednesday" <?php selected( $wpvulnerability_settings['day'], 'wednesday' ); ?>><?php esc_html_e( 'Wednesday', 'wpvulnerability' ); ?></option>
						<option value="thursday" <?php selected( $wpvulnerability_settings['day'], 'thursday' ); ?>><?php esc_html_e( 'Thursday', 'wpvulnerability' ); ?></option>
						<option value="friday" <?php selected( $wpvulnerability_settings['day'], 'friday' ); ?>><?php esc_html_e( 'Friday', 'wpvulnerability' ); ?></option>
						<option value="saturday" <?php selected( $wpvulnerability_settings['day'], 'saturday' ); ?>><?php esc_html_e( 'Saturday', 'wpvulnerability' ); ?></option>
						<option value="sunday" <?php selected( $wpvulnerability_settings['day'], 'sunday' ); ?>><?php esc_html_e( 'Sunday', 'wpvulnerability' ); ?></option>
				</select>
		</div>
		<div id="wpvulnerability_time_wrap">
				<br/>
				<label for="wpvulnerability_hour"><?php esc_html_e( 'Hour', 'wpvulnerability' ); ?></label>
				<input type="number" min="0" max="23" name="wpvulnerability-config[hour]" id="wpvulnerability_hour" value="<?php echo esc_attr( (string) $wpvulnerability_settings['hour'] ); ?>" />
				<label for="wpvulnerability_minute"><?php esc_html_e( 'Minute', 'wpvulnerability' ); ?></label>
				<input type="number" min="0" max="59" name="wpvulnerability-config[minute]" id="wpvulnerability_minute" value="<?php echo esc_attr( (string) $wpvulnerability_settings['minute'] ); ?>" />
		</div>
		</div>
		<?php
}

/**
 * Print where to send the notifications.
 *
 * @since 3.6.0
 *
 * @return void
 */
function wpvulnerability_admin_notify_callback() {
		$wpvulnerability_settings = get_site_option( 'wpvulnerability-config', array() );
			$defaults             = array(
				'email' => 'y',
				'slack' => 'n',
				'teams' => 'n',
			);

			if ( ! isset( $wpvulnerability_settings['notify'] ) || ! is_array( $wpvulnerability_settings['notify'] ) ) {
							$wpvulnerability_settings['notify'] = $defaults;
			} else {
							$wpvulnerability_settings['notify'] = wp_parse_args( $wpvulnerability_settings['notify'], $defaults );
			}

			$wpvulnerability_settings['notify'] = wpvulnerability_normalize_notify_settings( $wpvulnerability_settings['notify'] );

			$email_enabled = wpvulnerability_is_yes( $wpvulnerability_settings['notify']['email'] );
			$slack_enabled = wpvulnerability_is_yes( $wpvulnerability_settings['notify']['slack'] );
			$teams_enabled = wpvulnerability_is_yes( $wpvulnerability_settings['notify']['teams'] );

			?>
		<div id="wpvulnerability_notify">
		<label>
							<input type="checkbox" name="wpvulnerability-config[notify][email]" value="y" <?php checked( $email_enabled ); ?> />
				<?php esc_html_e( 'Email', 'wpvulnerability' ); ?>
		</label>
		<br/>
		<label>
							<input type="checkbox" name="wpvulnerability-config[notify][slack]" value="y" <?php checked( $slack_enabled ); ?> />
				<?php esc_html_e( 'Slack', 'wpvulnerability' ); ?>
		</label>
		<br/>
		<label>
							<input type="checkbox" name="wpvulnerability-config[notify][teams]" value="y" <?php checked( $teams_enabled ); ?> />
				<?php esc_html_e( 'Microsoft Teams', 'wpvulnerability' ); ?>
		</label>
		</div>
		<?php
}

/**
 * Print the Slack webhook input field.
 *
 * @since 3.6.0
 *
 * @return void
 */
function wpvulnerability_admin_slack_callback() {

		$wpvulnerability_settings = get_site_option( 'wpvulnerability-config', array() );
		$slack_webhook            = isset( $wpvulnerability_settings['slack_webhook'] ) ? $wpvulnerability_settings['slack_webhook'] : '';

	?>
								<input class="regular-text" type="text" name="wpvulnerability-config[slack_webhook]" id="wpvulnerability_slack_webhook" placeholder="<?php echo esc_attr( 'https://hooks.slack.com/services/...' ); ?>" value="<?php echo esc_attr( (string) $slack_webhook ); ?>" />
				<?php
}

/**
 * Print the Teams webhook input field.
 *
 * @since 3.6.0
 *
 * @return void
 */
function wpvulnerability_admin_teams_callback() {

		$wpvulnerability_settings = get_site_option( 'wpvulnerability-config', array() );
		$teams_webhook            = isset( $wpvulnerability_settings['teams_webhook'] ) ? $wpvulnerability_settings['teams_webhook'] : '';

	?>
								<input class="regular-text" type="text" name="wpvulnerability-config[teams_webhook]" id="wpvulnerability_teams_webhook" placeholder="<?php echo esc_attr( 'https://outlook.office.com/webhook/...' ); ?>" value="<?php echo esc_attr( (string) $teams_webhook ); ?>" />
				<?php
}

/**
 * Displays the WPVulnerability plugin analysis settings in the admin panel.
 *
 * This function retrieves the current WPVulnerability analysis settings and
 * ensures all necessary options are set. It then outputs a multiple-select
 * field allowing the user to select which components (core, plugins, themes,
 * php, apache, nginx) to analyze.
 *
 * @since 3.3.0
 *
 * @return void
 */
function wpvulnerability_admin_analyze_callback() {

	// Retrieve the WPVulnerability plugin settings.
	$wpvulnerability_analyze = get_site_option( 'wpvulnerability-analyze', array() );
	$components              = array( 'core', 'plugins', 'themes', 'php', 'apache', 'nginx', 'mariadb', 'mysql', 'imagemagick', 'curl', 'memcached', 'redis', 'sqlite' );
	$forced                  = array();

	foreach ( $components as $component ) {
		if ( ! isset( $wpvulnerability_analyze[ $component ] ) ) {
			$wpvulnerability_analyze[ $component ] = 0;
		}
		$constant             = 'WPVULNERABILITY_HIDE_' . strtoupper( (string) $component );
		$forced[ $component ] = defined( $constant ) && constant( $constant );
		if ( $forced[ $component ] ) {
			$wpvulnerability_analyze[ $component ] = 1;
		}
	}
	?>
	<div id="wpvulnerability_analyze">
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[core]" value="core" <?php checked( $wpvulnerability_analyze['core'] ); ?> <?php disabled( $forced['core'] ); ?> />
		<?php esc_html_e( 'Core', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[plugins]" value="plugins" <?php checked( $wpvulnerability_analyze['plugins'] ); ?> <?php disabled( $forced['plugins'] ); ?> />
		<?php esc_html_e( 'Plugins', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[themes]" value="themes" <?php checked( $wpvulnerability_analyze['themes'] ); ?> <?php disabled( $forced['themes'] ); ?> />
		<?php esc_html_e( 'Themes', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[php]" value="php" <?php checked( $wpvulnerability_analyze['php'] ); ?> <?php disabled( $forced['php'] ); ?> />
		<?php esc_html_e( 'PHP', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[apache]" value="apache" <?php checked( $wpvulnerability_analyze['apache'] ); ?> <?php disabled( $forced['apache'] ); ?> />
		<?php esc_html_e( 'Apache HTTPD', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[nginx]" value="nginx" <?php checked( $wpvulnerability_analyze['nginx'] ); ?> <?php disabled( $forced['nginx'] ); ?> />
		<?php esc_html_e( 'nginx', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[mariadb]" value="mariadb" <?php checked( $wpvulnerability_analyze['mariadb'] ); ?> <?php disabled( $forced['mariadb'] ); ?> />
		<?php esc_html_e( 'MariaDB', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[mysql]" value="mysql" <?php checked( $wpvulnerability_analyze['mysql'] ); ?> <?php disabled( $forced['mysql'] ); ?> />
		<?php esc_html_e( 'MySQL', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[imagemagick]" value="imagemagick" <?php checked( $wpvulnerability_analyze['imagemagick'] ); ?> <?php disabled( $forced['imagemagick'] ); ?> />
		<?php esc_html_e( 'ImageMagick', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[curl]" value="curl" <?php checked( $wpvulnerability_analyze['curl'] ); ?> <?php disabled( $forced['curl'] ); ?> />
		<?php esc_html_e( 'curl', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[memcached]" value="memcached" <?php checked( $wpvulnerability_analyze['memcached'] ); ?> <?php disabled( $forced['memcached'] ); ?> />
		<?php esc_html_e( 'memcached', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[redis]" value="redis" <?php checked( $wpvulnerability_analyze['redis'] ); ?> <?php disabled( $forced['redis'] ); ?> />
		<?php esc_html_e( 'Redis', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[sqlite]" value="sqlite" <?php checked( $wpvulnerability_analyze['sqlite'] ); ?> <?php disabled( $forced['sqlite'] ); ?> />
		<?php esc_html_e( 'SQLite', 'wpvulnerability' ); ?>
	</label>
	<p><a href="https://www.wpvulnerability.com/plugin/#force-hiding-checks" target="_blank"><small><i><?php esc_html_e( 'Read more about how to force the deactivation of an item.', 'wpvulnerability' ); ?></i></small></a></p>
	</div>
	<?php
}

/**
 * Content for the Dashboard Widget
 *
 * @since 2.2.0
 *
 * @return void
 */
function wpvulnerability_admin_dashboard_content() {
	// Inline CSS for dashboard widget.

	// Get vulnerability counts for all components.
	$core_count        = (int) json_decode( get_site_option( 'wpvulnerability-core-vulnerable' ), true );
	$plugins_count     = (int) json_decode( get_site_option( 'wpvulnerability-plugins-vulnerable' ), true );
	$themes_count      = (int) json_decode( get_site_option( 'wpvulnerability-themes-vulnerable' ), true );
	$php_count         = (int) json_decode( get_site_option( 'wpvulnerability-php-vulnerable' ), true );
	$apache_count      = (int) json_decode( get_site_option( 'wpvulnerability-apache-vulnerable' ), true );
	$nginx_count       = (int) json_decode( get_site_option( 'wpvulnerability-nginx-vulnerable' ), true );
	$mariadb_count     = (int) json_decode( get_site_option( 'wpvulnerability-mariadb-vulnerable' ), true );
	$mysql_count       = (int) json_decode( get_site_option( 'wpvulnerability-mysql-vulnerable' ), true );
	$imagemagick_count = (int) json_decode( get_site_option( 'wpvulnerability-imagemagick-vulnerable' ), true );
	$curl_count        = (int) json_decode( get_site_option( 'wpvulnerability-curl-vulnerable' ), true );
	$memcached_count   = (int) json_decode( get_site_option( 'wpvulnerability-memcached-vulnerable' ), true );
	$redis_count       = (int) json_decode( get_site_option( 'wpvulnerability-redis-vulnerable' ), true );
	$sqlite_count      = (int) json_decode( get_site_option( 'wpvulnerability-sqlite-vulnerable' ), true );

	// Calculate total vulnerabilities (only for enabled components).
	$total_vulnerabilities = 0;
	if ( wpvulnerability_analyze_filter( 'core' ) ) {
		$total_vulnerabilities += $core_count;
	}
	if ( wpvulnerability_analyze_filter( 'plugins' ) ) {
		$total_vulnerabilities += $plugins_count;
	}
	if ( wpvulnerability_analyze_filter( 'themes' ) ) {
		$total_vulnerabilities += $themes_count;
	}
	if ( wpvulnerability_analyze_filter( 'php' ) ) {
		$total_vulnerabilities += $php_count;
	}
	if ( wpvulnerability_analyze_filter( 'apache' ) ) {
		$total_vulnerabilities += $apache_count;
	}
	if ( wpvulnerability_analyze_filter( 'nginx' ) ) {
		$total_vulnerabilities += $nginx_count;
	}
	if ( wpvulnerability_analyze_filter( 'mariadb' ) ) {
		$total_vulnerabilities += $mariadb_count;
	}
	if ( wpvulnerability_analyze_filter( 'mysql' ) ) {
		$total_vulnerabilities += $mysql_count;
	}
	if ( wpvulnerability_analyze_filter( 'imagemagick' ) ) {
		$total_vulnerabilities += $imagemagick_count;
	}
	if ( wpvulnerability_analyze_filter( 'curl' ) ) {
		$total_vulnerabilities += $curl_count;
	}
	if ( wpvulnerability_analyze_filter( 'memcached' ) ) {
		$total_vulnerabilities += $memcached_count;
	}
	if ( wpvulnerability_analyze_filter( 'redis' ) ) {
		$total_vulnerabilities += $redis_count;
	}
	if ( wpvulnerability_analyze_filter( 'sqlite' ) ) {
		$total_vulnerabilities += $sqlite_count;
	}

	// Determine status badge.
	$status_class = 'wpvuln-status-secure';
	$status_text  = '✓ ' . __( 'All Clear', 'wpvulnerability' );
	$status_icon  = '✓';

	if ( $total_vulnerabilities > 0 ) {
		if ( ( wpvulnerability_analyze_filter( 'core' ) && $core_count > 0 ) ||
			( wpvulnerability_analyze_filter( 'php' ) && $php_count > 0 ) ||
			$total_vulnerabilities > 5 ) {
			$status_class = 'wpvuln-status-critical';
			$status_icon  = '✕';
			/* translators: %d: number of vulnerabilities */
			$status_text = sprintf( _n( '%d Critical Issue Found', '%d Critical Issues Found', $total_vulnerabilities, 'wpvulnerability' ), $total_vulnerabilities );
		} else {
			$status_class = 'wpvuln-status-warning';
			$status_icon  = '⚠';
			/* translators: %d: number of vulnerabilities */
			$status_text = sprintf( _n( '%d Issue Found', '%d Issues Found', $total_vulnerabilities, 'wpvulnerability' ), $total_vulnerabilities );
		}
	}

	// Get last check time.
	$core_cache      = json_decode( get_site_option( 'wpvulnerability-core-cache' ), true );
	$last_check_text = __( 'Never checked', 'wpvulnerability' );
	if ( $core_cache && is_numeric( $core_cache ) ) {
		$cache_hours     = wpvulnerability_cache_hours();
		$last_check_time = $core_cache - ( $cache_hours * 3600 );
		$time_diff       = time() - $last_check_time;
		if ( $time_diff < 3600 ) {
			/* translators: %d: number of minutes */
			$last_check_text = sprintf( _n( '%d minute ago', '%d minutes ago', floor( $time_diff / 60 ), 'wpvulnerability' ), floor( $time_diff / 60 ) );
		} else {
			/* translators: %d: number of hours */
			$last_check_text = sprintf( _n( '%d hour ago', '%d hours ago', floor( $time_diff / 3600 ), 'wpvulnerability' ), floor( $time_diff / 3600 ) );
		}
	}

	// Status badge.
	echo '<div class="wpvuln-status-badge ' . esc_attr( $status_class ) . '">';
	echo esc_html( $status_text );
	echo '</div>';

	// Meta information.
	echo '<div class="wpvuln-meta">';
	/* translators: %s: time since last check */
	echo esc_html( sprintf( __( 'Last checked: %s', 'wpvulnerability' ), $last_check_text ) );
	echo ' | <a href="' . esc_url( network_admin_url( 'settings.php?page=wpvulnerability-options&tab=tools' ) ) . '" class="wpvuln-refresh-btn">↻ ' . esc_html( __( 'Refresh Now', 'wpvulnerability' ) ) . '</a>';
	echo '</div>';

	// If no vulnerabilities, show empty state.
	if ( 0 === $total_vulnerabilities ) {
		echo '<div class="wpvuln-empty-state">';
		echo '<div class="wpvuln-empty-state-icon">✓</div>';
		echo '<div class="wpvuln-empty-state-title">' . esc_html( __( 'No vulnerabilities detected', 'wpvulnerability' ) ) . '</div>';
		echo '<div class="wpvuln-empty-state-text">' . esc_html( __( 'Your site is up to date and secure', 'wpvulnerability' ) ) . '</div>';
		echo '</div>';
	} else {
		// WordPress Components section.
		echo '<div class="wpvuln-section-title">' . esc_html( __( 'WordPress Components', 'wpvulnerability' ) ) . '</div>';

		// Core.
		if ( wpvulnerability_analyze_filter( 'core' ) ) {
			$badge_class = $core_count > 0 ? 'wpvuln-badge-critical' : 'wpvuln-badge-secure';
			$badge_icon  = $core_count > 0 ? '✕' : '✓';
			echo '<div class="wpvuln-component">';
			echo '<img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-wordpress.svg" width="16" height="16" alt="">';
			echo '<span class="wpvuln-component-name">' . esc_html( __( 'WordPress Core', 'wpvulnerability' ) ) . '</span>';
			echo '<span class="wpvuln-badge ' . esc_attr( $badge_class ) . '">' . esc_html( $badge_icon ) . ' ' . esc_html( (string) $core_count ) . '</span>';
			echo '</div>';
		}

		// Plugins.
		if ( wpvulnerability_analyze_filter( 'plugins' ) ) {
			$badge_class = $plugins_count > 0 ? 'wpvuln-badge-critical' : 'wpvuln-badge-secure';
			$badge_icon  = $plugins_count > 0 ? '✕' : '✓';
			echo '<div class="wpvuln-component">';
			echo '<img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-plugin.svg" width="16" height="16" alt="">';
			echo '<span class="wpvuln-component-name">' . esc_html( __( 'Plugins', 'wpvulnerability' ) ) . '</span>';
			echo '<span class="wpvuln-badge ' . esc_attr( $badge_class ) . '">' . esc_html( $badge_icon ) . ' ' . esc_html( (string) $plugins_count ) . '</span>';
			echo '</div>';
			if ( $plugins_count > 0 ) {
				echo '<div class="wpvuln-plugin-list">';
				echo wpvulnerability_list_plugins(); // phpcs:ignore
				echo '</div>';
			}
		}

		// Themes.
		if ( wpvulnerability_analyze_filter( 'themes' ) ) {
			$badge_class = $themes_count > 0 ? 'wpvuln-badge-critical' : 'wpvuln-badge-secure';
			$badge_icon  = $themes_count > 0 ? '✕' : '✓';
			echo '<div class="wpvuln-component">';
			echo '<img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-theme.svg" width="16" height="16" alt="">';
			echo '<span class="wpvuln-component-name">' . esc_html( __( 'Themes', 'wpvulnerability' ) ) . '</span>';
			echo '<span class="wpvuln-badge ' . esc_attr( $badge_class ) . '">' . esc_html( $badge_icon ) . ' ' . esc_html( (string) $themes_count ) . '</span>';
			echo '</div>';
			if ( $themes_count > 0 ) {
				echo '<div class="wpvuln-plugin-list">';
				echo wpvulnerability_list_themes(); // phpcs:ignore
				echo '</div>';
			}
		}

		// Server Software section.
		$php_version         = wpvulnerability_detect_php();
		$webserver           = wpvulnerability_detect_webserver();
		$sqlserver           = wpvulnerability_detect_sqlserver();
		$show_server_section = false;

		// Check if any server software is detected.
		if ( ( wpvulnerability_analyze_filter( 'php' ) && $php_version ) ||
			( isset( $webserver['id'] ) && isset( $webserver['version'] ) ) ||
			( isset( $sqlserver['id'] ) && isset( $sqlserver['version'] ) ) ||
			wpvulnerability_get_software_version( 'imagemagick' ) ||
			wpvulnerability_get_software_version( 'curl' ) ||
			wpvulnerability_get_software_version( 'memcached' ) ||
			wpvulnerability_get_software_version( 'redis' ) ||
			wpvulnerability_get_software_version( 'sqlite' ) ) {
			$show_server_section = true;
		}

		if ( $show_server_section ) {
			echo '<div class="wpvuln-section-title">' . esc_html( __( 'Server Software', 'wpvulnerability' ) ) . '</div>';
			echo '<div class="wpvuln-grid">';

			// PHP.
			if ( wpvulnerability_analyze_filter( 'php' ) && $php_version ) {
				$badge_class = $php_count > 0 ? 'wpvuln-badge-critical' : 'wpvuln-badge-secure';
				$badge_icon  = $php_count > 0 ? '✕' : '✓';
				echo '<div class="wpvuln-component">';
				echo '<img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-php.svg" width="16" height="16" alt="">';
				echo '<span class="wpvuln-component-name">PHP ' . esc_html( $php_version ) . '</span>';
				echo '<span class="wpvuln-badge ' . esc_attr( $badge_class ) . '">' . esc_html( $badge_icon ) . ' ' . esc_html( (string) $php_count ) . '</span>';
				echo '</div>';
			}

			// Web server (Apache/nginx).
			if ( isset( $webserver['id'] ) && isset( $webserver['version'] ) ) {
				if ( 'apache' === $webserver['id'] && wpvulnerability_analyze_filter( 'apache' ) ) {
					$badge_class = $apache_count > 0 ? 'wpvuln-badge-warning' : 'wpvuln-badge-secure';
					$badge_icon  = $apache_count > 0 ? '✕' : '✓';
					echo '<div class="wpvuln-component">';
					echo '<img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-apache.svg" width="16" height="16" alt="">';
					echo '<span class="wpvuln-component-name">Apache ' . esc_html( $webserver['version'] ) . '</span>';
					echo '<span class="wpvuln-badge ' . esc_attr( $badge_class ) . '">' . esc_html( $badge_icon ) . ' ' . esc_html( (string) $apache_count ) . '</span>';
					echo '</div>';
				} elseif ( 'nginx' === $webserver['id'] && wpvulnerability_analyze_filter( 'nginx' ) ) {
					$badge_class = $nginx_count > 0 ? 'wpvuln-badge-warning' : 'wpvuln-badge-secure';
					$badge_icon  = $nginx_count > 0 ? '✕' : '✓';
					echo '<div class="wpvuln-component">';
					echo '<img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-nginx.svg" width="16" height="16" alt="">';
					echo '<span class="wpvuln-component-name">nginx ' . esc_html( $webserver['version'] ) . '</span>';
					echo '<span class="wpvuln-badge ' . esc_attr( $badge_class ) . '">' . esc_html( $badge_icon ) . ' ' . esc_html( (string) $nginx_count ) . '</span>';
					echo '</div>';
				}
			}

			// Database (MariaDB/MySQL).
			if ( isset( $sqlserver['id'] ) && isset( $sqlserver['version'] ) ) {
				if ( 'mariadb' === $sqlserver['id'] && wpvulnerability_analyze_filter( 'mariadb' ) ) {
					$badge_class = $mariadb_count > 0 ? 'wpvuln-badge-warning' : 'wpvuln-badge-secure';
					$badge_icon  = $mariadb_count > 0 ? '✕' : '✓';
					echo '<div class="wpvuln-component">';
					echo '<img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-mariadb.svg" width="16" height="16" alt="">';
					echo '<span class="wpvuln-component-name">MariaDB ' . esc_html( $sqlserver['version'] ) . '</span>';
					echo '<span class="wpvuln-badge ' . esc_attr( $badge_class ) . '">' . esc_html( $badge_icon ) . ' ' . esc_html( (string) $mariadb_count ) . '</span>';
					echo '</div>';
				} elseif ( 'mysql' === $sqlserver['id'] && wpvulnerability_analyze_filter( 'mysql' ) ) {
					$badge_class = $mysql_count > 0 ? 'wpvuln-badge-warning' : 'wpvuln-badge-secure';
					$badge_icon  = $mysql_count > 0 ? '✕' : '✓';
					echo '<div class="wpvuln-component">';
					echo '<img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-mysql.svg" width="16" height="16" alt="">';
					echo '<span class="wpvuln-component-name">MySQL ' . esc_html( $sqlserver['version'] ) . '</span>';
					echo '<span class="wpvuln-badge ' . esc_attr( $badge_class ) . '">' . esc_html( $badge_icon ) . ' ' . esc_html( (string) $mysql_count ) . '</span>';
					echo '</div>';
				}
			}

			// ImageMagick.
			$imagemagick_version = wpvulnerability_get_software_version( 'imagemagick' );
			if ( $imagemagick_version && wpvulnerability_analyze_filter( 'imagemagick' ) ) {
				$badge_class = $imagemagick_count > 0 ? 'wpvuln-badge-warning' : 'wpvuln-badge-secure';
				$badge_icon  = $imagemagick_count > 0 ? '✕' : '✓';
				echo '<div class="wpvuln-component">';
				echo '<img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-imagemagick.svg" width="16" height="16" alt="">';
				echo '<span class="wpvuln-component-name">ImageMagick ' . esc_html( $imagemagick_version ) . '</span>';
				echo '<span class="wpvuln-badge ' . esc_attr( $badge_class ) . '">' . esc_html( $badge_icon ) . ' ' . esc_html( (string) $imagemagick_count ) . '</span>';
				echo '</div>';
			}

			// curl.
			$curl_version = wpvulnerability_get_software_version( 'curl' );
			if ( $curl_version && wpvulnerability_analyze_filter( 'curl' ) ) {
				$badge_class = $curl_count > 0 ? 'wpvuln-badge-warning' : 'wpvuln-badge-secure';
				$badge_icon  = $curl_count > 0 ? '✕' : '✓';
				echo '<div class="wpvuln-component">';
				echo '<img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-curl.svg" width="16" height="16" alt="">';
				echo '<span class="wpvuln-component-name">curl ' . esc_html( $curl_version ) . '</span>';
				echo '<span class="wpvuln-badge ' . esc_attr( $badge_class ) . '">' . esc_html( $badge_icon ) . ' ' . esc_html( (string) $curl_count ) . '</span>';
				echo '</div>';
			}

			// memcached.
			$memcached_version = wpvulnerability_get_software_version( 'memcached' );
			if ( $memcached_version && wpvulnerability_analyze_filter( 'memcached' ) ) {
				$badge_class = $memcached_count > 0 ? 'wpvuln-badge-warning' : 'wpvuln-badge-secure';
				$badge_icon  = $memcached_count > 0 ? '✕' : '✓';
				echo '<div class="wpvuln-component">';
				echo '<img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-memcached.svg" width="16" height="16" alt="">';
				echo '<span class="wpvuln-component-name">memcached ' . esc_html( $memcached_version ) . '</span>';
				echo '<span class="wpvuln-badge ' . esc_attr( $badge_class ) . '">' . esc_html( $badge_icon ) . ' ' . esc_html( (string) $memcached_count ) . '</span>';
				echo '</div>';
			}

			// Redis.
			$redis_version = wpvulnerability_get_software_version( 'redis' );
			if ( $redis_version && wpvulnerability_analyze_filter( 'redis' ) ) {
				$badge_class = $redis_count > 0 ? 'wpvuln-badge-warning' : 'wpvuln-badge-secure';
				$badge_icon  = $redis_count > 0 ? '✕' : '✓';
				echo '<div class="wpvuln-component">';
				echo '<img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-redis.svg" width="16" height="16" alt="">';
				echo '<span class="wpvuln-component-name">Redis ' . esc_html( $redis_version ) . '</span>';
				echo '<span class="wpvuln-badge ' . esc_attr( $badge_class ) . '">' . esc_html( $badge_icon ) . ' ' . esc_html( (string) $redis_count ) . '</span>';
				echo '</div>';
			}

			// SQLite.
			$sqlite_version = wpvulnerability_get_software_version( 'sqlite' );
			if ( $sqlite_version && wpvulnerability_analyze_filter( 'sqlite' ) ) {
				$badge_class = $sqlite_count > 0 ? 'wpvuln-badge-warning' : 'wpvuln-badge-secure';
				$badge_icon  = $sqlite_count > 0 ? '✕' : '✓';
				echo '<div class="wpvuln-component">';
				echo '<img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-sqlite.svg" width="16" height="16" alt="">';
				echo '<span class="wpvuln-component-name">SQLite ' . esc_html( $sqlite_version ) . '</span>';
				echo '<span class="wpvuln-badge ' . esc_attr( $badge_class ) . '">' . esc_html( $badge_icon ) . ' ' . esc_html( (string) $sqlite_count ) . '</span>';
				echo '</div>';
			}

			echo '</div>';
		}
	}

	// Footer links.
	echo '<div class="wpvuln-footer">';
	if ( version_compare( get_bloginfo( 'version' ), '5.2', '>=' ) ) {
		echo '<a href="' . esc_url( network_admin_url( 'site-health.php' ) ) . '">' . esc_html( __( 'Site Health', 'wpvulnerability' ) ) . '</a> | ';
	}
	echo '<a href="' . esc_url( network_admin_url( 'settings.php?page=wpvulnerability-options' ) ) . '">' . esc_html( __( 'Settings', 'wpvulnerability' ) ) . '</a>';
	echo '</div>';
}

/**
 * Created a widget in the WordPress dashboard with vulnerability info.
 *
 * @since 2.2.0
 *
 * @return void
 */
function wpvulnerability_admin_dashboard() {

	if ( wpvulnerability_capabilities() ) {
		wp_add_dashboard_widget(
			'wpvulnerability',
			__( 'WPVulnerability Status', 'wpvulnerability' ),
			'wpvulnerability_admin_dashboard_content',
			null,
			null,
			'side',
			'high'
		);
	}
}
add_action( 'wp_network_dashboard_setup', 'wpvulnerability_admin_dashboard' );

/**
 * Strictly sanitizes the main configuration (emails and periods).
 *
 * @since 2.0.0
 *
 * @param array $input Input values.
 * @return array Sanitized values.
 */
function wpvulnerability_sanitize_config( $input ) {
	$defaults = array(
		'emails'        => null,
		'period'        => 'weekly',
		'day'           => 'monday',
		'hour'          => 0,
		'minute'        => 0,
		'cache'         => 12,
		'log_retention' => 0,
		'notify'        => array(
			'email' => 'n',
			'slack' => 'n',
			'teams' => 'n',
		),
		'slack_webhook' => '',
		'teams_webhook' => '',
	);

	$current_values = get_site_option( 'wpvulnerability-config', array() );
	if ( ! is_array( $current_values ) ) {
		$current_values = array();
	}

	$sanitized = $current_values;

	// Emails (comma-separated list).
	if ( isset( $input['emails'] ) ) {
		$emails_raw       = explode( ',', $input['emails'] );
		$sanitized_emails = array();

		foreach ( $emails_raw as $email ) {
			$email = sanitize_email( trim( (string) $email ) );
			if ( is_email( $email ) ) {
				$sanitized_emails[] = $email;
			}
		}

		$sanitized['emails'] = count( $sanitized_emails ) ? implode( ',', $sanitized_emails ) : null;
	}

	// Period (daily, weekly, never).
	$allowed_periods = array( 'daily', 'weekly', 'never' );
	if ( isset( $input['period'] ) && in_array( $input['period'], $allowed_periods, true ) ) {
		$sanitized['period'] = $input['period'];
	}

	// Day of week for weekly schedule.
	if ( isset( $input['day'] ) ) {
		$day              = strtolower( (string) $input['day'] );
		$valid            = array( 'sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday' );
		$sanitized['day'] = in_array( $day, $valid, true ) ? $day : 'monday';
	}

	// Time (hour and minute).
	if ( isset( $input['hour'] ) ) {
		$hour              = (int) $input['hour'];
		$sanitized['hour'] = max( 0, min( 23, $hour ) );
	}
	if ( isset( $input['minute'] ) ) {
		$minute              = (int) $input['minute'];
		$sanitized['minute'] = max( 0, min( 59, $minute ) );
	}

	// Cache expiration.
	if ( isset( $input['cache'] ) ) {
		$cache              = (int) $input['cache'];
		$sanitized['cache'] = in_array( $cache, array( 1, 6, 12, 24 ), true ) ? $cache : 12;
	}
	if ( isset( $input['log_retention'] ) ) {
		$retention = (int) $input['log_retention'];
		if ( in_array( $retention, wpvulnerability_get_log_retention_values(), true ) ) {
			$sanitized['log_retention'] = $retention;
		}
	}

	// Notification methods.
	if ( isset( $input['notify'] ) && is_array( $input['notify'] ) ) {
		$notify_input        = array_map( 'sanitize_text_field', (array) wp_unslash( $input['notify'] ) );
		$sanitized['notify'] = wpvulnerability_normalize_notify_settings( $notify_input );
	}

	// Webhooks.
	if ( isset( $input['slack_webhook'] ) ) {
		$slack_url = trim( (string) $input['slack_webhook'] );
		if ( '' !== $slack_url ) {
			$validated_slack = wpvulnerability_validate_webhook_url(
				$slack_url,
				array( 'hooks.slack.com' )
			);
			if ( '' === $validated_slack ) {
				add_settings_error(
					'wpvulnerability-config',
					'invalid-slack-webhook',
					__( 'Invalid Slack webhook URL. Must be a valid HTTPS URL from hooks.slack.com', 'wpvulnerability' ),
					'error'
				);
				$sanitized['slack_webhook'] = '';
			} else {
				$sanitized['slack_webhook'] = $validated_slack;
			}
		} else {
			$sanitized['slack_webhook'] = '';
		}
	}

	if ( isset( $input['teams_webhook'] ) ) {
		$teams_url = trim( (string) $input['teams_webhook'] );
		if ( '' !== $teams_url ) {
			$validated_teams = wpvulnerability_validate_webhook_url(
				$teams_url,
				array( 'office.com', 'office365.com', 'api.hooks.microsoft.com' )
			);
			if ( '' === $validated_teams ) {
				add_settings_error(
					'wpvulnerability-config',
					'invalid-teams-webhook',
					__( 'Invalid Microsoft Teams webhook URL. Must be a valid HTTPS URL from office.com, office365.com, or api.hooks.microsoft.com', 'wpvulnerability' ),
					'error'
				);
				$sanitized['teams_webhook'] = '';
			} else {
				$sanitized['teams_webhook'] = $validated_teams;
			}
		} else {
			$sanitized['teams_webhook'] = '';
		}
	}

	if ( isset( $input['discord_webhook'] ) ) {
		$discord_url = trim( (string) $input['discord_webhook'] );
		if ( '' !== $discord_url ) {
			$validated_discord = wpvulnerability_validate_webhook_url(
				$discord_url,
				array( 'discord.com', 'discordapp.com' )
			);
			if ( '' === $validated_discord ) {
				add_settings_error(
					'wpvulnerability-config',
					'invalid-discord-webhook',
					__( 'Invalid Discord webhook URL. Must be a valid HTTPS URL from discord.com or discordapp.com', 'wpvulnerability' ),
					'error'
				);
				$sanitized['discord_webhook'] = '';
			} else {
				$sanitized['discord_webhook'] = $validated_discord;
			}
		} else {
			$sanitized['discord_webhook'] = '';
		}
	}

	if ( isset( $input['telegram_bot_token'] ) ) {
		$telegram_bot_token = sanitize_text_field( trim( (string) $input['telegram_bot_token'] ) );
		if ( '' !== $telegram_bot_token ) {
			if ( ! preg_match( '/^\d+:[A-Za-z0-9_-]+$/', $telegram_bot_token ) ) {
				add_settings_error(
					'wpvulnerability-config',
					'invalid-telegram-token',
					__( 'Invalid Telegram bot token format. Must be like: 123456:ABC-DEF1234ghIkl-zyx57W2v1u123ew11', 'wpvulnerability' ),
					'error'
				);
				$sanitized['telegram_bot_token'] = '';
			} else {
				$sanitized['telegram_bot_token'] = $telegram_bot_token;
			}
		} else {
			$sanitized['telegram_bot_token'] = '';
		}
	}

	if ( isset( $input['telegram_chat_id'] ) ) {
		$sanitized['telegram_chat_id'] = sanitize_text_field( trim( (string) $input['telegram_chat_id'] ) );
	}

	$sanitized = wp_parse_args( $sanitized, $defaults );

	// Schedule notification based on sanitized values.
	wpvulnerability_schedule_notification_event( $sanitized );

	return $sanitized;
}

/**
 * Sanitizes the messages generated by the plugin (simple messages).
 *
 * @since 2.0.0
 *
 * @param array $input Input values.
 * @return array Sanitized values.
 */
function wpvulnerability_sanitize_messages( $input ) {
	$sanitized = array();

	foreach ( $input as $key => $message ) {
		$sanitized[ sanitize_key( $key ) ] = sanitize_text_field( $message );
	}

	return $sanitized;
}

/**
 * Strictly sanitizes the analysis options (booleans).
 *
 * @since 3.3.0
 *
 * @param array $input Input values.
 * @return array Sanitized values.
 */
function wpvulnerability_sanitize_analyze( $input ) {
	$components = array(
		'core',
		'plugins',
		'themes',
		'php',
		'apache',
		'nginx',
		'mariadb',
		'mysql',
		'imagemagick',
		'curl',
		'memcached',
		'redis',
		'sqlite',
	);

	$sanitized = array();

	foreach ( $components as $component ) {
		$sanitized[ $component ] = isset( $input[ $component ] ) ? (bool) $input[ $component ] : false;

		$constant = 'WPVULNERABILITY_HIDE_' . strtoupper( (string) $component );
		if ( defined( $constant ) && constant( $constant ) ) {
			$sanitized[ $component ] = true;
		}
	}

	return $sanitized;
}

/**
 * Initializes the WP-Admin settings page for the WP Vulnerability plugin.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_init() {

	// Register the plugin settings to be saved in the database.
	register_setting(
		'admin_wpvulnerability_settings',
		'wpvulnerability-config',
		array(
			'sanitize_callback' => 'wpvulnerability_sanitize_config',
			'default'           => array(),
			'show_in_rest'      => false,
			'type'              => 'array',
		)
	);

	// Register messages.
	register_setting(
		'admin_wpvulnerability_settings',
		'wpvulnerability-messages',
		array(
			'sanitize_callback' => 'wpvulnerability_sanitize_messages',
			'default'           => array(),
			'show_in_rest'      => false,
			'type'              => 'array',
		)
	);

	// Add a section to the settings page.
		add_settings_section(
			'admin_wpvulnerability_settings',
			__( 'Receive vulnerability notifications', 'wpvulnerability' ),
			'wpvulnerability_admin_section_notifications',
			'wpvulnerability-config'
		);

		// Add a field to the settings page for the cache expiration time.
		add_settings_field(
			'wpvulnerability_cache',
			__( 'Cache expiration time', 'wpvulnerability' ),
			'wpvulnerability_admin_cache_callback',
			'wpvulnerability-config',
			'admin_wpvulnerability_settings'
		);

		// Add a field to the settings page for the notification period.
		add_settings_field(
			'wpvulnerability_period',
			__( 'How often you want to receive notifications', 'wpvulnerability' ),
			'wpvulnerability_admin_period_callback',
			'wpvulnerability-config',
			'admin_wpvulnerability_settings'
		);

				// Add a field for notification methods.
				add_settings_field(
					'wpvulnerability_notify',
					__( 'Where do you want to receive notifications?', 'wpvulnerability' ),
					'wpvulnerability_admin_notify_callback',
					'wpvulnerability-config',
					'admin_wpvulnerability_settings'
				);

		// Add a field to the settings page for the email addresses.
		add_settings_field(
			'wpvulnerability_emails',
			__( 'Email addresses to notify (separated by commas)', 'wpvulnerability' ),
			'wpvulnerability_admin_emails_callback',
			'wpvulnerability-config',
			'admin_wpvulnerability_settings'
		);

				// Add a field for the Slack webhook.
				add_settings_field(
					'wpvulnerability_slack_webhook',
					__( 'Slack webhook URL', 'wpvulnerability' ),
					'wpvulnerability_admin_slack_callback',
					'wpvulnerability-config',
					'admin_wpvulnerability_settings'
				);

				// Add a field for the Teams webhook.
				add_settings_field(
					'wpvulnerability_teams_webhook',
					__( 'Teams webhook URL', 'wpvulnerability' ),
					'wpvulnerability_admin_teams_callback',
					'wpvulnerability-config',
					'admin_wpvulnerability_settings'
				);

	// Register the plugin settings to be saved in the database.
	register_setting(
		'admin_wpvulnerability_analyze',
		'wpvulnerability-analyze',
		array(
			'sanitize_callback' => 'wpvulnerability_sanitize_analyze',
			'default'           => array(),
			'show_in_rest'      => false,
			'type'              => 'array',
		)
	);

	// Add a section to the settings page.
	add_settings_section(
		'admin_wpvulnerability_analyze',
		__( 'Vulnerabilities to hide', 'wpvulnerability' ),
		'wpvulnerability_admin_section_analyze',
		'wpvulnerability-analyze'
	);

	// Add a field to the settings page for analyzing components.
	add_settings_field(
		'wpvulnerability_analyze',
		__( 'What do you want to hide?', 'wpvulnerability' ),
		'wpvulnerability_admin_analyze_callback',
		'wpvulnerability-analyze',
		'admin_wpvulnerability_analyze'
	);
}
add_action( 'admin_init', 'wpvulnerability_admin_init' );

/**
 * Outputs the Security tab contents for the multisite network admin.
 *
 * Reuses the single-site security display functions since the functionality
 * is identical for both single-site and multisite contexts.
 *
 * @since 4.3.0
 *
 * @return void
 */
function wpvulnerability_render_network_admin_tab_security() {
	?>
	<section class="section wpvulnerability-security-panel">

		<?php wpvulnerability_display_security_status(); ?>
		<?php wpvulnerability_display_detection_methods(); ?>
		<?php wpvulnerability_display_security_logs(); ?>
	</section>
	<?php
}

/**
 * Displays the security status section.
 *
 * Shows current security mode, shell_exec availability, and logging status.
 *
 * @since 4.3.0
 *
 * @return void
 */
function wpvulnerability_display_security_status() {
	$security_mode      = wpvulnerability_get_security_mode();
	$shell_exec_enabled = wpvulnerability_can_shell_exec();
	$log_retention      = wpvulnerability_log_retention_days();
	$total_logs         = wpvulnerability_count_shell_exec_logs();
	$last_log           = null;

	$recent_logs = wpvulnerability_get_shell_exec_logs( 1, 1 );
	if ( ! empty( $recent_logs ) ) {
		$last_log = $recent_logs[0];
	}

	$mode_labels = array(
		'standard' => __( 'Standard (Hybrid Detection)', 'wpvulnerability' ),
		'strict'   => __( 'Strict (Extensions Only)', 'wpvulnerability' ),
		'disabled' => __( 'Disabled (No Detection)', 'wpvulnerability' ),
	);

	$mode_label = isset( $mode_labels[ $security_mode ] ) ? $mode_labels[ $security_mode ] : $security_mode;
	?>
	<div class="wpvulnerability-security-section">
		<h3><?php esc_html_e( 'Security Status', 'wpvulnerability' ); ?></h3>

		<div class="wpvulnerability-status-grid">
			<div class="wpvulnerability-status-item">
				<strong><?php esc_html_e( 'Security Mode', 'wpvulnerability' ); ?></strong>
				<span><?php echo esc_html( $mode_label ); ?></span>
			</div>

			<div class="wpvulnerability-status-item">
				<strong><?php esc_html_e( 'Shell Execution', 'wpvulnerability' ); ?></strong>
				<?php if ( $shell_exec_enabled ) : ?>
					<span class="wpvulnerability-status-badge success"><?php esc_html_e( 'Enabled', 'wpvulnerability' ); ?></span>
				<?php else : ?>
					<span class="wpvulnerability-status-badge disabled"><?php esc_html_e( 'Disabled', 'wpvulnerability' ); ?></span>
				<?php endif; ?>
			</div>

			<div class="wpvulnerability-status-item">
				<strong><?php esc_html_e( 'Audit Logging', 'wpvulnerability' ); ?></strong>
				<?php if ( $log_retention > 0 ) : ?>
					<span class="wpvulnerability-status-badge success">
						<?php
						/* translators: %d: number of days */
						echo esc_html( sprintf( __( 'Enabled (%d days)', 'wpvulnerability' ), $log_retention ) );
						?>
					</span>
				<?php else : ?>
					<span class="wpvulnerability-status-badge disabled"><?php esc_html_e( 'Disabled', 'wpvulnerability' ); ?></span>
				<?php endif; ?>
			</div>

			<div class="wpvulnerability-status-item">
				<strong><?php esc_html_e( 'Total Log Entries', 'wpvulnerability' ); ?></strong>
				<span><?php echo esc_html( number_format_i18n( $total_logs ) ); ?></span>
			</div>
		</div>

		<?php if ( $last_log ) : ?>
			<div class="wpvulnerability-status-item" style="margin-top: 10px;">
				<strong><?php esc_html_e( 'Last Shell Execution', 'wpvulnerability' ); ?></strong>
				<span>
					<?php
					echo esc_html(
						sprintf(
							/* translators: %s: time ago */
							__( '%s ago', 'wpvulnerability' ),
							human_time_diff( strtotime( $last_log->post_date ), time() )
						)
					);
					?>
				</span>
			</div>
		<?php endif; ?>

		<div class="wpvulnerability-info-box">
			<p>
				<?php
				esc_html_e( 'Configure security settings via wp-config.php constants: WPVULNERABILITY_DISABLE_SHELL_EXEC, WPVULNERABILITY_SECURITY_MODE, WPVULNERABILITY_SHELL_EXEC_WHITELIST.', 'wpvulnerability' );
				?>
				<br>
				<?php
				printf(
					/* translators: %s: URL to security documentation */
					wp_kses_post( __( 'See <a href="%s" target="_blank" rel="noopener noreferrer">security configuration documentation</a> for details.', 'wpvulnerability' ) ),
					esc_url( 'https://www.wpvulnerability.com/plugin/#security-configuration' )
				);
				?>
			</p>
		</div>
	</div>
	<?php
}

/**
 * Displays the detection methods section.
 *
 * Shows which detection method was used for each software component
 * with reliability scoring.
 *
 * @since 4.3.0
 *
 * @return void
 */
function wpvulnerability_display_detection_methods() {
	$components = array(
		'imagemagick' => __( 'ImageMagick', 'wpvulnerability' ),
		'redis'       => __( 'Redis', 'wpvulnerability' ),
		'memcached'   => __( 'Memcached', 'wpvulnerability' ),
		'sqlite'      => __( 'SQLite', 'wpvulnerability' ),
	);

	$detections = array();

	foreach ( array_keys( $components ) as $component ) {
		$detection = null;

		switch ( $component ) {
			case 'imagemagick':
				$detection = wpvulnerability_detect_imagemagick();
				break;
			case 'redis':
				$detection = wpvulnerability_detect_redis();
				break;
			case 'memcached':
				$detection = wpvulnerability_detect_memcached();
				break;
			case 'sqlite':
				$detection = wpvulnerability_detect_sqlite();
				break;
		}

		if ( is_array( $detection ) ) {
			$detections[ $component ] = $detection;
		}
	}

	$method_labels = array(
		'imagick_extension'   => __( 'Imagick Extension', 'wpvulnerability' ),
		'redis_extension'     => __( 'Redis Extension', 'wpvulnerability' ),
		'memcached_extension' => __( 'Memcached Extension', 'wpvulnerability' ),
		'memcache_extension'  => __( 'Memcache Extension', 'wpvulnerability' ),
		'sqlite3_extension'   => __( 'SQLite3 Extension', 'wpvulnerability' ),
		'pdo_sqlite'          => __( 'PDO SQLite', 'wpvulnerability' ),
		'shell_exec'          => __( 'Shell Command', 'wpvulnerability' ),
		'binary_exists'       => __( 'Binary Check', 'wpvulnerability' ),
		'none'                => __( 'Not Detected', 'wpvulnerability' ),
	);
	?>
	<div class="wpvulnerability-security-section">
		<h3><?php esc_html_e( 'Software Detection Methods', 'wpvulnerability' ); ?></h3>
		<p><?php esc_html_e( 'Shows which detection method was used for each software component and the reliability score.', 'wpvulnerability' ); ?></p>

		<table class="wpvulnerability-detection-table">
			<thead>
				<tr>
					<th><?php esc_html_e( 'Component', 'wpvulnerability' ); ?></th>
					<th><?php esc_html_e( 'Version', 'wpvulnerability' ); ?></th>
					<th><?php esc_html_e( 'Detection Method', 'wpvulnerability' ); ?></th>
					<th><?php esc_html_e( 'Reliability', 'wpvulnerability' ); ?></th>
				</tr>
			</thead>
			<tbody>
				<?php foreach ( $detections as $component => $detection ) : ?>
					<tr>
						<td><strong><?php echo esc_html( $components[ $component ] ); ?></strong></td>
						<td>
							<?php
							if ( ! empty( $detection['version'] ) && 'unknown' !== $detection['version'] ) {
								echo esc_html( $detection['version'] );
							} else {
								echo '<span style="color: #646970;">' . esc_html__( 'Not detected', 'wpvulnerability' ) . '</span>';
							}
							?>
						</td>
						<td>
							<?php
							$method = isset( $method_labels[ $detection['method'] ] ) ? $method_labels[ $detection['method'] ] : $detection['method'];
							echo esc_html( $method );
							?>
						</td>
						<td>
							<?php
							$reliability = isset( $detection['reliability'] ) ? (int) $detection['reliability'] : 0;

							if ( $reliability > 0 ) {
								$reliability_class = 'low';
								if ( $reliability >= 80 ) {
									$reliability_class = '';
								} elseif ( $reliability >= 50 ) {
									$reliability_class = 'medium';
								}
								?>
								<span class="wpvulnerability-reliability-bar">
									<span class="wpvulnerability-reliability-fill <?php echo esc_attr( $reliability_class ); ?>" style="width: <?php echo esc_attr( $reliability ); ?>%;"></span>
								</span>
								<span><?php echo esc_html( $reliability ); ?>%</span>
								<?php
							} else {
								?>
								<span style="color: #646970;">—</span>
								<?php
							}
							?>
						</td>
					</tr>
				<?php endforeach; ?>
			</tbody>
		</table>
	</div>
	<?php
}

/**
 * Displays the shell execution audit logs section.
 *
 * Shows recent shell_exec calls with full details for security auditing.
 *
 * @since 4.3.0
 *
 * @return void
 */
function wpvulnerability_display_security_logs() {
	$per_page = 20;
	$logs     = wpvulnerability_get_shell_exec_logs( $per_page, 1 );
	?>
	<div class="wpvulnerability-security-section">
		<h3><?php esc_html_e( 'Shell Execution Audit Logs', 'wpvulnerability' ); ?></h3>
		<p><?php esc_html_e( 'Complete audit trail of shell command executions for security monitoring.', 'wpvulnerability' ); ?></p>

		<?php if ( empty( $logs ) ) : ?>
			<div class="wpvulnerability-empty-state">
				<div class="wpvulnerability-empty-state-icon">🔒</div>
				<p><strong><?php esc_html_e( 'No shell execution logs found.', 'wpvulnerability' ); ?></strong></p>
				<p><?php esc_html_e( 'Logs will appear here when shell commands are executed for software detection.', 'wpvulnerability' ); ?></p>
			</div>
		<?php else : ?>
			<table class="wpvulnerability-logs-table">
				<thead>
					<tr>
						<th><?php esc_html_e( 'Time', 'wpvulnerability' ); ?></th>
						<th><?php esc_html_e( 'Component', 'wpvulnerability' ); ?></th>
						<th><?php esc_html_e( 'Command', 'wpvulnerability' ); ?></th>
						<th><?php esc_html_e( 'Status', 'wpvulnerability' ); ?></th>
						<th><?php esc_html_e( 'User', 'wpvulnerability' ); ?></th>
						<th><?php esc_html_e( 'Output', 'wpvulnerability' ); ?></th>
					</tr>
				</thead>
				<tbody>
					<?php foreach ( $logs as $log ) : ?>
						<?php
						$log_data = json_decode( $log->post_content, true );
						if ( ! $log_data ) {
							continue;
						}
						?>
						<tr>
							<td>
								<?php
								echo esc_html(
									sprintf(
										/* translators: %s: time ago */
										__( '%s ago', 'wpvulnerability' ),
										human_time_diff( strtotime( $log->post_date ), time() )
									)
								);
								?>
							</td>
							<td><code><?php echo esc_html( $log_data['component'] ); ?></code></td>
							<td><code><?php echo esc_html( $log_data['command'] ); ?></code></td>
							<td>
								<?php if ( $log_data['success'] ) : ?>
									<span class="wpvulnerability-status-badge success"><?php echo esc_html( ucfirst( $log_data['reason'] ) ); ?></span>
								<?php else : ?>
									<span class="wpvulnerability-status-badge warning"><?php echo esc_html( ucfirst( $log_data['reason'] ) ); ?></span>
								<?php endif; ?>
							</td>
							<td><?php echo esc_html( $log_data['user'] ); ?></td>
							<td>
								<?php if ( ! empty( $log_data['output'] ) ) : ?>
									<div class="wpvulnerability-log-output" title="<?php echo esc_attr( $log_data['output'] ); ?>">
										<?php echo esc_html( substr( $log_data['output'], 0, 50 ) ); ?>
										<?php if ( strlen( $log_data['output'] ) > 50 ) : ?>
											<span>...</span>
										<?php endif; ?>
									</div>
								<?php else : ?>
									<span style="color: #646970;">—</span>
								<?php endif; ?>
							</td>
						</tr>
					<?php endforeach; ?>
				</tbody>
			</table>

			<p style="margin-top: 15px; color: #646970; font-size: 13px;">
				<?php
				/* translators: %d: number of logs shown */
				echo esc_html( sprintf( __( 'Showing the %d most recent log entries.', 'wpvulnerability' ), $per_page ) );
				?>
			</p>
		<?php endif; ?>
	</div>
	<?php
}

/**
 * AJAX handler for testing API connectivity.
 *
 * @since 4.3.0
 *
 * @return void
 */

/**
 * Outputs the Debug tab contents for the multisite network admin.
 *
 * Reuses the single-site debug display functions since the functionality
 * is identical for both single-site and multisite contexts.
 *
 * @since 4.3.0
 *
 * @return void
 */
function wpvulnerability_render_network_admin_tab_debug() {
	// Security check: only show in debug mode.
	if ( ! defined( 'WP_DEBUG' ) || ! WP_DEBUG ) {
		return;
	}

	// Check permissions for multisite.
	if ( ! current_user_can( 'manage_network_options' ) ) {
		echo '<div class="notice notice-error"><p>';
		esc_html_e( 'You do not have permission to access this page.', 'wpvulnerability' );
		echo '</p></div>';
		return;
	}

	// Load debug functions if not already loaded.
	if ( ! function_exists( 'wpvulnerability_debug_get_system_info' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-debug.php';
	}

	?>
	<div class="wpvulnerability-section">
		<h2><?php esc_html_e( 'Debug Information', 'wpvulnerability' ); ?></h2>
		<p style="color: #d63638; font-weight: 500;">
			<?php esc_html_e( 'This tab is only visible when WP_DEBUG is enabled. Use with caution.', 'wpvulnerability' ); ?>
		</p>

		<?php
		// Section 1: System Information.
		wpvulnerability_render_debug_section_system_info();

		// Section 2: Component Detection.
		wpvulnerability_render_debug_section_components();

		// Section 4: Configuration Summary.
		wpvulnerability_render_debug_section_config();

		// Section 5: Cron Status.
		wpvulnerability_render_debug_section_cron();

		// Section 3: API Testing.
		wpvulnerability_render_debug_section_api_testing();

		// Section 6: Database Options Viewer.
		wpvulnerability_render_debug_section_database_options();

		// Section 7: Quick Actions.
		wpvulnerability_render_debug_section_quick_actions();
		?>
	</div>
	<?php
}

/**
 * AJAX handler for testing API connectivity in multisite.
 *
 * @since 4.3.0
 *
 * @return void
 */
function wpvulnerability_ajax_network_test_api() {
	// Verify nonce.
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_key( $_POST['nonce'] ), 'wpvulnerability_test_api' ) ) {
		wp_send_json_error( array( 'message' => __( 'Invalid nonce.', 'wpvulnerability' ) ) );
	}

	// Check permissions.
	if ( ! current_user_can( 'manage_network_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'wpvulnerability' ) ) );
	}

	// Get component.
	$component = isset( $_POST['component'] ) ? sanitize_key( $_POST['component'] ) : '';
	if ( empty( $component ) ) {
		wp_send_json_error( array( 'message' => __( 'No component specified.', 'wpvulnerability' ) ) );
	}

	// Load debug functions.
	if ( ! function_exists( 'wpvulnerability_debug_test_api_component' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-debug.php';
	}

	// Test API.
	$wpvulnerability_result = wpvulnerability_debug_test_api_component( $component );

	wp_send_json_success( $wpvulnerability_result );
}
add_action( 'wp_ajax_wpvulnerability_test_api', 'wpvulnerability_ajax_network_test_api' );

/**
 * Renders Section 1: System Information.
 *
 * @since 4.3.0
 *
 * @return void
 */
function wpvulnerability_render_debug_section_system_info() {
	$system_info = wpvulnerability_debug_get_system_info();
	?>
	<div class="wpvulnerability-subsection" style="margin-top: 20px;">
		<h3><?php esc_html_e( 'System Information', 'wpvulnerability' ); ?></h3>
		<table class="widefat" style="margin-top: 10px;">
			<tbody>
				<tr>
					<td style="width: 200px; font-weight: 600;"><?php esc_html_e( 'WordPress Version', 'wpvulnerability' ); ?></td>
					<td><?php echo esc_html( $system_info['wordpress']['version'] ); ?></td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Multisite', 'wpvulnerability' ); ?></td>
					<td><?php echo $system_info['wordpress']['multisite'] ? esc_html__( 'Yes', 'wpvulnerability' ) : esc_html__( 'No', 'wpvulnerability' ); ?></td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Language', 'wpvulnerability' ); ?></td>
					<td><?php echo esc_html( $system_info['wordpress']['language'] ); ?></td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'PHP Version', 'wpvulnerability' ); ?></td>
					<td><?php echo esc_html( $system_info['php']['version'] ); ?></td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'PHP Extensions', 'wpvulnerability' ); ?></td>
					<td>
						<?php foreach ( $system_info['php']['extensions'] as $ext => $loaded ) : ?>
							<span style="margin-right: 10px;">
								<span style="color: <?php echo $loaded ? '#00a32a' : '#d63638'; ?>;">●</span>
								<?php echo esc_html( $ext ); ?>
							</span>
						<?php endforeach; ?>
					</td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'PHP Memory Limit', 'wpvulnerability' ); ?></td>
					<td><?php echo esc_html( $system_info['php']['memory']['limit'] ); ?></td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Database', 'wpvulnerability' ); ?></td>
					<td><?php echo esc_html( $system_info['database']['type'] . ' ' . $system_info['database']['version'] ); ?></td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Web Server', 'wpvulnerability' ); ?></td>
					<td><?php echo esc_html( $system_info['webserver']['software'] ); ?></td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'WP_DEBUG', 'wpvulnerability' ); ?></td>
					<td>
						<span style="color: <?php echo $system_info['debug']['wp_debug'] ? '#00a32a' : '#d63638'; ?>; font-weight: 600;">
							<?php echo $system_info['debug']['wp_debug'] ? esc_html__( 'Enabled', 'wpvulnerability' ) : esc_html__( 'Disabled', 'wpvulnerability' ); ?>
						</span>
					</td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'WP_DEBUG_LOG', 'wpvulnerability' ); ?></td>
					<td>
						<span style="color: <?php echo $system_info['debug']['wp_debug_log'] ? '#00a32a' : '#d63638'; ?>; font-weight: 600;">
							<?php echo $system_info['debug']['wp_debug_log'] ? esc_html__( 'Enabled', 'wpvulnerability' ) : esc_html__( 'Disabled', 'wpvulnerability' ); ?>
						</span>
						<?php if ( $system_info['debug']['wp_debug_log'] && isset( $system_info['debug']['log_file'] ) ) : ?>
							<?php $log_file = $system_info['debug']['log_file']; ?>
							<?php if ( ! empty( $log_file['path'] ) ) : ?>
								<br>
								<span style="color: #646970; font-size: 12px;">
									<?php esc_html_e( 'Path:', 'wpvulnerability' ); ?>
									<code style="background: #f0f0f0; padding: 2px 6px; border-radius: 3px;"><?php echo esc_html( $log_file['path'] ); ?></code>
								</span>
								<?php if ( $log_file['exists'] ) : ?>
									<br>
									<span style="color: #646970; font-size: 12px;">
										<?php
										/* translators: %s: file size */
										echo esc_html( sprintf( __( 'Size: %s', 'wpvulnerability' ), size_format( $log_file['size'] ) ) );
										?>
									</span>
									<?php if ( $log_file['accessible'] && ! empty( $log_file['url'] ) ) : ?>
										<br>
										<a href="<?php echo esc_url( $log_file['url'] ); ?>" target="_blank" rel="noopener noreferrer" class="button button-small" style="margin-top: 5px;">
											<?php esc_html_e( 'View Log File', 'wpvulnerability' ); ?>
											<span class="dashicons dashicons-external" style="font-size: 13px; margin-top: 3px;"></span>
										</a>
									<?php elseif ( $log_file['exists'] ) : ?>
										<br>
										<span style="color: #d63638; font-size: 12px;">
											<?php esc_html_e( 'Log file is not web-accessible (outside wp-content)', 'wpvulnerability' ); ?>
										</span>
									<?php endif; ?>
								<?php else : ?>
									<br>
									<span style="color: #646970; font-size: 12px;">
										<?php esc_html_e( 'Log file does not exist yet', 'wpvulnerability' ); ?>
									</span>
								<?php endif; ?>
							<?php endif; ?>
						<?php endif; ?>
					</td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Plugin Version', 'wpvulnerability' ); ?></td>
					<td><?php echo esc_html( $system_info['plugin']['version'] ); ?></td>
				</tr>
			</tbody>
		</table>
	</div>
	<?php
}

/**
 * Renders Section 2: Component Detection.
 *
 * @since 4.3.0
 *
 * @return void
 */
function wpvulnerability_render_debug_section_components() {
	$components = wpvulnerability_debug_get_component_status();
	?>
	<div class="wpvulnerability-subsection" style="margin-top: 30px;">
		<h3><?php esc_html_e( 'Component Detection', 'wpvulnerability' ); ?></h3>
		<table class="widefat striped" style="margin-top: 10px;">
			<thead>
				<tr>
					<th><?php esc_html_e( 'Component', 'wpvulnerability' ); ?></th>
					<th><?php esc_html_e( 'Status', 'wpvulnerability' ); ?></th>
					<th><?php esc_html_e( 'Version Detected', 'wpvulnerability' ); ?></th>
					<th><?php esc_html_e( 'Analyzed', 'wpvulnerability' ); ?></th>
					<th><?php esc_html_e( 'Cache Status', 'wpvulnerability' ); ?></th>
				</tr>
			</thead>
			<tbody>
				<?php foreach ( $components as $comp ) : ?>
					<tr>
						<td style="font-weight: 600;"><?php echo esc_html( ucfirst( $comp['component'] ) ); ?></td>
						<td>
							<span style="color: <?php echo $comp['detected'] ? '#00a32a' : '#d63638'; ?>; font-size: 16px;">
								<?php echo $comp['detected'] ? '✓' : '✗'; ?>
							</span>
						</td>
						<td><?php echo esc_html( $comp['version'] ); ?></td>
						<td>
							<span style="color: <?php echo $comp['analyzed'] ? '#00a32a' : '#d63638'; ?>;">
								<?php echo $comp['analyzed'] ? esc_html__( 'Yes', 'wpvulnerability' ) : esc_html__( 'No', 'wpvulnerability' ); ?>
							</span>
						</td>
						<td>
							<?php
							$badge_color = '#646970';
							if ( strpos( $comp['cache_status'], 'Fresh' ) !== false ) {
								$badge_color = '#00a32a';
							} elseif ( strpos( $comp['cache_status'], 'Expired' ) !== false ) {
								$badge_color = '#d63638';
							}
							?>
							<span style="background: <?php echo esc_attr( $badge_color ); ?>; color: white; padding: 3px 8px; border-radius: 3px; font-size: 11px; font-weight: 600;">
								<?php echo esc_html( $comp['cache_status'] ); ?>
							</span>
						</td>
					</tr>
				<?php endforeach; ?>
			</tbody>
		</table>
	</div>
	<?php
}

/**
 * Renders Section 4: Configuration Summary.
 *
 * @since 4.3.0
 *
 * @return void
 */
function wpvulnerability_render_debug_section_config() {
	$config = is_multisite()
		? get_site_option( 'wpvulnerability-config', array() )
		: get_option( 'wpvulnerability-config', array() );
	?>
	<div class="wpvulnerability-subsection" style="margin-top: 30px;">
		<h3><?php esc_html_e( 'Configuration Summary', 'wpvulnerability' ); ?></h3>
		<table class="widefat" style="margin-top: 10px;">
			<tbody>
				<tr>
					<td style="width: 250px; font-weight: 600;"><?php esc_html_e( 'Cache Duration', 'wpvulnerability' ); ?></td>
					<td>
						<?php
						$cache_hours = isset( $config['cache'] ) ? absint( $config['cache'] ) : 12;
						/* translators: %d: number of hours */
						echo esc_html( sprintf( _n( '%d hour', '%d hours', $cache_hours, 'wpvulnerability' ), $cache_hours ) );
						?>
					</td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Log Retention', 'wpvulnerability' ); ?></td>
					<td>
						<?php
						$retention_days = isset( $config['log_retention_days'] ) ? absint( $config['log_retention_days'] ) : 14;
						/* translators: %d: number of days */
						echo esc_html( sprintf( _n( '%d day', '%d days', $retention_days, 'wpvulnerability' ), $retention_days ) );
						?>
					</td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Components Analyzed', 'wpvulnerability' ); ?></td>
					<td>
						<?php
						$components     = array( 'core', 'plugins', 'themes', 'php', 'apache', 'nginx', 'mysql', 'mariadb', 'imagemagick', 'curl', 'memcached', 'redis', 'sqlite' );
						$analyzed_count = 0;
						foreach ( $components as $component ) {
							if ( wpvulnerability_analyze_filter( $component ) ) {
								++$analyzed_count;
							}
						}
						echo esc_html( sprintf( '%d / %d', $analyzed_count, count( $components ) ) );
						?>
					</td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Notification Email', 'wpvulnerability' ); ?></td>
					<td><?php echo isset( $config['emails'] ) ? esc_html( $config['emails'] ) : '—'; ?></td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Notification Frequency', 'wpvulnerability' ); ?></td>
					<td>
						<?php
						$period = isset( $config['period'] ) ? $config['period'] : 'weekly';
						echo esc_html( ucfirst( $period ) );
						if ( 'weekly' === $period && isset( $config['day'] ) ) {
							echo ' (' . esc_html( ucfirst( $config['day'] ) ) . ')';
						}
						if ( isset( $config['hour'] ) && isset( $config['minute'] ) ) {
							echo ' ' . esc_html( sprintf( '%02d:%02d', absint( $config['hour'] ), absint( $config['minute'] ) ) );
						}
						?>
					</td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Slack Webhook', 'wpvulnerability' ); ?></td>
					<td>
						<?php
						$slack_configured = ! empty( $config['slack_webhook'] );
						$slack_enabled    = isset( $config['notify']['slack'] ) && 'y' === $config['notify']['slack'];
						?>
						<span style="color: <?php echo ( $slack_configured && $slack_enabled ) ? '#00a32a' : '#646970'; ?>;">
							<?php
							if ( $slack_configured && $slack_enabled ) {
								esc_html_e( 'Configured & Enabled', 'wpvulnerability' );
							} elseif ( $slack_configured ) {
								esc_html_e( 'Configured (Disabled)', 'wpvulnerability' );
							} else {
								esc_html_e( 'Not configured', 'wpvulnerability' );
							}
							?>
						</span>
					</td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Teams Webhook', 'wpvulnerability' ); ?></td>
					<td>
						<?php
						$teams_configured = ! empty( $config['teams_webhook'] );
						$teams_enabled    = isset( $config['notify']['teams'] ) && 'y' === $config['notify']['teams'];
						?>
						<span style="color: <?php echo ( $teams_configured && $teams_enabled ) ? '#00a32a' : '#646970'; ?>;">
							<?php
							if ( $teams_configured && $teams_enabled ) {
								esc_html_e( 'Configured & Enabled', 'wpvulnerability' );
							} elseif ( $teams_configured ) {
								esc_html_e( 'Configured (Disabled)', 'wpvulnerability' );
							} else {
								esc_html_e( 'Not configured', 'wpvulnerability' );
							}
							?>
						</span>
					</td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Discord Webhook', 'wpvulnerability' ); ?></td>
					<td>
						<?php
						$discord_configured = ! empty( $config['discord_webhook'] );
						$discord_enabled    = isset( $config['notify']['discord'] ) && 'y' === $config['notify']['discord'];
						?>
						<span style="color: <?php echo ( $discord_configured && $discord_enabled ) ? '#00a32a' : '#646970'; ?>;">
							<?php
							if ( $discord_configured && $discord_enabled ) {
								esc_html_e( 'Configured & Enabled', 'wpvulnerability' );
							} elseif ( $discord_configured ) {
								esc_html_e( 'Configured (Disabled)', 'wpvulnerability' );
							} else {
								esc_html_e( 'Not configured', 'wpvulnerability' );
							}
							?>
						</span>
					</td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Telegram Bot', 'wpvulnerability' ); ?></td>
					<td>
						<?php
						$telegram_configured = ! empty( $config['telegram_bot_token'] ) && ! empty( $config['telegram_chat_id'] );
						$telegram_enabled    = isset( $config['notify']['telegram'] ) && 'y' === $config['notify']['telegram'];
						?>
						<span style="color: <?php echo ( $telegram_configured && $telegram_enabled ) ? '#00a32a' : '#646970'; ?>;">
							<?php
							if ( $telegram_configured && $telegram_enabled ) {
								esc_html_e( 'Configured & Enabled', 'wpvulnerability' );
							} elseif ( $telegram_configured ) {
								esc_html_e( 'Configured (Disabled)', 'wpvulnerability' );
							} else {
								esc_html_e( 'Not configured', 'wpvulnerability' );
							}
							?>
						</span>
					</td>
				</tr>
			</tbody>
		</table>
	</div>
	<?php
}

/**
 * Renders Section 5: Cron Status.
 *
 * @since 4.3.0
 *
 * @return void
 */
function wpvulnerability_render_debug_section_cron() {
	$cron_status = wpvulnerability_debug_get_cron_status();
	$can_manage  = current_user_can( 'manage_network_options' );
	?>
	<div class="wpvulnerability-subsection" style="margin-top: 30px;">
		<h3><?php esc_html_e( 'Cron Status', 'wpvulnerability' ); ?></h3>
		<table class="widefat" style="margin-top: 10px;">
			<tbody>
				<tr>
					<td style="width: 250px; font-weight: 600;"><?php esc_html_e( 'Update Database', 'wpvulnerability' ); ?></td>
					<td>
						<?php if ( $cron_status['update_database']['scheduled'] ) : ?>
							<?php
							/* translators: %s: date and time */
							echo esc_html( sprintf( __( 'Next run: %s', 'wpvulnerability' ), wp_date( 'Y-m-d H:i:s', $cron_status['update_database']['next_run'] ) ) );
							?>
						<?php else : ?>
							<span style="color: #d63638;"><?php esc_html_e( 'Not scheduled', 'wpvulnerability' ); ?></span>
						<?php endif; ?>
					</td>
				</tr>
				<tr>
					<td style="font-weight: 600;"><?php esc_html_e( 'Send Notification', 'wpvulnerability' ); ?></td>
					<td>
						<?php if ( $cron_status['send_notification']['scheduled'] ) : ?>
							<?php
							/* translators: %s: date and time */
							echo esc_html( sprintf( __( 'Next run: %s', 'wpvulnerability' ), wp_date( 'Y-m-d H:i:s', $cron_status['send_notification']['next_run'] ) ) );
							?>
						<?php else : ?>
							<span style="color: #646970;"><?php esc_html_e( 'Not scheduled', 'wpvulnerability' ); ?></span>
						<?php endif; ?>
					</td>
				</tr>
			</tbody>
		</table>

		<?php if ( $can_manage ) : ?>
			<div style="margin-top: 15px;">
				<form method="post" style="display: inline-block; margin-right: 10px;">
					<?php wp_nonce_field( 'wpvulnerability_cron_action', 'wpvulnerability_cron_nonce' ); ?>
					<input type="hidden" name="wpvulnerability_run_update" value="1">
					<button type="submit" class="button button-secondary">
						<?php esc_html_e( 'Run Update Now', 'wpvulnerability' ); ?>
					</button>
				</form>

				<form method="post" style="display: inline-block;">
					<?php wp_nonce_field( 'wpvulnerability_cron_action', 'wpvulnerability_cron_nonce' ); ?>
					<input type="hidden" name="wpvulnerability_run_notification" value="1">
					<button type="submit" class="button button-secondary">
						<?php esc_html_e( 'Run Notification Now', 'wpvulnerability' ); ?>
					</button>
				</form>
			</div>
		<?php else : ?>
			<p style="color: #d63638; margin-top: 10px;">
				<?php esc_html_e( 'You do not have permission to run these actions.', 'wpvulnerability' ); ?>
			</p>
		<?php endif; ?>
	</div>
	<?php
}

/**
 * Renders Section 3: API Testing.
 *
 * @since 4.3.0
 *
 * @return void
 */
function wpvulnerability_render_debug_section_api_testing() {
	$can_manage = current_user_can( 'manage_network_options' );
	$components = array( 'core', 'plugins', 'themes', 'php', 'apache', 'nginx', 'mysql', 'mariadb', 'imagemagick', 'curl', 'memcached', 'redis', 'sqlite' );
	?>
	<div class="wpvulnerability-subsection" style="margin-top: 30px;">
		<h3><?php esc_html_e( 'API Testing', 'wpvulnerability' ); ?></h3>

		<?php if ( $can_manage ) : ?>
			<p><?php esc_html_e( 'Test API connectivity for each component. Click a button to send a test request.', 'wpvulnerability' ); ?></p>

			<div id="wpvulnerability-api-test-buttons" style="margin-top: 15px;">
				<?php foreach ( $components as $component ) : ?>
					<button type="button" class="button button-secondary wpvulnerability-test-api-btn" data-component="<?php echo esc_attr( $component ); ?>" style="margin: 5px;">
						<?php
						/* translators: %s: component name */
						echo esc_html( sprintf( __( 'Test %s', 'wpvulnerability' ), ucfirst( $component ) ) );
						?>
					</button>
				<?php endforeach; ?>
			</div>

			<div id="wpvulnerability-api-test-results" style="margin-top: 20px;"></div>

			<script type="text/javascript">
			jQuery(document).ready(function($) {
				$('.wpvulnerability-test-api-btn').on('click', function() {
					var $btn = $(this);
					var component = $btn.data('component');
					var $results = $('#wpvulnerability-api-test-results');

					$btn.prop('disabled', true).text('<?php echo esc_js( __( 'Testing...', 'wpvulnerability' ) ); ?>');

					$.ajax({
						url: typeof ajaxurl !== 'undefined' ? ajaxurl : wpvulnerabilityAjax.ajaxurl,
						type: 'POST',
						data: {
							action: 'wpvulnerability_test_api',
							component: component,
							nonce: '<?php echo esc_js( wp_create_nonce( 'wpvulnerability_test_api' ) ); ?>'
						},
						success: function(response) {
							if (response.success) {
								var result = response.data;
								var statusColor = result.success ? '#00a32a' : '#d63638';
								var resultHtml = '<div style="border: 1px solid ' + statusColor + '; padding: 15px; margin-top: 10px; border-radius: 4px;">';
								resultHtml += '<h4 style="margin-top: 0; color: ' + statusColor + ';">' + component.toUpperCase() + ' - ' + result.message + '</h4>';
								resultHtml += '<p><strong><?php echo esc_js( __( 'HTTP Code:', 'wpvulnerability' ) ); ?></strong> ' + result.http_code + '</p>';
								resultHtml += '<p><strong><?php echo esc_js( __( 'Response Time:', 'wpvulnerability' ) ); ?></strong> ' + result.response_time + ' ms</p>';
								if (result.data_preview) {
									resultHtml += '<details style="margin-top: 10px;"><summary style="cursor: pointer; font-weight: 600;"><?php echo esc_js( __( 'Response Preview', 'wpvulnerability' ) ); ?></summary>';
									resultHtml += '<pre style="background: #f0f0f0; padding: 10px; overflow-x: auto; margin-top: 10px;">' + result.data_preview + '</pre>';
									resultHtml += '</details>';
								}
								resultHtml += '</div>';
								$results.prepend(resultHtml);
							}
							$btn.prop('disabled', false).text('<?php echo esc_js( __( 'Test', 'wpvulnerability' ) ); ?> ' + component.charAt(0).toUpperCase() + component.slice(1));
						},
						error: function() {
							$results.prepend('<div style="border: 1px solid #d63638; padding: 15px; margin-top: 10px; border-radius: 4px; color: #d63638;"><strong><?php echo esc_js( __( 'Error:', 'wpvulnerability' ) ); ?></strong> <?php echo esc_js( __( 'AJAX request failed.', 'wpvulnerability' ) ); ?></div>');
							$btn.prop('disabled', false).text('<?php echo esc_js( __( 'Test', 'wpvulnerability' ) ); ?> ' + component.charAt(0).toUpperCase() + component.slice(1));
						}
					});
				});
			});
			</script>
		<?php else : ?>
			<p style="color: #d63638;">
				<?php esc_html_e( 'You do not have permission to test API connectivity.', 'wpvulnerability' ); ?>
			</p>
		<?php endif; ?>
	</div>
	<?php
}

/**
 * Renders Section 6: Database Options Viewer.
 *
 * @since 4.3.0
 *
 * @return void
 */
function wpvulnerability_render_debug_section_database_options() {
	$option_names    = wpvulnerability_debug_get_option_names();
	$selected_option = filter_input( INPUT_GET, 'debug_option', FILTER_SANITIZE_SPECIAL_CHARS );
	$selected_option = $selected_option ? sanitize_key( $selected_option ) : '';
	$page_value      = filter_input( INPUT_GET, 'page', FILTER_SANITIZE_SPECIAL_CHARS );
	?>
	<div class="wpvulnerability-subsection" style="margin-top: 30px;">
		<h3><?php esc_html_e( 'Database Options Viewer', 'wpvulnerability' ); ?></h3>
		<p><?php esc_html_e( 'View the raw data stored in WordPress options.', 'wpvulnerability' ); ?></p>

		<form method="get" style="margin-top: 15px;">
			<input type="hidden" name="page" value="<?php echo $page_value ? esc_attr( $page_value ) : ''; ?>">
			<input type="hidden" name="tab" value="debug">
			<select name="debug_option" style="min-width: 300px;">
				<option value=""><?php esc_html_e( 'Select an option...', 'wpvulnerability' ); ?></option>
				<?php foreach ( $option_names as $option ) : ?>
					<option value="<?php echo esc_attr( $option ); ?>" <?php selected( $selected_option, $option ); ?>>
						<?php echo esc_html( $option ); ?>
					</option>
				<?php endforeach; ?>
			</select>
			<button type="submit" class="button button-secondary"><?php esc_html_e( 'View', 'wpvulnerability' ); ?></button>
		</form>

		<?php if ( $selected_option && in_array( $selected_option, $option_names, true ) ) : ?>
			<?php
			$option_value = wpvulnerability_debug_get_option_value( $selected_option );
			?>
			<div style="margin-top: 20px;">
				<h4><?php echo esc_html( $selected_option ); ?></h4>
				<?php if ( null !== $option_value ) : ?>
					<pre style="background: #f0f0f0; padding: 15px; overflow-x: auto; border: 1px solid #ddd; border-radius: 4px;"><?php echo esc_html( wp_json_encode( $option_value, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES ) ); ?></pre>
				<?php else : ?>
					<p style="color: #646970;"><?php esc_html_e( 'Option not found or empty.', 'wpvulnerability' ); ?></p>
				<?php endif; ?>
			</div>
		<?php endif; ?>
	</div>
	<?php
}

/**
 * Renders Section 7: Quick Actions.
 *
 * @since 4.3.0
 *
 * @return void
 */
function wpvulnerability_render_debug_section_quick_actions() {
	$can_manage = current_user_can( 'manage_network_options' );
	?>
	<div class="wpvulnerability-subsection" style="margin-top: 30px;">
		<h3><?php esc_html_e( 'Quick Actions', 'wpvulnerability' ); ?></h3>
		<p><?php esc_html_e( 'Perform debugging actions. Use with caution.', 'wpvulnerability' ); ?></p>

		<?php if ( $can_manage ) : ?>
			<div style="margin-top: 15px;">
				<form method="post" style="display: inline-block; margin-right: 10px;" onsubmit="return confirm('<?php echo esc_js( __( 'Are you sure you want to clear all caches?', 'wpvulnerability' ) ); ?>');">
					<?php wp_nonce_field( 'wpvulnerability_debug_action', 'wpvulnerability_debug_nonce' ); ?>
					<input type="hidden" name="wpvulnerability_debug_clear_caches" value="1">
					<button type="submit" class="button button-secondary">
						<?php esc_html_e( 'Clear All Caches', 'wpvulnerability' ); ?>
					</button>
				</form>

				<form method="post" style="display: inline-block; margin-right: 10px;" onsubmit="return confirm('<?php echo esc_js( __( 'Are you sure you want to reset signatures?', 'wpvulnerability' ) ); ?>');">
					<?php wp_nonce_field( 'wpvulnerability_debug_action', 'wpvulnerability_debug_nonce' ); ?>
					<input type="hidden" name="wpvulnerability_debug_reset_signatures" value="1">
					<button type="submit" class="button button-secondary">
						<?php esc_html_e( 'Reset Signatures', 'wpvulnerability' ); ?>
					</button>
				</form>

				<form method="post" style="display: inline-block;">
					<?php wp_nonce_field( 'wpvulnerability_debug_action', 'wpvulnerability_debug_nonce' ); ?>
					<input type="hidden" name="wpvulnerability_debug_export" value="1">
					<button type="submit" class="button button-primary">
						<?php esc_html_e( 'Export Debug Info', 'wpvulnerability' ); ?>
					</button>
				</form>
			</div>
		<?php else : ?>
			<p style="color: #d63638;">
				<?php esc_html_e( 'You do not have permission to perform these actions.', 'wpvulnerability' ); ?>
			</p>
		<?php endif; ?>
	</div>
	<?php
}
