<?php
/**
 * WP-CLI commands for WPVulnerability.
 *
 * Provides WP-CLI commands to inspect WordPress core, plugins,
 * and themes for known vulnerabilities.
 *
 * @package WPVulnerability
 */

defined( 'ABSPATH' ) || exit;

if ( defined( 'WP_CLI' ) && WP_CLI ) {
	/**
	 * Basic WPVulnerability commands.
	 */
	class WPVulnerability_CLI extends WP_CLI_Command {
		/**
		 * Display core vulnerabilities.
		 *
		 * ## OPTIONS
		 *
		 * [--format=<format>]
		 * : Render output in a specific format. Accepted values: table, json. Default: table.
		 *
		 * ## EXAMPLES
		 *
		 *     wp wpvulnerability core
		 *     wp wpvulnerability core --format=json
		 *
		 * @when after_wp_load
		 *
		 * @param array $args       Positional arguments.
		 * @param array $assoc_args Associative arguments.
		 */
		public function core( $args, $assoc_args ) {
			$format = isset( $assoc_args['format'] ) ? $assoc_args['format'] : 'table';

			$core_vulnerabilities = array();
			if ( function_exists( 'wpvulnerability_analyze_filter' ) && wpvulnerability_analyze_filter( 'core' ) ) {
				$core_vulnerabilities = wpvulnerability_core_get_vulnerabilities();
			}

			$items = array();
			if ( $core_vulnerabilities && is_array( $core_vulnerabilities ) ) {
				foreach ( $core_vulnerabilities as $vulnerability ) {
					$item             = array();
					$item['version']  = trim( html_entity_decode( wp_kses( (string) $vulnerability['name'], 'strip' ) ) );
					$item['severity'] = null;
					if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
						$item['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
					}

					$item['cwe'] = array();
					if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {
						foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
							$item['cwe'][] = array(
								'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) ) ),
								'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['description'], 'strip' ) ) ),
							);
						}
					}

					$item['score'] = null;
					if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
						$item['score'] = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
					}

					$item['source'] = array();
					if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {
						foreach ( $vulnerability['source'] as $vulnerability_source ) {
							$item['source'][] = array(
								'name' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['name'], 'strip' ) ) ),
								'link' => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
							);
						}
					}

					$items[] = $item;
				}
			}

			if ( 'json' === $format ) {
				echo wp_json_encode( $items );
				return;
			}

			$vulnerabilities = array();
			foreach ( $items as $c_vuln ) {
				$v_description_array = array();
				foreach ( $c_vuln['cwe'] as $c_cwe ) {
					if ( isset( $c_cwe['name'] ) ) {
						$v_description_array[] = $c_cwe['name'];
					}
				}
				$vulnerabilities[] = array(
					'version'     => $c_vuln['version'],
					'score'       => $c_vuln['score'],
					'severity'    => $c_vuln['severity'],
					'description' => trim( implode( ' + ', $v_description_array ) ),
				);
			}

			\WP_CLI\Utils\format_items( 'table', $vulnerabilities, array( 'version', 'score', 'severity', 'description' ) );
		}

		/**
		 * Display plugin vulnerabilities.
		 *
		 * ## OPTIONS
		 *
		 * [--format=<format>]
		 * : Render output in a specific format. Accepted values: table, json. Default: table.
		 *
		 * ## EXAMPLES
		 *
		 *     wp wpvulnerability plugins
		 *     wp wpvulnerability plugins --format=json
		 *
		 * @when after_wp_load
		 *
		 * @param array $args       Positional arguments.
		 * @param array $assoc_args Associative arguments.
		 */
		public function plugins( $args, $assoc_args ) {
			$format = isset( $assoc_args['format'] ) ? $assoc_args['format'] : 'table';

			$plugin_vulnerabilities = array();
			if ( function_exists( 'wpvulnerability_analyze_filter' ) && wpvulnerability_analyze_filter( 'plugins' ) ) {
				$plugin_vulnerabilities = wpvulnerability_plugin_get_vulnerabilities();
			}

			$items = array();
			foreach ( $plugin_vulnerabilities as $plugin ) {
				if ( 1 === $plugin['vulnerable'] ) {
					$plugin_temp         = array();
					$plugin_temp['name'] = trim( html_entity_decode( wp_kses( (string) $plugin['Name'], 'strip' ) ) );
					$plugin_temp['slug'] = trim( html_entity_decode( wp_kses( (string) $plugin['slug'], 'strip' ) ) );
					foreach ( $plugin['vulnerabilities'] as $vulnerability ) {
						$vul             = array();
						$vul['severity'] = null;
						if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
							$vul['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
						}
						$vul['version'] = trim( html_entity_decode( wp_kses( (string) $vulnerability['versions'], 'strip' ) ) );
						$vul['name']    = trim( html_entity_decode( wp_kses( (string) $vulnerability['name'], 'strip' ) ) );
						$vul['closed']  = (int) $vulnerability['closed'];
						$vul['unfixed'] = (int) $vulnerability['unfixed'];

						$vul['cwe'] = array();
						if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {
							foreach ( $vulnerability['impact']['cwe'] as $cwe ) {
								$vul['cwe'][] = array(
									'name'        => trim( html_entity_decode( wp_kses( (string) $cwe['name'], 'strip' ) ) ),
									'description' => trim( html_entity_decode( wp_kses( (string) $cwe['description'], 'strip' ) ) ),
								);
							}
						}

						$vul['score'] = null;
						if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
							$vul['score'] = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
						}

						$vul['source'] = array();
						if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {
							foreach ( $vulnerability['source'] as $source ) {
								$vul['source'][] = array(
									'name' => trim( html_entity_decode( wp_kses( (string) $source['name'], 'strip' ) ) ),
									'link' => esc_url_raw( (string) $source['link'], 'strip' ),
								);
							}
						}

						$plugin_temp['vulnerabilities'][] = $vul;
					}
					$items[] = $plugin_temp;
				}
			}

			if ( 'json' === $format ) {
				echo wp_json_encode( $items );
				return;
			}

			$vulnerabilities = array();
			foreach ( $items as $p_vuln ) {
				$v_name = $p_vuln['slug'];
				foreach ( $p_vuln['vulnerabilities'] as $p_vul ) {
					$v_fixed             = $p_vul['unfixed'] ? 'no' : 'yes';
					$v_closed            = $p_vul['closed'] ? 'yes' : 'no';
					$v_description_array = array();
					foreach ( $p_vul['cwe'] as $p_cwe ) {
						if ( isset( $p_cwe['name'] ) ) {
							$v_description_array[] = $p_cwe['name'];
						}
					}
					$vulnerabilities[] = array(
						'name'        => $v_name,
						'version'     => $p_vul['version'],
						'fixed'       => $v_fixed,
						'closed'      => $v_closed,
						'score'       => $p_vul['score'],
						'severity'    => $p_vul['severity'],
						'description' => trim( implode( ' + ', $v_description_array ) ),
					);
				}
			}

			\WP_CLI\Utils\format_items( 'table', $vulnerabilities, array( 'name', 'version', 'fixed', 'closed', 'score', 'severity', 'description' ) );
		}

		/**
		 * Display theme vulnerabilities.
		 *
		 * ## OPTIONS
		 *
		 * [--format=<format>]
		 * : Render output in a specific format. Accepted values: table, json. Default: table.
		 *
		 * ## EXAMPLES
		 *
		 *     wp wpvulnerability themes
		 *     wp wpvulnerability themes --format=json
		 *
		 * @when after_wp_load
		 *
		 * @param array $args       Positional arguments.
		 * @param array $assoc_args Associative arguments.
		 */
		public function themes( $args, $assoc_args ) {
			$format = isset( $assoc_args['format'] ) ? $assoc_args['format'] : 'table';

			$theme_vulnerabilities = array();
			if ( function_exists( 'wpvulnerability_analyze_filter' ) && wpvulnerability_analyze_filter( 'themes' ) ) {
				$theme_vulnerabilities = wpvulnerability_theme_get_vulnerabilities();
			}

			$items = array();
			foreach ( $theme_vulnerabilities as $theme ) {
				if ( isset( $theme['wpvulnerability']['vulnerabilities'] ) && count( $theme['wpvulnerability']['vulnerabilities'] ) ) {
					$theme_temp         = array();
					$theme_temp['name'] = trim( html_entity_decode( wp_kses( (string) $theme['wpvulnerability']['name'], 'strip' ) ) );
					$theme_temp['slug'] = trim( html_entity_decode( wp_kses( (string) $theme['wpvulnerability']['slug'], 'strip' ) ) );

					foreach ( $theme['wpvulnerability']['vulnerabilities'] as $vulnerability ) {
						$vul             = array();
						$vul['severity'] = null;
						if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
							$vul['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
						}
						$vul['version'] = trim( html_entity_decode( wp_kses( (string) $vulnerability['versions'], 'strip' ) ) );
						$vul['name']    = trim( html_entity_decode( wp_kses( (string) $vulnerability['name'], 'strip' ) ) );
						$vul['closed']  = (int) $vulnerability['closed'];
						$vul['unfixed'] = (int) $vulnerability['unfixed'];

						$vul['cwe'] = array();
						if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {
							foreach ( $vulnerability['impact']['cwe'] as $cwe ) {
								$vul['cwe'][] = array(
									'name'        => trim( html_entity_decode( wp_kses( (string) $cwe['name'], 'strip' ) ) ),
									'description' => trim( html_entity_decode( wp_kses( (string) $cwe['description'], 'strip' ) ) ),
								);
							}
						}

						$vul['score'] = null;
						if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
							$vul['score'] = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
						}

						$vul['source'] = array();
						if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {
							foreach ( $vulnerability['source'] as $source ) {
								$vul['source'][] = array(
									'name' => trim( html_entity_decode( wp_kses( (string) $source['name'], 'strip' ) ) ),
									'link' => esc_url_raw( (string) $source['link'], 'strip' ),
								);
							}
						}

						$theme_temp['vulnerabilities'][] = $vul;
					}
					$items[] = $theme_temp;
				}
			}

			if ( 'json' === $format ) {
				echo wp_json_encode( $items );
				return;
			}

			$vulnerabilities = array();
			foreach ( $items as $t_vuln ) {
				$v_name = $t_vuln['slug'];
				foreach ( $t_vuln['vulnerabilities'] as $t_vul ) {
					$v_fixed             = $t_vul['unfixed'] ? 'no' : 'yes';
					$v_closed            = $t_vul['closed'] ? 'yes' : 'no';
					$v_description_array = array();
					foreach ( $t_vul['cwe'] as $t_cwe ) {
						if ( isset( $t_cwe['name'] ) ) {
							$v_description_array[] = $t_cwe['name'];
						}
					}
					$vulnerabilities[] = array(
						'name'        => $v_name,
						'version'     => $t_vul['version'],
						'fixed'       => $v_fixed,
						'closed'      => $v_closed,
						'score'       => $t_vul['score'],
						'severity'    => $t_vul['severity'],
						'description' => trim( implode( ' + ', $v_description_array ) ),
					);
				}
			}

			\WP_CLI\Utils\format_items( 'table', $vulnerabilities, array( 'name', 'version', 'fixed', 'closed', 'score', 'severity', 'description' ) );
		}

		/**
		 * Display vulnerabilities for a piece of server software.
		 *
		 * @param string $software   Software slug.
		 * @param array  $args       Positional arguments.
		 * @param array  $assoc_args Associative arguments.
		 */
		private function software( $software, $args, $assoc_args ) {
			$format = isset( $assoc_args['format'] ) ? $assoc_args['format'] : 'table';

			$software_vulnerabilities = array();
			switch ( $software ) {
				case 'php':
				case 'apache':
				case 'nginx':
				case 'mariadb':
				case 'mysql':
				case 'imagemagick':
				case 'curl':
				case 'memcached':
				case 'redis':
				case 'sqlite':
					$software_data = wpvulnerability_software_get_vulnerabilities( $software );
					if ( is_array( $software_data ) && isset( $software_data['vulnerabilities'] ) && is_array( $software_data['vulnerabilities'] ) ) {
						$software_vulnerabilities = $software_data['vulnerabilities'];
					}
					break;
				default:
					$software_vulnerabilities = array();
			}

			$items = array();
			if ( isset( $software_vulnerabilities ) && is_array( $software_vulnerabilities ) ) {
				foreach ( $software_vulnerabilities as $item ) {
					$complete_temp             = array();
					$complete_temp['version']  = trim( html_entity_decode( wp_kses( (string) $item['version'], 'strip' ) ) );
					$complete_temp['affected'] = trim( html_entity_decode( wp_kses( (string) $item['versions'], 'strip' ) ) );
					$complete_temp['unfixed']  = (int) $item['unfixed'];
					$complete_temp['source']   = array();
					if ( isset( $item['source'] ) && count( $item['source'] ) ) {
						foreach ( $item['source'] as $source ) {
							$complete_temp['source'][] = array(
								'name'        => trim( html_entity_decode( wp_kses( (string) $source['id'], 'strip' ) ) ),
								'description' => trim( html_entity_decode( wp_kses( (string) $source['description'], 'strip' ) ) ),
								'link'        => esc_url_raw( (string) $source['link'], 'strip' ),
							);
						}
					}
					$items[] = $complete_temp;
				}
			}

			if ( 'json' === $format ) {
				echo wp_json_encode( $items );
				return;
			}

			$vulnerabilities = array();
			foreach ( $items as $vuln ) {
				$v_fixed       = $vuln['unfixed'] ? 'no' : 'yes';
				$v_description = array();
				foreach ( $vuln['source'] as $source ) {
					$v_description[] = $source['name'] . ': ' . $source['description'];
				}
				$vulnerabilities[] = array(
					'version'     => $vuln['version'],
					'affected'    => $vuln['affected'],
					'fixed'       => $v_fixed,
					'description' => trim( implode( ' + ', $v_description ) ),
				);
			}

			\WP_CLI\Utils\format_items( 'table', $vulnerabilities, array( 'version', 'affected', 'fixed', 'description' ) );
		}

		/**
		 * Display PHP vulnerabilities.
		 *
		 * ## OPTIONS
		 *
		 * [--format=<format>]
		 * : Render output in a specific format. Accepted values: table, json. Default: table.
		 *
		 * ## EXAMPLES
		 *
		 *     wp wpvulnerability php
		 *     wp wpvulnerability php --format=json
		 *
		 * @when after_wp_load
		 *
		 * @param array $args       Positional arguments.
		 * @param array $assoc_args Associative arguments.
		 */
		public function php( $args, $assoc_args ) {
			$this->software( 'php', $args, $assoc_args );
		}

		/**
		 * Display Apache vulnerabilities.
		 *
		 * ## OPTIONS
		 *
		 * [--format=<format>]
		 * : Render output in a specific format. Accepted values: table, json. Default: table.
		 *
		 * ## EXAMPLES
		 *
		 *     wp wpvulnerability apache
		 *     wp wpvulnerability apache --format=json
		 *
		 * @when after_wp_load
		 *
		 * @param array $args       Positional arguments.
		 * @param array $assoc_args Associative arguments.
		 */
		public function apache( $args, $assoc_args ) {
			$this->software( 'apache', $args, $assoc_args );
		}

		/**
		 * Display Nginx vulnerabilities.
		 *
		 * ## OPTIONS
		 *
		 * [--format=<format>]
		 * : Render output in a specific format. Accepted values: table, json. Default: table.
		 *
		 * ## EXAMPLES
		 *
		 *     wp wpvulnerability nginx
		 *     wp wpvulnerability nginx --format=json
		 *
		 * @when after_wp_load
		 *
		 * @param array $args       Positional arguments.
		 * @param array $assoc_args Associative arguments.
		 */
		public function nginx( $args, $assoc_args ) {
			$this->software( 'nginx', $args, $assoc_args );
		}

		/**
		 * Display MariaDB vulnerabilities.
		 *
		 * ## OPTIONS
		 *
		 * [--format=<format>]
		 * : Render output in a specific format. Accepted values: table, json. Default: table.
		 *
		 * ## EXAMPLES
		 *
		 *     wp wpvulnerability mariadb
		 *     wp wpvulnerability mariadb --format=json
		 *
		 * @when after_wp_load
		 *
		 * @param array $args       Positional arguments.
		 * @param array $assoc_args Associative arguments.
		 */
		public function mariadb( $args, $assoc_args ) {
			$this->software( 'mariadb', $args, $assoc_args );
		}

		/**
		 * Display MySQL vulnerabilities.
		 *
		 * ## OPTIONS
		 *
		 * [--format=<format>]
		 * : Render output in a specific format. Accepted values: table, json. Default: table.
		 *
		 * ## EXAMPLES
		 *
		 *     wp wpvulnerability mysql
		 *     wp wpvulnerability mysql --format=json
		 *
		 * @when after_wp_load
		 *
		 * @param array $args       Positional arguments.
		 * @param array $assoc_args Associative arguments.
		 */
		public function mysql( $args, $assoc_args ) {
			$this->software( 'mysql', $args, $assoc_args );
		}

		/**
		 * Display ImageMagick vulnerabilities.
		 *
		 * ## OPTIONS
		 *
		 * [--format=<format>]
		 * : Render output in a specific format. Accepted values: table, json. Default: table.
		 *
		 * ## EXAMPLES
		 *
		 *     wp wpvulnerability imagemagick
		 *     wp wpvulnerability imagemagick --format=json
		 *
		 * @when after_wp_load
		 *
		 * @param array $args       Positional arguments.
		 * @param array $assoc_args Associative arguments.
		 */
		public function imagemagick( $args, $assoc_args ) {
			$this->software( 'imagemagick', $args, $assoc_args );
		}

		/**
		 * Display cURL vulnerabilities.
		 *
		 * ## OPTIONS
		 *
		 * [--format=<format>]
		 * : Render output in a specific format. Accepted values: table, json. Default: table.
		 *
		 * ## EXAMPLES
		 *
		 *     wp wpvulnerability curl
		 *     wp wpvulnerability curl --format=json
		 *
		 * @when after_wp_load
		 *
		 * @param array $args       Positional arguments.
		 * @param array $assoc_args Associative arguments.
		 */
		public function curl( $args, $assoc_args ) {
			$this->software( 'curl', $args, $assoc_args );
		}

		/**
		 * Display Memcached vulnerabilities.
		 *
		 * ## OPTIONS
		 *
		 * [--format=<format>]
		 * : Render output in a specific format. Accepted values: table, json. Default: table.
		 *
		 * ## EXAMPLES
		 *
		 *     wp wpvulnerability memcached
		 *     wp wpvulnerability memcached --format=json
		 *
		 * @when after_wp_load
		 *
		 * @param array $args       Positional arguments.
		 * @param array $assoc_args Associative arguments.
		 */
		public function memcached( $args, $assoc_args ) {
			$this->software( 'memcached', $args, $assoc_args );
		}

		/**
		 * Display Redis vulnerabilities.
		 *
		 * ## OPTIONS
		 *
		 * [--format=<format>]
		 * : Render output in a specific format. Accepted values: table, json. Default: table.
		 *
		 * ## EXAMPLES
		 *
		 *     wp wpvulnerability redis
		 *     wp wpvulnerability redis --format=json
		 *
		 * @when after_wp_load
		 *
		 * @param array $args       Positional arguments.
		 * @param array $assoc_args Associative arguments.
		 */
		public function redis( $args, $assoc_args ) {
			$this->software( 'redis', $args, $assoc_args );
		}

		/**
		 * Display SQLite vulnerabilities.
		 *
		 * ## OPTIONS
		 *
		 * [--format=<format>]
		 * : Render output in a specific format. Accepted values: table, json. Default: table.
		 *
		 * ## EXAMPLES
		 *
		 *     wp wpvulnerability sqlite
		 *     wp wpvulnerability sqlite --format=json
		 *
		 * @when after_wp_load
		 *
		 * @param array $args       Positional arguments.
		 * @param array $assoc_args Associative arguments.
		 */
		public function sqlite( $args, $assoc_args ) {
			$this->software( 'sqlite', $args, $assoc_args );
		}
	}

	WP_CLI::add_command( 'wpvulnerability', 'WPVulnerability_CLI' );
}
