<?php
/**
 * Software functions
 *
 * @package WPVulnerability
 *
 * @version 3.5.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Retrieves the specified software version.
 *
 * This function returns the version of PHP, Apache, Nginx, MySQL, or MariaDB
 * after performing necessary validations. It ensures that the returned value
 * is clean and sanitized.
 *
 * @since 3.5.0
 *
 * @param string $software The name of the software ('php', 'apache', 'nginx', 'mysql', 'mariadb').
 * @return string|null The sanitized version of the software, or null if not found.
 */
function wpvulnerability_get_software_version( $software ) {
	switch ( $software ) {
		case 'php':
			$php_version = wpvulnerability_detect_php();
			if ( null !== $php_version && '' !== $php_version ) {
				return wp_kses( (string) $php_version, 'strip' );
			}

			break;

		case 'apache':
		case 'nginx':
			$webserver = wpvulnerability_detect_webserver();
			if ( isset( $webserver['id'] ) && $webserver['id'] === $software && ! empty( $webserver['version'] ) ) {
				return wp_kses( (string) $webserver['version'], 'strip' );
			}
			break;

		case 'mysql':
		case 'mariadb':
			$sqlserver = wpvulnerability_detect_sqlserver();
			if ( isset( $sqlserver['id'] ) && $sqlserver['id'] === $software && ! empty( $sqlserver['version'] ) ) {
				return wp_kses( (string) $sqlserver['version'], 'strip' );
			}
			break;

		case 'imagemagick':
			$imagemagick_version = wpvulnerability_detect_imagemagick();
			if ( null !== $imagemagick_version && '' !== $imagemagick_version ) {
				return wp_kses( (string) $imagemagick_version, 'strip' );
			}

			break;

		case 'curl':
			$curl_version = wpvulnerability_detect_curl();
			if ( null !== $curl_version && '' !== $curl_version ) {
				return wp_kses( (string) $curl_version, 'strip' );
			}

			break;

		case 'memcached':
			$memcached_version = wpvulnerability_detect_memcached();
			if ( null !== $memcached_version && '' !== $memcached_version ) {
				return wp_kses( (string) $memcached_version, 'strip' );
			}

			break;

		case 'redis':
			$redis_version = wpvulnerability_detect_redis();
			if ( null !== $redis_version && '' !== $redis_version ) {
				return wp_kses( (string) $redis_version, 'strip' );
			}

			break;

		case 'sqlite':
			$sqlite_version = wpvulnerability_detect_sqlite();
			if ( null !== $sqlite_version && '' !== $sqlite_version ) {
				return wp_kses( (string) $sqlite_version, 'strip' );
			}

			break;

		default:
			break;
	}

	return null;
}

/**
 * Retrieves vulnerabilities for a given software version and updates its data.
 *
 * This function detects the installed software version, checks for vulnerabilities using an external API,
 * and updates the data array with the vulnerabilities found.
 *
 * @since 3.5.0
 *
 * @param string $software The software name (e.g., 'php', 'apache', 'nginx', 'mysql', 'mariadb').
 *
 * @return array The updated data array containing vulnerability information.
 */
function wpvulnerability_get_fresh_vulnerabilities( $software ) {

	$version = null;
	$data    = array(
		'vulnerabilities' => null,
		'vulnerable'      => 0,
	);

	switch ( $software ) {
		case 'php':
		case 'apache':
		case 'nginx':
		case 'mysql':
		case 'mariadb':
		case 'imagemagick':
		case 'curl':
		case 'memcached':
		case 'redis':
		case 'sqlite':
			$version = wpvulnerability_get_software_version( $software );
			break;

		default:
			return $data;
	}

	if ( $version ) {
		switch ( $software ) {
			case 'php':
			case 'apache':
			case 'nginx':
			case 'mysql':
			case 'mariadb':
			case 'imagemagick':
			case 'curl':
			case 'memcached':
			case 'redis':
			case 'sqlite':
				$api_response = wpvulnerability_get_vulnerabilities( $software, $version, 0 );
				break;
		}

		if ( ! empty( $api_response ) ) {
			$data['vulnerabilities'] = $api_response;
			$data['vulnerable']      = 1;
		}
	}

	return $data;
}


/**
 * Get Installed Software
 *
 * Retrieves the list of installed software versions, checks for vulnerabilities,
 * caches the data, and sends an email notification if vulnerabilities are detected.
 *
 * @since 3.5.0
 *
 * @param string $software The software name (e.g., 'php', 'apache').
 *
 * @return string JSON-encoded array of software data with vulnerabilities and vulnerable status.
 */
function wpvulnerability_get_installed( $software ) {

	$wpvulnerability_software_vulnerable = 0;

	// Retrieve fresh vulnerabilities for the installed software version.
	$data = wpvulnerability_get_fresh_vulnerabilities( $software );

	// Check if the software version is vulnerable and count the vulnerabilities.
	if ( isset( $data['vulnerable'] ) && (int) $data['vulnerable'] ) {
		$wpvulnerability_software_vulnerable = count( $data['vulnerabilities'] );
	}

	// Cache the vulnerability data and the timestamp for cache expiration.
	if ( is_multisite() ) {
				update_site_option( 'wpvulnerability-' . $software, wp_json_encode( $data ) );
				update_site_option( 'wpvulnerability-' . $software . '-vulnerable', wp_json_encode( number_format( $wpvulnerability_software_vulnerable, 0, '.', '' ) ) );
				update_site_option( 'wpvulnerability-' . $software . '-cache', wp_json_encode( number_format( time() + ( 3600 * wpvulnerability_cache_hours() ), 0, '.', '' ) ) );
	} else {
				update_option( 'wpvulnerability-' . $software, wp_json_encode( $data ) );
				update_option( 'wpvulnerability-' . $software . '-vulnerable', wp_json_encode( number_format( $wpvulnerability_software_vulnerable, 0, '.', '' ) ) );
				update_option( 'wpvulnerability-' . $software . '-cache', wp_json_encode( number_format( time() + ( 3600 * wpvulnerability_cache_hours() ), 0, '.', '' ) ) );
	}

	// Return the JSON-encoded array of software data.
	return wp_json_encode( $data );
}

/**
 * Get the cached vulnerabilities or update the cache if it's stale or missing.
 *
 * @since 3.5.0
 *
 * @param string $software The software name (e.g., 'php', 'apache').
 *
 * @return array|null Array of software data with vulnerabilities, or null if software is invalid.
 */
function wpvulnerability_software_get_vulnerabilities( $software ) {

	$valid_software = array( 'php', 'apache', 'mariadb', 'mysql', 'nginx', 'imagemagick', 'curl', 'memcached', 'redis', 'sqlite' );

	// Use strict comparison for in_array.
	if ( in_array( $software, $valid_software, true ) ) {
		if ( is_multisite() ) {

			// Get the cached data and decode it.
			$data_cache = json_decode( get_site_option( 'wpvulnerability-' . $software . '-cache' ) );

			// Get the installed data and decode it.
			$data = json_decode( get_site_option( 'wpvulnerability-' . $software ), true );

		} else {

			// Get the cached data and decode it.
			$data_cache = json_decode( get_option( 'wpvulnerability-' . $software . '-cache' ) );

			// Get the installed data and decode it.
			$data = json_decode( get_option( 'wpvulnerability-' . $software ), true );

		}

		// If the cache is stale or the data is empty, update the cache.
		if ( $data_cache < time() || empty( $data ) ) {

			// Get the installed data and update the cache.
			$data = json_decode( wpvulnerability_get_installed( $software ), true );

		}

		return $data;
	} else {
		return null;
	}
}

/**
 * Update the software cache and remove any old cache data.
 *
 * @since 3.0.0
 *
 * @param string $software The software name (e.g., 'php', 'apache').
 *
 * @return void
 */
function wpvulnerability_get_vulnerabilities_clean( $software ) {

	// Update the installed software cache.
	wpvulnerability_get_installed( $software );
}
