<?php
/**
 * Scheduling functions
 *
 * @package WPVulnerability
 *
 * @version 3.1.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Schedule Automatic Vulnerability Database Update.
 * If the 'wpvulnerability_update_database' event is not already scheduled, schedule it to run twice daily.
 *
 * @since 2.0.0
 *
 * @return void
 */
if ( ! wp_next_scheduled( 'wpvulnerability_update_database' ) ) {
	// Schedule the event only for the main site in a multisite network or for a single site.
	if ( ! is_multisite() || ( is_multisite() && is_main_site() ) ) {
		wp_schedule_event( time(), 'twicedaily', 'wpvulnerability_update_database' );
	}
}

// Hook the event to the function that updates the database.
add_action( 'wpvulnerability_update_database', 'wpvulnerability_update_database_data' );

if ( ! wp_next_scheduled( 'wpvulnerability_cleanup_logs' ) ) {
	if ( ! is_multisite() || ( is_multisite() && is_main_site() ) ) {
		wp_schedule_event( time(), 'daily', 'wpvulnerability_cleanup_logs' );
	}
}

/**
 * Calculate the next notification timestamp based on plugin settings.
 *
 * @since 4.1.1
 *
 * @param array $config Plugin configuration.
 * @return int Timestamp for next notification.
 */
function wpvulnerability_get_next_notification_timestamp( $config ) {
	$hour   = isset( $config['hour'] ) ? max( 0, min( 23, (int) $config['hour'] ) ) : 0;
	$minute = isset( $config['minute'] ) ? max( 0, min( 59, (int) $config['minute'] ) ) : 0;

	if ( function_exists( 'wp_timezone' ) ) {
		$timezone = wp_timezone();
	} else {
		$timezone_string = get_option( 'timezone_string' );
		if ( $timezone_string ) {
			$timezone = new DateTimeZone( $timezone_string );
		} else {
			$offset    = (float) get_option( 'gmt_offset' );
			$hours     = (int) $offset;
			$minutes   = $offset - $hours;
			$sign      = ( $offset < 0 ) ? '-' : '+';
			$abs_hour  = abs( $hours );
			$abs_mins  = (int) round( abs( $minutes ) * 60 );
			$tz_offset = sprintf( '%s%02d:%02d', $sign, $abs_hour, $abs_mins );
			$timezone  = new DateTimeZone( $tz_offset );
		}
	}

	$current_time   = new DateTime( 'now', $timezone );
	$scheduled_time = new DateTime( 'now', $timezone );
	$scheduled_time->setTime( $hour, $minute, 0 );

	if ( isset( $config['period'] ) && 'weekly' === $config['period'] ) {
		$day       = isset( $config['day'] ) ? strtolower( (string) $config['day'] ) : 'monday';
		$weekdays  = array( 'sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday' );
		$day_index = array_search( $day, $weekdays, true );
		if ( false === $day_index ) {
			$day_index = 1; // Monday.
		}
		while ( (int) $scheduled_time->format( 'w' ) !== $day_index || $scheduled_time->getTimestamp() <= $current_time->getTimestamp() ) {
			$scheduled_time->modify( '+1 day' );
		}
	} elseif ( $scheduled_time->getTimestamp() <= $current_time->getTimestamp() ) {
		$scheduled_time->modify( '+1 day' );
	}

	return (int) $scheduled_time->getTimestamp();
}

/**
 * Schedule vulnerability notifications.
 *
 * @since 4.1.1
 *
 * @param array $config Plugin configuration.
 *
 * @return void
 */
function wpvulnerability_schedule_notification_event( $config ) {
		wp_clear_scheduled_hook( 'wpvulnerability_notification' );
	if ( ! isset( $config['period'] ) || 'never' === $config['period'] ) {
			return;
	}

	if ( ! is_multisite() || ( is_multisite() && is_main_site() ) ) {
			$timestamp = wpvulnerability_get_next_notification_timestamp( $config );
			wp_schedule_event( $timestamp, $config['period'], 'wpvulnerability_notification' );
	}
}

$wpvulnerability_s = is_multisite() ? get_site_option( 'wpvulnerability-config' ) : get_option( 'wpvulnerability-config' );
wpvulnerability_schedule_notification_event( $wpvulnerability_s );
add_action( 'wpvulnerability_notification', 'wpvulnerability_execute_notification' );
unset( $wpvulnerability_s );
