<?php
/**
 * Process functions
 *
 * @package WPVulnerability
 *
 * @since 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Convert vulnerabilities into pretty HTML
 *
 * @since 2.0.0
 *
 * @param string $type Type: core, plugin, theme, php, apache, nginx, mariadb, mysql, imagemagick, curl.
 * @param array  $vulnerabilities Vulnerability data.
 *
 * @return string The HTML representation of vulnerabilities.
 */
function wpvulnerability_html( $type, $vulnerabilities ) {
	$html = '';

	if ( in_array( $type, array( 'plugin', 'theme' ), true ) ) {
		foreach ( $vulnerabilities as $vulnerability ) {
			$what = array();
			if ( isset( $vulnerability['impact']['cwe'] ) ) {
				foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
					$what[] = '<div><b>' . wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) . '</b></div><div><i>' . wp_kses_post( (string) $vulnerability_cwe['description'] ) . '</i></div>';
				}
			}

			$sources = array();
			if ( isset( $vulnerability['source'] ) ) {
				foreach ( $vulnerability['source'] as $vulnerability_source ) {
					$sources[] = '<a href="' . esc_url( (string) $vulnerability_source['link'], 'strip' ) . '" target="_blank">[+]</a>&nbsp;' . wp_kses( (string) $vulnerability_source['name'], 'strip' );
				}
			}

			$source = count( $sources ) ? '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>' : '';

			$score    = isset( $vulnerability['impact']['cvss']['score'] ) ? number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' ) : null;
			$severity = isset( $vulnerability['impact']['cvss']['severity'] ) ? wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] ) : null;

			$html .= '<h4>' . wp_kses( (string) $vulnerability['name'], 'strip' ) . '</h4>';
			if ( (int) $vulnerability['closed'] || (int) $vulnerability['unfixed'] ) {
				$html .= '<div style="padding-bottom: 5px;">';
				if ( (int) $vulnerability['closed'] ) {
					$html .= '<div class="text-red">' . __( 'This plugin is closed. Please replace it with another.', 'wpvulnerability' ) . '</div>';
				}
				if ( (int) $vulnerability['unfixed'] ) {
					$html .= '<div class="text-red">' . __( 'This vulnerability appears to be unpatched. Stay tuned for upcoming plugin updates.', 'wpvulnerability' ) . '</div>';
				}
				$html .= '</div>';
			}

			if ( count( $what ) ) {
				$html .= '<div style="padding-bottom: 5px;">' . implode( '', $what ) . '</div>';
			}

			if ( ! is_null( $score ) || ! is_null( $severity ) ) {
				$html .= '<div style="padding-bottom: 5px;">';
				if ( ! is_null( $score ) ) {
					$html .= '<div>' . __( 'Global score: ', 'wpvulnerability' ) . $score . ' / 10</div>';
				}
				if ( ! is_null( $severity ) ) {
					$html .= '<div>' . __( 'Severity: ', 'wpvulnerability' ) . $severity . '</div>';
				}
				$html .= '</div>';
			}

			$html .= wp_kses( (string) $source, 'post' );
		}
	} elseif ( 'core' === $type ) {
		foreach ( $vulnerabilities as $vulnerability ) {
			$what = array();
			foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
				$what[] = '<div><b>' . wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) . '</b></div><div><i>' . wp_kses_post( (string) $vulnerability_cwe['description'] ) . '</i></div>';
			}

			$sources = array();
			foreach ( $vulnerability['source'] as $vulnerability_source ) {
				$sources[] = '<a href="' . esc_url( (string) $vulnerability_source['link'], 'strip' ) . '" target="_blank">[+]</a>&nbsp;' . wp_kses( (string) $vulnerability_source['name'], 'strip' );
			}
			$source = '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>';

			$score    = isset( $vulnerability['impact']['cvss']['score'] ) ? number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' ) : null;
			$severity = isset( $vulnerability['impact']['cvss']['severity'] ) ? wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] ) : null;

			$html .= '<h3> WordPress ' . wp_kses( (string) $vulnerability['name'], 'strip' ) . '</h3>';
			if ( count( $what ) ) {
				$html .= '<div style="padding-bottom: 5px;">' . implode( '', $what ) . '</div>';
			}

			if ( ! is_null( $score ) || ! is_null( $severity ) ) {
				$html .= '<div style="padding-bottom: 5px;">';
				if ( ! is_null( $score ) ) {
					$html .= '<div>' . __( 'Global score: ', 'wpvulnerability' ) . $score . ' / 10</div>';
				}
				if ( ! is_null( $severity ) ) {
					$html .= '<div>' . __( 'Severity: ', 'wpvulnerability' ) . $severity . '</div>';
				}
				$html .= '</div>';
			}

			$html .= wp_kses( (string) $source, 'post' );
		}
	} elseif ( in_array( $type, array( 'php', 'apache', 'nginx', 'mariadb', 'mysql', 'imagemagick', 'curl', 'memcached', 'redis', 'sqlite' ), true ) ) {
		foreach ( $vulnerabilities as $vulnerability ) {
			$sources = array();
			foreach ( $vulnerability['source'] as $vulnerability_source ) {
				$sources[] = '<a href="' . esc_url( (string) $vulnerability_source['link'], 'strip' ) . '" target="_blank">[+]</a>&nbsp;' . wp_kses( (string) $vulnerability_source['id'], 'strip' ) . '<br>' . wp_kses( (string) $vulnerability_source['description'], 'strip' );
			}
			$source = '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>';

			$html .= '<h4> ' . wp_kses( (string) $vulnerability['name'], 'strip' ) . '</h4>';
			$html .= '<div style="padding-bottom: 5px;"></div>';
			$html .= wp_kses( (string) $source, 'post' );
		}
	}

	return $html;
}

/**
 * Convert vulnerabilities into HTML format.
 *
 * @since 3.5.0
 *
 * @param string $type Type of software (php, apache, nginx, mariadb, mysql, imagemagick, curl).
 * @return string|false The HTML output if vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_software( $type ) {
	$html              = '';
	$found             = false;
		$software_name = null;

	// Include the software vulnerabilities file.
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-software.php';

	// Map software types to their names.
	$software_names = array(
		'php'         => 'PHP',
		'apache'      => 'Apache HTTP',
		'nginx'       => 'Nginx',
		'mariadb'     => 'MariaDB',
		'mysql'       => 'MySQL',
		'imagemagick' => 'ImageMagick',
		'curl'        => 'curl',
		'memcached'   => 'memcached',
		'redis'       => 'redis',
		'sqlite'      => 'sqlite',
	);

	// Check if the type is valid and get the software name.
	if ( isset( $software_names[ $type ] ) ) {
		$software_name = $software_names[ $type ];
	} else {
		return false; // Invalid type.
	}

		$version     = wpvulnerability_sanitize_and_validate_version( wpvulnerability_get_software_version( $type ) );
	$vulnerabilities = wpvulnerability_get_vulnerabilities( $type, $version );

	// Check if vulnerabilities were found.
	if ( is_array( $vulnerabilities ) && 0 < count( $vulnerabilities ) ) {
		$found = true;

		// translators: the type of vulnerability.
		$html .= '<h3>' . sprintf( esc_html__( '%s running', 'wpvulnerability' ), esc_html( $software_name ) ) . ': ' . wp_kses( (string) $version, 'strip' ) . '</h3>';
		$html .= wpvulnerability_html( $type, $vulnerabilities );

	}

	return $found ? $html : false;
}

/**
 * Convert plugin vulnerabilities into HTML format.
 *
 * @since 2.0.0
 *
 * @return string|false The HTML output if plugin vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_plugins() {
	$html  = '';
	$found = false;

	$plugins = wpvulnerability_plugin_get_vulnerabilities();

	foreach ( $plugins as $file_path => $plugin_data ) {
		// Check if the plugin is marked as vulnerable.
		if ( isset( $plugin_data['vulnerable'] ) && 1 === $plugin_data['vulnerable'] ) {
			$found = true;

			// Generate HTML markup for the plugin vulnerability.
			$html .= '<h3>' . esc_html__( 'Plugin', 'wpvulnerability' ) . ': ' . wp_kses( (string) $plugin_data['Name'], 'strip' ) . '</h3>';
			$html .= wpvulnerability_html( 'plugin', $plugin_data['vulnerabilities'] );
		}
	}

	// Return the HTML if vulnerabilities were found.
	return $found ? $html : false;
}

/**
 * Convert plugin vulnerabilities into list format.
 *
 * @since 2.2.0
 *
 * @return string|false The HTML output if plugin vulnerabilities were found, false otherwise.
 */
function wpvulnerability_list_plugins() {
	$html  = '<ul class="inside">';
	$found = false;

	// Get vulnerabilities data for plugins.
	$plugins = wpvulnerability_plugin_get_vulnerabilities();

	// Iterate through each plugin's data.
	foreach ( $plugins as $file_path => $plugin_data ) {
		// Check if the plugin is marked as vulnerable.
		if ( isset( $plugin_data['vulnerable'] ) && 1 === $plugin_data['vulnerable'] ) {
			$found = true;

			// Generate HTML markup for the plugin vulnerability.
			$html .= '<li>' . wp_kses( (string) $plugin_data['Name'], 'strip' ) . '</li>';
		}
	}

	$html .= '</ul>';

	// Return the HTML if vulnerabilities were found.
	return $found ? $html : false;
}

/**
 * Convert theme vulnerabilities into HTML format.
 *
 * @since 2.0.0
 *
 * @return string|false The HTML output if theme vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_themes() {
	$html  = '';
	$found = false;

	// Get vulnerabilities data for themes.
	$themes = wpvulnerability_theme_get_vulnerabilities();

	// Iterate through each theme's data.
	foreach ( $themes as $theme_data ) {
		// Check if the theme is marked as vulnerable.
		if ( isset( $theme_data['wpvulnerability']['vulnerable'] ) && 1 === $theme_data['wpvulnerability']['vulnerable'] ) {
			$found = true;

			// Generate HTML markup for the theme vulnerability.
			$html .= '<h3>' . esc_html__( 'Theme', 'wpvulnerability' ) . ': ' . wp_kses( (string) $theme_data['wpvulnerability']['name'], 'strip' ) . '</h3>';
			$html .= wpvulnerability_html( 'theme', $theme_data['wpvulnerability']['vulnerabilities'] );
		}
	}

	// Return the HTML if vulnerabilities were found.
	return $found ? $html : false;
}

/**
 * Convert theme vulnerabilities into list format.
 *
 * @since 2.2.0
 *
 * @return string|false The HTML output if theme vulnerabilities were found, false otherwise.
 */
function wpvulnerability_list_themes() {
	$html  = '<ul class="inside">';
	$found = false;

	// Get vulnerabilities data for themes.
	$themes = wpvulnerability_theme_get_vulnerabilities();

	// Iterate through each theme's data.
	foreach ( $themes as $theme_data ) {
		// Check if the theme is marked as vulnerable.
		if ( isset( $theme_data['wpvulnerability']['vulnerable'] ) && 1 === $theme_data['wpvulnerability']['vulnerable'] ) {
			$found = true;

			// Generate HTML markup for the theme vulnerability.
			$html .= '<li>' . wp_kses( (string) $theme_data['wpvulnerability']['name'], 'strip' ) . '</li>';
		}
	}

	$html .= '</ul>';

	// Return the HTML if vulnerabilities were found.
	return $found ? $html : false;
}

/**
 * Convert core vulnerabilities into HTML format.
 *
 * @since 2.0.0
 *
 * @return string|false The HTML output if core vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_core() {
	$html  = '';
	$found = false;

	// Get vulnerabilities data for WordPress core.
	$core = wpvulnerability_core_get_vulnerabilities();

	// Check if there are any vulnerabilities.
	if ( is_array( $core ) && count( $core ) ) {
		$found = true;

		// Generate HTML markup for the core vulnerabilities.
		$html .= wpvulnerability_html( 'core', $core );
	}

	// Return the HTML if vulnerabilities were found.
	return $found ? $html : false;
}
