<?php
/**
 * General functions
 *
 * @package WPVulnerability
 *
 * @since 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Clear the existing cache for the specified type.
 *
 * @since 2.0.0
 *
 * @param string $type The type of cache to clear (core, plugins, themes).
 * @return void
 */
function wpvulnerability_clear_cache( $type ) {
	$additional_keys = array(
		'core'    => array( 'wpvulnerability-core-version' ),
		'plugins' => array(
			'wpvulnerability-plugins-signature',
			'wpvulnerability-plugins-data',
			'wpvulnerability-plugins-cache-data',
		),
		'themes'  => array( 'wpvulnerability-themes-signature' ),
	);

	if ( is_multisite() ) {
		delete_site_option( "wpvulnerability-{$type}" );
		delete_site_option( "wpvulnerability-{$type}-vulnerable" );
		delete_site_option( "wpvulnerability-{$type}-cache" );

		if ( isset( $additional_keys[ $type ] ) ) {
			foreach ( $additional_keys[ $type ] as $option_name ) {
				delete_site_option( $option_name );
			}
		}
	} else {
		delete_option( "wpvulnerability-{$type}" );
		delete_option( "wpvulnerability-{$type}-vulnerable" );
		delete_option( "wpvulnerability-{$type}-cache" );

		if ( isset( $additional_keys[ $type ] ) ) {
			foreach ( $additional_keys[ $type ] as $option_name ) {
				delete_option( $option_name );
			}
		}
	}
}

/**
 * Checks and validates user capabilities for managing vulnerability settings in a WordPress environment.
 *
 * This function verifies if the current user has the appropriate permissions to manage network settings
 * in a multisite installation or manage options in a single site installation. It ensures that only
 * Administrators in a single site and Super Administrators in multisite can access these settings.
 *
 * @since 3.0.0
 *
 * @return bool Returns true if the current user has the required capabilities, false otherwise.
 */
function wpvulnerability_capabilities() {
	// Check if the user is logged in.
	if ( ! is_user_logged_in() ) {
		return false;
	}

	// Check if in a Multisite environment.
	if ( is_multisite() && is_super_admin() && ( is_network_admin() || is_main_site() ) ) {
		return true;
	} elseif ( is_admin() && current_user_can( 'manage_options' ) ) {
			return true;
	}

	// Return false if the user does not have the required capabilities.
	return false;
}

/**
 * Checks if the `shell_exec` function can be used.
 *
 * This function verifies if the `shell_exec` function is not disabled in the server's
 * configuration and is able to execute a basic shell command. It also checks for
 * safe mode, which is relevant for older PHP versions before 5.4.
 *
 * @since 3.4.0
 *
 * @return bool True if `shell_exec` is available and working, false otherwise.
 */
function wpvulnerability_can_shell_exec() {
	// Check if `shell_exec` is disabled.
	if ( in_array( 'shell_exec', array_map( 'trim', explode( ',', (string) ini_get( 'disable_functions' ) ) ), true ) ) {
		return false; // `shell_exec` is disabled.
	}

	// Try to execute a simple command to confirm functionality.
	$test = shell_exec( escapeshellcmd( 'echo test' ) ); // phpcs:ignore

	// If the command execution failed or returned null, shell_exec is not working.
	return null !== $test; // Return true if the command was successful.
}

/**
 * Conditionally log diagnostic messages for the plugin.
 *
 * This helper respects the WordPress debug mode and allows developers to hook into the
 * decision using the {@see 'wpvulnerability_should_log'} filter. Logged messages are
 * encoded as JSON when possible to provide structured context without breaking the
 * WordPress Coding Standards that discourage verbose debugging in production.
 *
 * @since 4.1.7
 *
 * @param string $message  Message to record in the debug log.
 * @param array  $context  Optional. Additional context about the message. Default empty array.
 *
 * @return void
 */
function wpvulnerability_maybe_log( $message, $context = array() ) {
	if ( empty( $message ) ) {
		return;
	}

	$should_log = defined( 'WP_DEBUG' ) && WP_DEBUG;

	/**
	 * Filter whether a diagnostic message should be logged.
	 *
	 * @since 4.1.7
	 *
	 * @param bool   $should_log Whether the message should be logged.
	 * @param string $message    Message to log.
	 * @param array  $context    Additional context data.
	 */
	$should_log = apply_filters( 'wpvulnerability_should_log', $should_log, $message, $context );

	if ( ! $should_log ) {
		return;
	}

	$log_entry = array(
		'plugin'  => 'wpvulnerability',
		'message' => (string) $message,
	);

	if ( ! empty( $context ) ) {
		$log_entry['context'] = (array) $context;
	}

	$encoded_entry = wp_json_encode( $log_entry );

	if ( false === $encoded_entry ) {
		$encoded_entry = sprintf(
			'wpvulnerability: %s',
			sanitize_text_field( $log_entry['message'] )
		);
	}

	error_log( $encoded_entry ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
}

/**
 * Retrieve the cache expiration in hours.
 *
 * The value can be defined via the WPVULNERABILITY_CACHE_HOURS constant,
 * configured in the plugin settings, or falls back to 12 hours.
 *
 * @since 4.1.0
 *
 * @return int Cache duration in hours.
 */
function wpvulnerability_cache_hours() {
	$default = 12;

	if ( defined( 'WPVULNERABILITY_CACHE_HOURS' ) && WPVULNERABILITY_CACHE_HOURS !== $default ) {
		return (int) WPVULNERABILITY_CACHE_HOURS;
	}

	$settings = is_multisite() ? get_site_option( 'wpvulnerability-config', array() ) : get_option( 'wpvulnerability-config', array() );
	if ( isset( $settings['cache'] ) ) {
		$cache = (int) $settings['cache'];
		if ( in_array( $cache, array( 1, 6, 12, 24 ), true ) ) {
			return $cache;
		}
	}

	return $default;
}

/**
 * Retrieve the supported log retention values.
 *
 * @since 4.2.0
 *
 * @return int[] Valid log retention periods expressed in days. The value "0" disables retention.
 */
function wpvulnerability_get_log_retention_values() {
	return array( 0, 1, 7, 14, 28 );
}

/**
 * Determine whether log retention is forced via a constant.
 *
 * @since 4.2.0
 *
 * @return int|null Number of days when forced, or null when editable.
 */
function wpvulnerability_forced_log_retention() {
	if ( defined( 'WPVULNERABILITY_LOG_RETENTION_DAYS' ) ) {
		$forced = (int) WPVULNERABILITY_LOG_RETENTION_DAYS;
		if ( in_array( $forced, wpvulnerability_get_log_retention_values(), true ) ) {
			return $forced;
		}
	}

	return null;
}

/**
 * Retrieve the configured log retention period in days.
 *
 * The value can be defined through the WPVULNERABILITY_LOG_RETENTION_DAYS constant,
 * configured via the settings UI, or falls back to seven days.
 *
 * @since 4.2.0
 *
 * @return int Log retention in days. Zero disables retention.
 */
function wpvulnerability_log_retention_days() {
	$default = 7;

	$forced = wpvulnerability_forced_log_retention();
	if ( null !== $forced ) {
		return $forced;
	}

	$settings = is_multisite() ? get_site_option( 'wpvulnerability-config', array() ) : get_option( 'wpvulnerability-config', array() );
	if ( isset( $settings['log_retention'] ) ) {
		$retention = (int) $settings['log_retention'];
		if ( in_array( $retention, wpvulnerability_get_log_retention_values(), true ) ) {
			return $retention;
		}
	}

	return $default;
}

/**
 * Register the custom post type used to store API logs.
 *
 * @since 4.2.0
 *
 * @return void
 */
function wpvulnerability_register_log_post_type() {
	register_post_type(
		'wpvulnerability_log',
		array(
			'labels'              => array(
				'name'          => __( 'WPVulnerability Logs', 'wpvulnerability' ),
				'singular_name' => __( 'WPVulnerability Log', 'wpvulnerability' ),
			),
			'public'              => false,
			'exclude_from_search' => true,
			'publicly_queryable'  => false,
			'show_ui'             => false,
			'show_in_menu'        => false,
			'supports'            => array( 'title', 'editor' ),
		)
	);
}
add_action( 'init', 'wpvulnerability_register_log_post_type' );

/**
 * Determine whether a URL should be logged as an API call.
 *
 * @since 4.2.0
 *
 * @param string $url Requested URL.
 *
 * @return bool True when the URL targets the configured API host, false otherwise.
 */
function wpvulnerability_should_log_api_request( $url ) {
	if ( 0 >= wpvulnerability_log_retention_days() ) {
		return false;
	}

	$target_host = wp_parse_url( $url, PHP_URL_HOST );
	$api_host    = wp_parse_url( WPVULNERABILITY_API_HOST, PHP_URL_HOST );

	if ( empty( $target_host ) || empty( $api_host ) ) {
		return false;
	}

	return 0 === strcasecmp( $target_host, $api_host );
}

/**
 * Convert the HTTP response into a storable string for the log entry.
 *
 * @since 4.2.0
 *
 * @param array|WP_Error $response Response returned by wp_remote_get().
 *
 * @return string Encoded response body ready for storage.
 */
function wpvulnerability_prepare_log_body( $response ) {
	if ( is_wp_error( $response ) ) {
		$error_data = array(
			'code'    => $response->get_error_code(),
			'message' => $response->get_error_message(),
			'errors'  => $response->errors,
			'data'    => $response->error_data,
		);
		$body       = wp_json_encode( $error_data );
		if ( false === $body ) {
			$body = serialize( $error_data ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.serialize_serialize
		}
		return $body;
	}

	$body = wp_remote_retrieve_body( $response );
	if ( '' === $body ) {
		return wp_json_encode( array( 'message' => 'empty body' ) );
	}

	return $body;
}

/**
 * Persist an API log entry when appropriate.
 *
 * @since 4.2.0
 *
 * @param string          $url      Requested URL.
 * @param array|WP_Error  $response Response returned by wp_remote_get().
 *
 * @return void
 */
function wpvulnerability_maybe_log_api_response( $url, $response ) {
	if ( ! wpvulnerability_should_log_api_request( $url ) ) {
		return;
	}

	$log_id = wp_insert_post(
		array(
			'post_type'    => 'wpvulnerability_log',
			'post_status'  => 'publish',
			'post_title'   => wp_strip_all_tags( $url ),
			'post_content' => wp_slash( wpvulnerability_prepare_log_body( $response ) ),
			'post_author'  => 0,
		),
		true
	);

	if ( is_wp_error( $log_id ) ) {
		return;
	}
}

/**
 * Retrieve the pagination sizes available for the logs table.
 *
 * @since 4.4.0
 *
 * @return int[] Array of valid per-page options.
 */
function wpvulnerability_get_log_per_page_options() {
	return array( 10, 50, 100, 250, 1000 );
}

/**
 * Retrieve the default pagination size for the logs table.
 *
 * @since 4.4.0
 *
 * @return int Default per-page value.
 */
function wpvulnerability_get_default_log_per_page() {
	return 100;
}

/**
 * Retrieve log posts for the administration table.
 *
 * @since 4.2.0
 * @since 4.4.0 Added the $paged argument and updated the default page size.
 *
 * @param int $per_page Optional. Number of logs to return per page. Default 100.
 * @param int $paged    Optional. Page number to retrieve. Default 1.
 *
 * @return WP_Post[] Array of log posts.
 */
function wpvulnerability_get_api_logs( $per_page = 100, $paged = 1 ) {
	$per_page = max( 1, (int) $per_page );
	$paged    = max( 1, (int) $paged );
	$offset   = ( $paged - 1 ) * $per_page;

	return get_posts(
		array(
			'post_type'      => 'wpvulnerability_log',
			'post_status'    => 'publish',
			'posts_per_page' => $per_page,
			'orderby'        => 'date',
			'order'          => 'DESC',
			'offset'         => $offset,
			'no_found_rows'  => true,
		)
	);
}

/**
 * Count the total amount of stored API log entries.
 *
 * @since 4.4.0
 *
 * @return int Number of log posts.
 */
function wpvulnerability_count_api_logs() {
	$counts = wp_count_posts( 'wpvulnerability_log' );

	if ( ! $counts || ! isset( $counts->publish ) ) {
		return 0;
	}

	return (int) $counts->publish;
}

/**
 * Retrieve a single log entry ensuring it belongs to the plugin log post type.
 *
 * @since 4.2.0
 *
 * @param int $log_id Log post ID.
 *
 * @return WP_Post|null Post object on success, null otherwise.
 */
function wpvulnerability_get_api_log( $log_id ) {
	$log_id = (int) $log_id;
	if ( $log_id <= 0 ) {
		return null;
	}

	$log = get_post( $log_id );
	if ( ! $log || 'wpvulnerability_log' !== $log->post_type ) {
		return null;
	}

	return $log;
}

/**
 * Format stored log content into a pretty printed JSON string when possible.
 *
 * @since 4.2.0
 *
 * @param string $content Stored log body.
 *
 * @return string Formatted content.
 */
function wpvulnerability_format_log_content( $content ) {
	$content = (string) $content;
	if ( '' === $content ) {
		return '';
	}

	$decoded = json_decode( $content, true );
	if ( null === $decoded || JSON_ERROR_NONE !== json_last_error() ) {
		return $content;
	}

	$pretty = wp_json_encode( $decoded, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES );
	if ( false === $pretty ) {
		return $content;
	}

	return $pretty;
}

/**
 * Format the log date using the site's date and time settings.
 *
 * @since 4.2.0
 *
 * @param WP_Post $log Log post.
 *
 * @return string Formatted date string.
 */
function wpvulnerability_format_log_date( WP_Post $log ) {
	$format = trim( (string) ( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) ) );
	if ( '' === $format ) {
		$format = 'Y-m-d H:i:s';
	}

	return mysql2date( $format, $log->post_date, true );
}

/**
 * Delete logs that fall outside of the configured retention window.
 *
 * @since 4.2.0
 *
 * @return void
 */
function wpvulnerability_delete_expired_logs() {
	$retention = wpvulnerability_log_retention_days();
	if ( $retention <= 0 ) {
		return;
	}

	$threshold = gmdate( 'Y-m-d H:i:s', strtotime( '-' . $retention . ' days', time() ) );

	do {
		$logs = get_posts(
			array(
				'post_type'              => 'wpvulnerability_log',
				'post_status'            => 'publish',
				'fields'                 => 'ids',
				'posts_per_page'         => 100,
				'orderby'                => 'date',
				'order'                  => 'ASC',
				'no_found_rows'          => true,
				'cache_results'          => false,
				'update_post_term_cache' => false,
				'update_post_meta_cache' => false,
				'date_query'             => array(
					array(
						'before'    => $threshold,
						'inclusive' => false,
					),
				),
			)
		);

		if ( empty( $logs ) ) {
			break;
		}

		foreach ( $logs as $log_id ) {
			wp_delete_post( $log_id, true );
		}
	} while ( count( $logs ) >= 100 );
}

add_action( 'wpvulnerability_cleanup_logs', 'wpvulnerability_delete_expired_logs' );

/**
 * Normalize various truthy and falsy values into the expected 'y' or 'n' format.
 *
 * This helper ensures that configuration options stored as booleans or integers
 * in previous plugin versions are converted into the new string-based format.
 *
 * @since 4.1.1
 *
 * @param mixed $value Value to normalize.
 *
 * @return string Returns 'y' when enabled, 'n' otherwise.
 */
function wpvulnerability_normalize_yes_no( $value ) {
	if ( is_string( $value ) ) {
	$value = strtolower( trim( (string) $value ) );
	}

		$truthy = array( 'y', 'yes', '1', 1, true, 'true', 'on' );

		return in_array( $value, $truthy, true ) ? 'y' : 'n';
}

/**
 * Determine if a stored yes/no value should be treated as enabled.
 *
 * @since 4.1.1
 *
 * @param mixed $value Value to evaluate.
 *
 * @return bool True when enabled, false otherwise.
 */
function wpvulnerability_is_yes( $value ) {
		return 'y' === wpvulnerability_normalize_yes_no( $value );
}

/**
 * Normalize the notification configuration array.
 *
 * @since 4.1.1
 *
 * @param mixed $notify Notification configuration values.
 *
 * @return array Normalized notification configuration containing 'email', 'slack', and 'teams'.
 */
function wpvulnerability_normalize_notify_settings( $notify ) {
		$defaults   = array(
			'email' => 'n',
			'slack' => 'n',
			'teams' => 'n',
		);
		$normalized = array();

		if ( is_array( $notify ) ) {
			foreach ( $notify as $channel => $value ) {
					$normalized[ $channel ] = wpvulnerability_normalize_yes_no( $value );
			}
		}

		return array_merge( $defaults, $normalized );
}

/**
 * Sanitize a version string.
 *
 * This function removes any leading or trailing whitespace from the version string
 * and strips out any non-alphanumeric characters except for hyphens, underscores, and dots.
 *
 * @since 2.0.0
 *
 * @param string $version The version string to sanitize.
 *
 * @return string The sanitized version string.
 */
function wpvulnerability_sanitize_version( $version ) {
	// Remove any leading or trailing whitespace.
	$version = trim( (string) $version );

	// Strip out any non-alphanumeric characters except for hyphens, underscores, and dots.
	$version = preg_replace( '/[^a-zA-Z0-9_\-.]+/', '', $version );

	// Normalize WordPress pre-release build suffixes such as "-beta1-12345" to "-beta1".
	if ( preg_match( '/^(\d+\.\d+(?:\.\d+)?-(?:beta|rc)\d+)(?:-\d+)$/i', $version, $matches ) ) {
		$version = $matches[1];
	}

	return $version;
}

/**
 * Sanitize a version string and validate its format.
 *
 * This function sanitizes the input version string and checks it against a regular expression
 * to match the standard versioning format (major.minor[.patch[.build]]). It returns the matched version
 * if it conforms to the expected format; otherwise, it returns the original version.
 *
 * @since 3.5.0 Introduced.
 *
 * @param string|null $version The version string to sanitize and validate.
 * @return string|null The sanitized version string if it matches the standard format; otherwise, the original version string, or null when empty.
 */
function wpvulnerability_sanitize_and_validate_version( $version ) {
	if ( null === $version ) {
		return null;
	}

	// Sanitize the version string using the base sanitizer.
	$version = wpvulnerability_sanitize_version( $version );

	if ( '' === $version ) {
		return null;
	}

	// Validate format (major.minor[.patch[.build]]) and sanitize.
	if ( preg_match( '/^\d+\.\d+(\.\d+){0,2}(\.\d+)?/', $version, $match ) ) {
		if ( isset( $match[0] ) ) {
			return trim( $match[0] );
		}
	}

	return $version;
}

/**
 * Detects the version of SQLite using the SQLite3 extension or system commands.
 *
 * @since 3.5.0 Introduced.
 *
 * @return string|null The version of SQLite in the format N.n.n, N.n, etc., or null if it cannot be detected.
 */
function wpvulnerability_detect_sqlite() {
	// Initialize the version variable.
	$version = null;

	// First method: use the SQLite3 extension of PHP.
	if ( class_exists( 'SQLite3' ) ) {
		$sqlite       = new SQLite3( ':memory:' ); // Create an in-memory SQLite database.
		$version_info = $sqlite->version();

		if ( isset( $version_info['versionString'] ) ) {
			$version = $version_info['versionString'];
			// Replace "-N" at the end with ".N" if present.
			if ( preg_match( '/-(\d+)$/', $version, $suffix_matches ) ) {
				$version = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $version );
			}
		}
	}

	// Second method: use PDO SQLite when the SQLite3 extension is unavailable.
	if ( empty( $version ) && ! class_exists( 'SQLite3' ) && class_exists( 'PDO' ) ) {
		$drivers = \PDO::getAvailableDrivers();

		if ( is_array( $drivers ) && in_array( 'sqlite', $drivers, true ) ) {
			try {
				$pdo = new \PDO( 'sqlite::memory:' );
				$pdo->setAttribute( \PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION );
				$statement = $pdo->query( 'SELECT sqlite_version()' );

				if ( $statement ) {
					$result = $statement->fetchColumn();

					if ( false !== $result ) {
						$version = (string) $result;

						if ( preg_match( '/-(\d+)$/', $version, $suffix_matches ) ) {
							$version = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $version );
						}
					}
				}
			} catch ( Exception $exception ) {
				wpvulnerability_maybe_log(
					'PDO SQLite detection failed',
					array(
						'exception' => array(
							'code'    => $exception->getCode(),
							'message' => $exception->getMessage(),
						),
					)
				);
			}

			if ( isset( $pdo ) ) {
				$pdo = null;
			}
		}
	}

	// Third method: use system commands if the previous attempts fail and shell_exec is available.
	if ( empty( $version ) && wpvulnerability_can_shell_exec() ) {
		// Command to check SQLite version.
		$version_output = shell_exec( escapeshellcmd( 'sqlite3 --version' ) ); // phpcs:ignore

		if ( ! empty( $version_output ) && preg_match( '/(\d+\.\d+(?:\.\d+)?(?:-\d+)?)/', $version_output, $matches ) ) {
			$version = $matches[1];
			// Replace "-N" at the end with ".N" if present.
			if ( preg_match( '/-(\d+)$/', $version, $suffix_matches ) ) {
				$version = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $version );
			}
		}
	}

	// Return the sanitized and validated version or null if it cannot be detected.
	return wpvulnerability_sanitize_and_validate_version( $version );
}

/**
 * Detects the version of Redis using the Redis extension or system commands.
 *
 * @since 3.5.0 Introduced.
 *
 * @return string|null The version of Redis in the format N.n.n, N.n, etc., or null if it cannot be detected.
 */
function wpvulnerability_detect_redis() {
	// Initialize the version variable.
	$version = null;

	// First method: use the Redis extension of PHP.
	if ( class_exists( 'Redis' ) ) {
		$redis_client = null;
		$temporary_connection = false;

		// Attempt to reuse an existing Redis connection from WordPress object cache implementations.
		$cache_instance = null;
		if ( function_exists( 'wp_cache_get_instance' ) ) {
			$cache_instance = wp_cache_get_instance();
		} elseif ( isset( $GLOBALS['wp_object_cache'] ) && is_object( $GLOBALS['wp_object_cache'] ) ) {
			$cache_instance = $GLOBALS['wp_object_cache'];
		}

		if ( is_object( $cache_instance ) ) {
			foreach ( array( 'redis', 'redis_client', 'client', 'redis_instance', 'connection' ) as $property ) {
				if ( isset( $cache_instance->{$property} ) && $cache_instance->{$property} instanceof Redis ) {
					$redis_client = $cache_instance->{$property};
					break;
				}
			}

			if ( ! $redis_client instanceof Redis ) {
				foreach ( array( 'get_redis', 'get_client', 'redis', 'redis_instance' ) as $method ) {
					if ( ! method_exists( $cache_instance, $method ) ) {
						continue;
					}

					try {
						$reflection_method = new ReflectionMethod( $cache_instance, $method );
						if ( $reflection_method->getNumberOfRequiredParameters() > 0 || ! $reflection_method->isPublic() ) {
							continue;
						}
					} catch ( ReflectionException $exception ) {
						continue;
					}

					if ( ! is_callable( array( $cache_instance, $method ) ) ) {
						continue;
					}

					$maybe_client = $cache_instance->{$method}();
					if ( $maybe_client instanceof Redis ) {
						$redis_client = $maybe_client;
						break;
					}
				}
			}
		}

		if ( ! $redis_client instanceof Redis ) {
			$redis_client = new Redis();
			$temporary_connection = true;

			$host = defined( 'WP_REDIS_HOST' ) ? (string) WP_REDIS_HOST : '127.0.0.1';
			$port = defined( 'WP_REDIS_PORT' ) ? (int) WP_REDIS_PORT : 6379;
			$timeout = defined( 'WP_REDIS_TIMEOUT' ) ? (float) WP_REDIS_TIMEOUT : 0.0;
			$connected = false;

			try {
				if ( defined( 'WP_REDIS_PATH' ) && '' !== (string) WP_REDIS_PATH ) {
					$connected = $redis_client->connect( (string) WP_REDIS_PATH );
				} elseif ( $timeout > 0 ) {
					$connected = $redis_client->connect( $host, $port, $timeout );
				} else {
					$connected = $redis_client->connect( $host, $port );
				}

				if ( $connected ) {
					$username = defined( 'WP_REDIS_USERNAME' ) ? (string) WP_REDIS_USERNAME : '';
					$password = null;
					if ( defined( 'WP_REDIS_PASSWORD' ) ) {
						$password = (string) WP_REDIS_PASSWORD;
					} elseif ( defined( 'WP_REDIS_AUTH' ) ) {
						$password = (string) WP_REDIS_AUTH;
					}

					if ( '' !== $username && null !== $password ) {
						$redis_client->auth( array( $username, $password ) );
					} elseif ( null !== $password && '' !== $password ) {
						$redis_client->auth( $password );
					}

					if ( defined( 'WP_REDIS_DATABASE' ) ) {
						$redis_client->select( (int) WP_REDIS_DATABASE );
					}
				} else {
					$redis_client = null;
					$temporary_connection = false;
				}
			} catch ( RedisException $e ) {
				$redis_client = null;
				$temporary_connection = false;
			} catch ( Exception $e ) {
				$redis_client = null;
				$temporary_connection = false;
			}
		}

		if ( $redis_client instanceof Redis ) {
			try {
				$redis_info = $redis_client->info();

				if ( isset( $redis_info['redis_version'] ) ) {
					$version = $redis_info['redis_version'];
					// Replace "-N" at the end with ".N" if present.
					if ( preg_match( '/-(\d+)$/', $version, $suffix_matches ) ) {
						$version = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $version );
					}
				}
			} catch ( RedisException $e ) {
				// There is the PHP extension, but no Redis.
			} catch ( Exception $e ) {
				// There is the PHP extension, but retrieving info failed.
			} finally {
				if ( $temporary_connection && $redis_client instanceof Redis && method_exists( $redis_client, 'close' ) ) {
					$redis_client->close();
				}
			}
		}
	}

	// Second method: use system commands if the first fails and shell_exec is available.
	if ( empty( $version ) && wpvulnerability_can_shell_exec() ) {
		// Command to check Redis version.
		$version_output = shell_exec( escapeshellcmd( 'redis-server --version' ) ); // phpcs:ignore

		if ( ! empty( $version_output ) && preg_match( '/redis-server\s+v=([\d.]+)/i', $version_output, $matches ) ) {
			$version = $matches[1];
			// Replace "-N" at the end with ".N" if present.
			if ( preg_match( '/-(\d+)$/', $version, $suffix_matches ) ) {
				$version = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $version );
			}
		}
	}

	// Return the sanitized and validated version or null if it cannot be detected.
	return wpvulnerability_sanitize_and_validate_version( $version );
}

/**
 * Normalizes Memcached version information returned by PHP extensions.
 *
 * @since 4.1.7
 *
 * @param mixed $version_info Version information as returned by Memcached::getVersion() or Memcache::getVersion().
 * @return string|null Normalized version string or null when it cannot be determined.
 */
function wpvulnerability_normalize_memcached_version_info( $version_info ) {
	$reported_versions = array();

	if ( is_array( $version_info ) ) {
		$reported_versions = $version_info;
	} elseif ( is_string( $version_info ) ) {
		$trimmed_version = trim( $version_info );
		if ( '' !== $trimmed_version ) {
			$reported_versions = array( $trimmed_version );
		}
	}

	foreach ( $reported_versions as $reported_version ) {
		$reported_version = (string) $reported_version;
		$reported_version = trim( $reported_version );

		if ( '' === $reported_version || '255.255.255' === $reported_version ) {
			continue;
		}

		if ( preg_match( '/-(\d+)$/', $reported_version, $suffix_matches ) ) {
			$reported_version = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $reported_version );
		}

		return $reported_version;
	}

	return null;
}

/**
 * Detects the version of Memcached using the Memcached extension or system commands.
 *
 * @since 3.5.0 Introduced.
 *
 * @return string|null The version of Memcached in the format N.n.n, N.n, etc., or null if it cannot be detected.
 */
function wpvulnerability_detect_memcached() {
	// Initialize the version variable.
	$version = null;

	// First method: use the Memcached extension of PHP.
	if ( class_exists( 'Memcached' ) ) {
		try {
			$memcached    = new Memcached();
			$version_info = method_exists( $memcached, 'getVersion' ) ? $memcached->getVersion() : null;
			$version      = wpvulnerability_normalize_memcached_version_info( $version_info );

			// Try to reuse any configured servers for persistent pools if no version was retrieved.
			if ( empty( $version ) && method_exists( $memcached, 'getServerList' ) ) {
				$servers = $memcached->getServerList();

				if ( is_array( $servers ) && ! empty( $servers ) ) {
					if ( method_exists( $memcached, 'resetServerList' ) ) {
						$memcached->resetServerList();
					}

					foreach ( $servers as $server ) {
						if ( empty( $server['host'] ) ) {
							continue;
						}

						$host   = (string) $server['host'];
						$port   = isset( $server['port'] ) ? (int) $server['port'] : 11211;
						$weight = isset( $server['weight'] ) ? (int) $server['weight'] : 0;

						if ( method_exists( $memcached, 'addServer' ) ) {
							$memcached->addServer( $host, $port, $weight );
						}
					}

					$version_info = method_exists( $memcached, 'getVersion' ) ? $memcached->getVersion() : null;
					$version      = wpvulnerability_normalize_memcached_version_info( $version_info );
				}
			}
		} catch ( MemcachedException $e ) {
			// There is the PHP extension, but no Memcached service available.
			unset( $memcached );
		} catch ( Exception $e ) {
			// Unexpected error while checking the Memcached extension.
			unset( $memcached );
		}
	}

	if ( empty( $version ) && class_exists( 'Memcache' ) ) {
		try {
			$memcache     = new Memcache();
			$version_info = method_exists( $memcache, 'getVersion' ) ? $memcache->getVersion() : null;
			$version      = wpvulnerability_normalize_memcached_version_info( $version_info );
		} catch ( Exception $e ) {
			// Ignore errors from the legacy Memcache extension.
			unset( $memcache );
		}
	}

	// Second method: use system commands if the first fails and shell_exec is available.
	if ( empty( $version ) && wpvulnerability_can_shell_exec() ) {
		// Command to check Memcached version.
		$version_output = shell_exec( escapeshellcmd( 'memcached -h' ) ); // phpcs:ignore

		if ( ! empty( $version_output ) && preg_match( '/memcached\s+(\d+\.\d+(?:\.\d+)?(?:-\d+)?)/i', $version_output, $matches ) ) {
			$version = $matches[1];
			// Replace "-N" at the end with ".N" if present.
			if ( preg_match( '/-(\d+)$/', $version, $suffix_matches ) ) {
				$version = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $version );
			}
		}
	}

	// Return the sanitized and validated version or null if it cannot be detected.
	return wpvulnerability_sanitize_and_validate_version( $version );
}

/**
 * Detects the installed PHP version using available runtime information.
 *
 * @since 2.0.0
 *
 * @return string|null The detected PHP version in N.n or N.n.n format, or null if unavailable.
 */
function wpvulnerability_detect_php() {
	// Initialize the version variable.
	$version = null;

	// First method: use the PHP_VERSION constant.
	if ( defined( 'PHP_VERSION' ) ) {
		$version = PHP_VERSION;
	}

	// First method: use the phpversion function.
	if ( empty( $version ) && function_exists( 'phpversion' ) ) {
		$version = phpversion();
	}

	// Second method: use system commands if the first fails and shell_exec is available.
	if ( empty( $version ) && wpvulnerability_can_shell_exec() ) {
		// Command to check PHP version.
		$version_output = shell_exec( escapeshellcmd( 'php -v' ) ); // phpcs:ignore

		if ( ! empty( $version_output ) && preg_match( '/PHP\s+(\d+\.\d+(?:\.\d+)?(?:-\d+)?)/i', $version_output, $matches ) ) {
			$version = $matches[1];
			// Replace "-N" at the end with ".N" if present.
			if ( preg_match( '/-(\d+)$/', $version, $suffix_matches ) ) {
				$version = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $version );
			}
		}
	}

	// Return the sanitized and validated PHP version or null if it cannot be detected.
	return wpvulnerability_sanitize_and_validate_version( $version );
}

/**
 * Detects the version of cURL using the cURL extension or system commands.
 *
 * @since 3.5.0 Introduced.
 *
 * @return string|null The version of cURL in the format N.n.n, N.n, etc., or null if it cannot be detected.
 */
function wpvulnerability_detect_curl() {
	// Product name for consistency.
	$version = null;

	// First method: use the cURL extension of PHP.
	if ( function_exists( 'curl_version' ) ) {
		$curl_info = curl_version();
		$version   = isset( $curl_info['version'] ) ? $curl_info['version'] : null;
	}

	// Second method: use system commands if the first fails and shell_exec is available.
	if ( empty( $version ) && wpvulnerability_can_shell_exec() ) {
		// Command to check cURL version.
		$version_output = shell_exec( escapeshellcmd( 'curl --version' ) ); // phpcs:ignore

		if ( ! empty( $version_output ) && preg_match( '/curl\s+(\d+\.\d+(?:\.\d+)?(?:-\d+)?)/i', $version_output, $matches ) ) {
			$version = $matches[1];
			// Replace "-N" at the end with ".N" if present.
			if ( preg_match( '/-(\d+)$/', $version, $suffix_matches ) ) {
				$version = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $version );
			}
		}
	}

	// Return the sanitized and validated version or null if it cannot be detected.
	return wpvulnerability_sanitize_and_validate_version( $version );
}

/**
 * Detects the version of ImageMagick using the Imagick extension or system commands.
 *
 * @since 3.5.0 Introduced.
 *
 * @return string|null The version of ImageMagick in the format N.n.n, N.n, etc., or null if it cannot be detected.
 */
function wpvulnerability_detect_imagemagick() {
	// Product name to avoid repetition and facilitate future changes.
	$version = null;

	// First method: use the Imagick extension of PHP.
	if ( extension_loaded( 'imagick' ) && class_exists( 'Imagick' ) ) {
		$version_info = null;

		try {
			$imagick      = new Imagick();
			$version_info = $imagick->getVersion();
		} catch ( \ImagickException $exception ) {
			$version_info = null;
			wpvulnerability_maybe_log(
				'ImageMagick version detection via the Imagick PHP extension failed.',
				array(
					'exception' => get_class( $exception ),
					'error'     => $exception->getMessage(),
				)
			);
		}

		if ( is_array( $version_info ) && isset( $version_info['versionString'] ) ) {
			// Extract the version using a regular expression.
			if ( preg_match( '/ImageMagick\s+(\d+\.\d+(?:\.\d+)?(?:-\d+)?)/i', $version_info['versionString'], $matches ) ) {
				$version = $matches[1];
				// Replace "-N" at the end with ".N" if present.
				if ( preg_match( '/-(\d+)$/', $version, $suffix_matches ) ) {
					$version = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $version );
				}
			}
		}
	}

	// Second method: use system commands if the first fails and shell_exec is available.
	if ( empty( $version ) && wpvulnerability_can_shell_exec() ) {
		// Try common CLI entry points in order of preference: magick, convert and identify.
		$commands = array( 'magick -version', 'convert -version', 'identify -version' );

		foreach ( $commands as $cmd ) {
			// Execute the command securely.
			$version_output = shell_exec( escapeshellcmd( $cmd ) ); // phpcs:ignore

			if ( ! empty( $version_output ) && preg_match( '/ImageMagick\s+(\d+\.\d+(?:\.\d+)?(?:-\d+)?)/i', $version_output, $matches ) ) {
				$version = $matches[1];
				// Replace "-N" at the end with ".N" if present.
				if ( preg_match( '/-(\d+)$/', $version, $suffix_matches ) ) {
					$version = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $version );
				}
				break; // Exit the loop once the version is found.
			}
		}
	}

	// Return the version or null if it cannot be detected.
	return wpvulnerability_sanitize_and_validate_version( $version );
}

/**
 * Retrieves the Apache HTTP Server version using available PHP APIs.
 *
 * The version is first gathered using the {@see apache_get_version()} function if it exists. The detected
 * version is sanitized and validated to ensure it matches the expected `major.minor.patch` format. The numeric
 * portion is extracted before sanitization so that decorated version strings are normalized. A filter
 * allows overriding the detected version, which is helpful for testing environments where the Apache API is
 * unavailable.
 *
 * @since 4.1.7
 *
 * @return string|null The sanitized Apache version or null when it cannot be determined.
 */
function wpvulnerability_get_apache_version() {
	$apache_version           = null;
	$normalize_apache_version = static function ( $value ) {
		if ( ! is_string( $value ) ) {
			return null;
		}

		$value = trim( $value );

		if ( '' === $value ) {
			return null;
		}

		if ( preg_match( '/(\d+\.\d+(?:\.\d+){0,2})/', $value, $matches ) ) {
			$value = $matches[1];
		}

		$value = wpvulnerability_sanitize_and_validate_version( $value );

		if ( null === $value ) {
			return null;
		}

		if ( ! preg_match( '/^\d/', $value ) ) {
			return null;
		}

		return $value;
	};

	if ( function_exists( 'apache_get_version' ) ) {
		$raw_version = apache_get_version();

		if ( is_string( $raw_version ) ) {
			$apache_version = $normalize_apache_version( $raw_version );
		}
	}

	/**
	 * Filter the detected Apache version.
	 *
	 * This filter allows overriding the detected Apache HTTP Server version. Returning a falsy value will cause
	 * the detection routine to fall back to other discovery mechanisms.
	 *
	 * @since 4.1.7
	 *
	 * @param string|null $apache_version The sanitized Apache version, or null if detection failed.
	 */
	$apache_version = apply_filters( 'wpvulnerability_detect_webserver_apache_version', $apache_version );

	if ( null !== $apache_version ) {
		$apache_version = $normalize_apache_version( $apache_version );
	}

	return $apache_version;
}

/**
 * Detects the web server software and version from the SERVER_SOFTWARE server variable.
 *
 * This function attempts to identify the web server software (e.g., Apache, nginx) and its version
 * based on the 'SERVER_SOFTWARE' environment variable provided by the server. It uses regular expressions
 * to parse the web server name and version. The function also sanitizes the detected version number
 * to a standard format (major.minor.patch).
 *
 * @since 3.2.0 Introduced.
 *
 * @return array Returns an associative array with three keys:
 *               'id' => A short, lowercase identifier for the web server (e.g., 'apache', 'nginx'),
 *               'name' => A more readable name for the web server (e.g., 'Apache HTTPD', 'nginx'),
 *               'version' => The detected version of the web server, sanitized to a standard format.
 */
function wpvulnerability_detect_webserver() {
	// Initialize an array to hold the web server information.
	$webserver = array(
		'id'      => null,
		'name'    => null,
		'version' => null,
	);

	$apache_version = wpvulnerability_get_apache_version();

	if ( null !== $apache_version ) {
		$webserver['id']      = 'apache';
		$webserver['name']    = 'Apache HTTPD';
		$webserver['version'] = $apache_version;

		return $webserver;
	}

	// Check if the SERVER_SOFTWARE variable is set.
	if ( isset( $_SERVER['SERVER_SOFTWARE'] ) ) {
		// Trim and sanitize the server software string.
		$webserver_software = trim( (string) wp_kses( wp_unslash( $_SERVER['SERVER_SOFTWARE'] ), 'strip' ) );

		// Use regular expressions to extract the web server name and version.
		if ( preg_match( '/^([^\s\/]+)\/?([^\s]*)/', $webserver_software, $matches ) ) {
			$webserver['name']    = isset( $matches[1] ) ? trim( (string) $matches[1] ) : null;
			$webserver['version'] = isset( $matches[2] ) ? trim( (string) $matches[2] ) : null;

			// Replace "-N" at the end of the version with ".N" if present.
			if ( preg_match( '/-(\d+)$/', $webserver['version'], $suffix_matches ) ) {
				$webserver['version'] = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $webserver['version'] );
			}
		}
	}

		// Normalize and set the web server ID based on the detected name.
	if ( ! empty( $webserver['name'] ) ) {
		$normalized_name = strtolower( (string) $webserver['name'] );
		$webserver['id'] = trim( (string) preg_replace( '/[^a-z0-9]+/', '-', $normalized_name ), '-' );
		switch ( $normalized_name ) {
			case 'httpd':
			case 'apache':
				$webserver['id']   = 'apache';
				$webserver['name'] = 'Apache HTTPD';
				break;
			case 'nginx':
				$webserver['id']   = 'nginx';
				$webserver['name'] = 'nginx';
				break;
			case 'openresty':
				$webserver['id']   = 'nginx';
				$webserver['name'] = 'OpenResty';
				break;
			case 'tengine':
				$webserver['id']   = 'nginx';
				$webserver['name'] = 'Tengine';
				break;
			// Additional web servers can be added here.
		}
	}

	// If the version is not detected, try to get it from the OS.
	if ( empty( $webserver['version'] ) && wpvulnerability_can_shell_exec() ) {
		if ( 'apache' === $webserver['id'] ) {
			$apache_version = shell_exec( escapeshellcmd( 'apache2 -v 2>&1' ) ); // phpcs:ignore
			if ( empty( $apache_version ) ) {
				$apache_version = shell_exec( escapeshellcmd( 'httpd -v 2>&1' ) ); // phpcs:ignore
			}
			if ( ! empty( $apache_version ) && preg_match( '/Apache\/([\d.]+)/', $apache_version, $version_matches ) ) {
				$webserver['version'] = $version_matches[1];
				// Replace "-N" at the end with ".N" if present.
				if ( preg_match( '/-(\d+)$/', $webserver['version'], $suffix_matches ) ) {
					$webserver['version'] = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $webserver['version'] );
				}
			}
		} elseif ( 'nginx' === $webserver['id'] ) {
			$nginx_version = shell_exec( escapeshellcmd( 'nginx -v 2>&1' ) ); // phpcs:ignore
			if ( ! empty( $nginx_version ) && preg_match( '/nginx\/([\d.]+)/', $nginx_version, $version_matches ) ) {
				$webserver['version'] = $version_matches[1];
				// Replace "-N" at the end with ".N" if present.
				if ( preg_match( '/-(\d+)$/', $webserver['version'], $suffix_matches ) ) {
					$webserver['version'] = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $webserver['version'] );
				}
			} else {
				$angie_version = shell_exec( escapeshellcmd( 'angie -v 2>&1' ) ); // phpcs:ignore
				if ( ! empty( $angie_version ) && preg_match( '/angie\/([\d.]+)/', $angie_version, $version_matches ) ) {
					$webserver['version'] = $version_matches[1];
					// Replace "-N" at the end with ".N" if present.
					if ( preg_match( '/-(\d+)$/', $webserver['version'], $suffix_matches ) ) {
						$webserver['version'] = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $webserver['version'] );
					}
				}
			}
	}
}

	// Sanitize and validate the web server version format.
	if ( null !== $webserver['version'] && '' !== $webserver['version'] ) {
		// Sanitize the version number to ensure it's in a 'major.minor.patch' format.
		$webserver['version'] = wpvulnerability_sanitize_and_validate_version( $webserver['version'] );

		if ( null !== $webserver['version'] && ! preg_match( '/^\d+(?:\.\d+)*$/', $webserver['version'] ) ) {
			$webserver['version'] = null;
		}
	}

	// Return the detected web server information.
	return $webserver;
}

/**
 * Detects the SQL server software and version from the database server.
 *
 * This function identifies the SQL server software (e.g., MariaDB, MySQL) and its version
 * by querying the database using the 'SHOW VARIABLES' command. It parses the server name
 * and version using the results and sanitizes the detected version number to a standard format (major.minor.patch).
 *
 * @since 3.4.0
 *
 * @return array Returns an associative array with three keys:
 *               'id' => A short, lowercase identifier for the SQL server (e.g., 'mariadb', 'mysql'),
 *               'name' => A more readable name for the SQL server (e.g., 'MariaDB', 'MySQL'),
 *               'version' => The detected version of the SQL server, sanitized to a standard format.
 */
function wpvulnerability_detect_sqlserver() {
	// Initialize an array to hold the SQL server information.
	$sqlserver = array(
		'id'      => null,
		'name'    => null,
		'version' => null,
	);

	global $wpdb;

	$version_source     = '';
	$server_info_string = '';

	// Query to get the database server type (version_comment).
	$database_results = $wpdb->get_results( $wpdb->prepare( 'SHOW VARIABLES LIKE %s', 'version_comment' ) ); // phpcs:ignore
	if ( $wpdb->last_error && defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		do_action( 'wpdb_last_error', $wpdb->last_error );
	}

	// Process the results to determine the database type.
	if ( ! empty( $database_results ) && isset( $database_results[0]->Value ) ) {
		$possible_database = trim( (string) $database_results[0]->Value );

		if ( false !== stripos( $possible_database, 'mariadb' ) ) {
			$sqlserver['id']   = 'mariadb';
			$sqlserver['name'] = 'MariaDB';
		} elseif ( false !== stripos( $possible_database, 'mysql' ) ) {
			$sqlserver['id']   = 'mysql';
			$sqlserver['name'] = 'MySQL';
		}
	}

	// Query to get the database server version.
	$version_results = $wpdb->get_results( $wpdb->prepare( 'SHOW VARIABLES LIKE %s', 'version' ) ); // phpcs:ignore
	if ( $wpdb->last_error && defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		do_action( 'wpdb_last_error', $wpdb->last_error );
	}

	if ( ! empty( $version_results ) && isset( $version_results[0]->Value ) ) {
		$version_source = trim( (string) $version_results[0]->Value );
	}

	if ( empty( $sqlserver['id'] ) ) {
		if ( method_exists( $wpdb, 'db_server_info' ) ) {
			$server_info_string = trim( (string) $wpdb->db_server_info() );
		}

		if ( '' === $server_info_string ) {
			$server_info_string = $wpdb->get_var( 'SELECT VERSION()' ); // phpcs:ignore
			if ( $wpdb->last_error && defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				do_action( 'wpdb_last_error', $wpdb->last_error );
			}
			$server_info_string = is_string( $server_info_string ) ? trim( $server_info_string ) : '';
		}

		if ( '' !== $server_info_string ) {
			if ( false !== stripos( $server_info_string, 'mariadb' ) ) {
				$sqlserver['id']   = 'mariadb';
				$sqlserver['name'] = 'MariaDB';
			} elseif ( false !== stripos( $server_info_string, 'mysql' ) ) {
				$sqlserver['id']   = 'mysql';
				$sqlserver['name'] = 'MySQL';
			}

			if ( '' === $version_source ) {
				$version_source = $server_info_string;
			}
		}
	}

	if ( '' !== $version_source ) {
		if ( preg_match( '/(\d+\.\d+\.\d+(?:-\d+)?)/', $version_source, $match ) ||
			preg_match( '/(\d+\.\d+(?:-\d+)?)/', $version_source, $match ) ) {
			$sqlserver['version'] = $match[1];
		} elseif ( 'mysql' === $sqlserver['id'] ) {
			// Fallback to the entire version string if regex doesn't match.
			$sqlserver['version'] = $version_source;
		}
	}

	// Replace "-N" at the end with ".N" if present.
	if ( ! empty( $sqlserver['version'] ) && preg_match( '/-(\d+)$/', $sqlserver['version'], $suffix_matches ) ) {
		$sqlserver['version'] = preg_replace( '/-(\d+)$/', '.' . $suffix_matches[1], $sqlserver['version'] );
	}

	// Sanitize and validate the version format.
	if ( ! empty( $sqlserver['version'] ) ) {
		$sqlserver['version'] = wpvulnerability_sanitize_and_validate_version( $sqlserver['version'] );
	}

	// Return the detected SQL server information.
	return $sqlserver;
}

/**
 * Returns a human-readable HTML entity for the given comparison operator.
 *
 * This function takes a comparison operator in string format and returns
 * its corresponding HTML entity for better readability in web contexts.
 *
 * @since 2.0.0
 *
 * @param string $op The operator string to prettify.
 *
 * @return string The pretty operator HTML string.
 */
function wpvulnerability_pretty_operator( $op ) {
	// Normalize the operator string to lowercase and trim whitespace.
	$op = trim( strtolower( (string) $op ) );

	// Define an associative array mapping operators to their HTML entities.
	$operator_map = array(
		'lt' => '&lt;&nbsp;',  // Less than.
		'le' => '&le;&nbsp;',  // Less than or equal to.
		'gt' => '&gt;&nbsp;',  // Greater than.
		'ge' => '&ge;&nbsp;',  // Greater than or equal to.
		'eq' => '&equals;&nbsp;', // Equal to.
		'ne' => '&ne;&nbsp;',  // Not equal to.
	);

	// Return the corresponding HTML entity, or the original operator if not recognized.
	return isset( $operator_map[ $op ] ) ? $operator_map[ $op ] : $op;
}

/**
 * Returns a human-readable severity level.
 *
 * This function takes a severity string and returns a human-readable
 * severity level, localized for translation.
 *
 * @since 2.0.0
 *
 * @param string $severity The severity string to prettify.
 *
 * @return string The human-readable severity string.
 */
function wpvulnerability_severity( $severity ) {
	// Normalize the severity string to lowercase and trim whitespace.
	$severity = trim( strtolower( (string) $severity ) );

	// Define an associative array mapping severity codes to their human-readable equivalents.
	$severity_map = array(
		'n' => __( 'None', 'wpvulnerability' ),      // No severity.
		'l' => __( 'Low', 'wpvulnerability' ),       // Low severity.
		'm' => __( 'Medium', 'wpvulnerability' ),    // Medium severity.
		'h' => __( 'High', 'wpvulnerability' ),      // High severity.
		'c' => __( 'Critical', 'wpvulnerability' ),   // Critical severity.
	);

	// Return the corresponding human-readable severity, or the original if not recognized.
	return isset( $severity_map[ $severity ] ) ? $severity_map[ $severity ] : $severity;
}

/**
 * Retrieves vulnerabilities information from the API.
 *
 * This function fetches vulnerability information based on the provided type and slug.
 * It supports caching to minimize API requests and improve performance.
 *
 * @since 2.0.0
 *
 * @param string $type  The type of vulnerability. Can be 'core', 'plugin', or 'theme'.
 * @param string $slug  The slug of the plugin or theme. For core vulnerabilities, it is the version string.
 * @param int    $cache Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|bool An array with the vulnerability information or false if there's an error.
 */
function wpvulnerability_get( $type, $slug = '', $cache = 1 ) {
	// Validate vulnerability type and normalize.
	$type        = strtolower( trim( (string) $type ) );
	$valid_types = array( 'core', 'plugin', 'theme' );

	if ( ! in_array( $type, $valid_types, true ) ) {
		wp_die( 'Unknown vulnerability type sent.' );
	}

	// Validate slug for plugin or theme.
	if ( ( 'plugin' === $type || 'theme' === $type ) && empty( sanitize_title( $slug ) ) ) {
		return false;
	}

	// Validate slug for core.
	if ( 'core' === $type && ! wpvulnerability_sanitize_version( $slug ) ) {
		return false;
	}

	// Cache key.
	$key = 'wpvulnerability_' . $type . '_' . $slug;

	// Attempt to retrieve cached data.
	$vulnerability_data = $cache ? ( is_multisite() ? get_site_transient( $key ) : get_transient( $key ) ) : null;

	// If not cached, fetch updated data.
	if ( empty( $vulnerability_data ) ) {
		$url      = WPVULNERABILITY_API_HOST . $type . '/' . $slug . '/';
		$response = wp_remote_get( $url, array( 'timeout' => 2500 ) );
		wpvulnerability_maybe_log_api_response( $url, $response );

		if ( ! is_wp_error( $response ) ) {
			$body = wp_remote_retrieve_body( $response );

			// Cache the response data.
			if ( is_multisite() ) {
					set_site_transient( $key, $body, HOUR_IN_SECONDS * wpvulnerability_cache_hours() );
			} else {
					set_transient( $key, $body, HOUR_IN_SECONDS * wpvulnerability_cache_hours() );
			}

			$vulnerability_data = $body; // Use the fresh data.
		}
	}

	return json_decode( $vulnerability_data, true );
}

/**
 * Retrieve vulnerabilities for a specific version of WordPress Core.
 *
 * This function fetches vulnerability information for a given version of WordPress Core.
 * If no version is provided, it retrieves vulnerabilities for the currently installed version.
 * It supports caching to minimize API requests and improve performance.
 *
 * @since 2.0.0
 *
 * @param string|null $version The version number of WordPress Core. If null, retrieves for the installed version.
 * @param int         $cache   Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false Array of vulnerabilities, or false on error.
 */
function wpvulnerability_get_core( $version = null, $cache = 1 ) {
	// Sanitize the version number.
	if ( ! wpvulnerability_sanitize_version( $version ) ) {
		$version = null; // Reset version if sanitization fails.
	}

	// If version number is null, retrieve for the installed version.
	if ( is_null( $version ) ) {
		$version = get_bloginfo( 'version' );
	}

	// Get vulnerabilities from the API.
	$response = wpvulnerability_get( 'core', $version, $cache );

	// Check for errors in the response.
	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Process vulnerabilities and return as an array.
	$vulnerabilities = array();
	foreach ( $response['data']['vulnerability'] as $v ) {
		$vulnerabilities[] = array(
			'name'   => isset( $v['name'] ) ? wp_kses( (string) $v['name'], 'strip' ) : null,
			'link'   => isset( $v['link'] ) ? esc_url_raw( (string) $v['link'] ) : null,
			'source' => isset( $v['source'] ) ? $v['source'] : null,
			'impact' => isset( $v['impact'] ) ? $v['impact'] : null,
		);
	}

	return $vulnerabilities;
}

/**
 * Determines if a vulnerability applies to the specified version of the plugin.
 *
 * @since 3.5.0 Introduced.
 *
 * @param array  $v The vulnerability data.
 * @param string $version The version of the plugin.
 *
 * @return bool True if the vulnerability applies, false otherwise.
 */
function wpvulnerability_is_vulnerability_applicable( $v, $version ) {
	// Check if the vulnerability has minimum and maximum versions.
	if ( isset( $v['operator']['min_operator'], $v['operator']['max_operator'] ) ) {
		return version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) &&
				version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] );
	}

	// Check if the vulnerability has only a maximum version.
	if ( isset( $v['operator']['max_operator'] ) ) {
		return version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] );
	}

	// Check if the vulnerability has only a minimum version.
	if ( isset( $v['operator']['min_operator'] ) ) {
		return version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] );
	}

	return false;
}

/**
 * Retrieves vulnerabilities for a specified plugin, optionally returning general plugin data.
 *
 * This function sanitizes the plugin slug and verifies the version number before querying the vulnerability API.
 * If `$data` is set to 1, it returns general information about the plugin instead of vulnerabilities.
 * The function returns an array of vulnerabilities or plugin data based on the `$data` parameter, or `false`
 * if no vulnerabilities are found or the version number is invalid and `$data` is not set.
 *
 * @since 2.0.0 Introduced.
 *
 * @param string $slug    The slug of the plugin to check for vulnerabilities.
 * @param string $version The version of the plugin to check. The function may return `false` if this is invalid and `$data` is not set.
 * @param int    $data    Optional. Set to 1 to return general plugin data instead of vulnerabilities. Default 0 (return vulnerabilities).
 * @param int    $cache   Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false An array of vulnerabilities or plugin data if `$data` is set to 1, or `false` if no vulnerabilities are found or the version number is invalid and `$data` is not set.
 */
function wpvulnerability_get_plugin( $slug, $version, $data = 0, $cache = 1 ) {
	// Sanitize the plugin slug.
	$slug = sanitize_title( $slug );

	// If the version number is invalid, return false unless $data is set.
	if ( ! wpvulnerability_sanitize_version( $version ) && ! $data ) {
		return false;
	}

	// Get the response from the vulnerability API.
	$response = wpvulnerability_get( 'plugin', $slug, $cache );

	// If $data is set to 1, return general plugin data.
	if ( 1 === $data && isset( $response['data'] ) ) {
		return array(
			'name'   => wp_kses( (string) $response['data']['name'], 'strip' ),
			'link'   => esc_url( (string) $response['data']['link'] ),
			'latest' => number_format( (int) $response['data']['latest'], 0, '.', '' ),
			'closed' => number_format( (int) $response['data']['closed'], 0, '.', '' ),
		);
	}

	// Check for errors or empty vulnerability data.
	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Create an empty array to store vulnerabilities.
	$vulnerabilities = array();

	// Loop through each vulnerability.
	foreach ( $response['data']['vulnerability'] as $v ) {
		// Check version constraints and add vulnerabilities accordingly.
		if ( wpvulnerability_is_vulnerability_applicable( $v, $version ) ) {
			$vulnerabilities[] = array(
				'name'        => wp_kses( (string) $v['name'], 'strip' ),
				'description' => wp_kses_post( (string) $v['description'] ),
				'versions'    => wp_kses(
					wpvulnerability_pretty_operator( isset( $v['operator']['min_operator'] ) ? $v['operator']['min_operator'] : '' ) .
					( isset( $v['operator']['min_version'] ) ? $v['operator']['min_version'] : '' ) . ' - ' .
					wpvulnerability_pretty_operator( isset( $v['operator']['max_operator'] ) ? $v['operator']['max_operator'] : '' ) .
					( isset( $v['operator']['max_version'] ) ? $v['operator']['max_version'] : '' ),
					'strip'
				),
				'version'     => wp_kses( (string) ( isset( $v['operator']['min_version'] ) ? $v['operator']['min_version'] : $v['operator']['max_version'] ), 'strip' ),
				'unfixed'     => (int) ( isset( $v['operator']['unfixed'] ) ? $v['operator']['unfixed'] : 0 ),
				'closed'      => (int) ( isset( $v['operator']['closed'] ) ? $v['operator']['closed'] : 0 ),
				'source'      => isset( $v['source'] ) ? $v['source'] : null,
				'impact'      => isset( $v['impact'] ) ? $v['impact'] : null,
			);
		}
	}

	return $vulnerabilities;
}

/**
 * Get vulnerabilities for a specific theme.
 *
 * This function retrieves and sanitizes the theme slug and version before querying the vulnerability API.
 * It returns an array of vulnerabilities if any are found, or false if there are none.
 *
 * @since 3.5.0
 *
 * @param string $slug    Slug of the theme.
 * @param string $version Version of the theme.
 * @param int    $cache   Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false Returns an array of vulnerabilities, or false if there are none.
 */
function wpvulnerability_get_theme( $slug, $version, $cache = 1 ) {
	// Sanitize the theme slug.
	$slug = sanitize_title( $slug );

	// Validate the version number.
	if ( ! wpvulnerability_sanitize_version( $version ) ) {
		return false; // Return false if the version is invalid.
	}

	// Get the response from the vulnerability API.
	$response = wpvulnerability_get( 'theme', $slug, $cache );

	// Check for errors or empty vulnerability data.
	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Process each vulnerability.
	$vulnerabilities = array();
	foreach ( $response['data']['vulnerability'] as $v ) {
		// Check if the version falls within the min and max operator range.
		if ( wpvulnerability_is_vulnerability_applicable( $v, $version ) ) {
			$vulnerabilities[] = array(
				'name'        => wp_kses( (string) $v['name'], 'strip' ),
				'description' => wp_kses_post( (string) $v['description'] ),
				'versions'    => wp_kses(
					wpvulnerability_pretty_operator( isset( $v['operator']['min_operator'] ) ? $v['operator']['min_operator'] : '' ) .
					( isset( $v['operator']['min_version'] ) ? $v['operator']['min_version'] : '' ) . ' - ' .
					wpvulnerability_pretty_operator( isset( $v['operator']['max_operator'] ) ? $v['operator']['max_operator'] : '' ) .
					( isset( $v['operator']['max_version'] ) ? $v['operator']['max_version'] : '' ),
					'strip'
				),
				'version'     => wp_kses(
					(string) ( isset( $v['operator']['min_version'] ) ? $v['operator']['min_version'] : $v['operator']['max_version'] ),
					'strip'
				),
				'unfixed'     => (int) ( isset( $v['operator']['unfixed'] ) ? $v['operator']['unfixed'] : 0 ),
				'closed'      => (int) ( isset( $v['operator']['closed'] ) ? $v['operator']['closed'] : 0 ),
				'source'      => isset( $v['source'] ) ? $v['source'] : null,
				'impact'      => isset( $v['impact'] ) ? $v['impact'] : null,
			);
		}
	}

	return $vulnerabilities;
}

/**
 * Get statistics.
 *
 * Returns an array with statistical information about vulnerabilities and their respective products.
 *
 * @since 2.0.0
 *
 * @param int $cache Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false Returns an array with the statistical information if successful, false otherwise.
 */
function wpvulnerability_get_statistics( $cache = 1 ) {
	$key = 'wpvulnerability_stats';

	// Attempt to get cached statistics.
	$vulnerability = $cache ? ( is_multisite() ? get_site_transient( $key ) : get_transient( $key ) ) : null;

	// If cached statistics are not available, retrieve them from the API.
	if ( empty( $vulnerability ) ) {
		$url      = WPVULNERABILITY_API_HOST;
	$response = wp_remote_get( $url, array( 'timeout' => 2500 ) );
	wpvulnerability_maybe_log_api_response( $url, $response );

		if ( ! is_wp_error( $response ) ) {
			$body = wp_remote_retrieve_body( $response );
			// Cache the response data.
			if ( is_multisite() ) {
							set_site_transient( $key, $body, HOUR_IN_SECONDS * wpvulnerability_cache_hours() );
			} else {
							set_transient( $key, $body, HOUR_IN_SECONDS * wpvulnerability_cache_hours() );
			}
			$vulnerability = $body; // Use the fresh data.
		}
	}

	// Decode the JSON response and check for statistics.
	$response = json_decode( $vulnerability, true );
	if ( ! isset( $response['stats'] ) ) {
		return false;
	}

	$sponsors = array();
	if ( isset( $response['behindtheproject']['sponsors'] ) && count( $response['behindtheproject']['sponsors'] ) ) {

		foreach ( $response['behindtheproject']['sponsors'] as $s ) {

			$sponsors[] = $s;

			unset( $s );
		}
	}

	$contributors = array();
	if ( isset( $response['behindtheproject']['contributors'] ) && count( $response['behindtheproject']['contributors'] ) ) {

		foreach ( $response['behindtheproject']['contributors'] as $s ) {

			$contributors[] = $s;

			unset( $s );
		}
	}

	// Return an array with statistical information.
	return array(
		'core'         => array(
			'versions' => (int) $response['stats']['products']['core'],
		),
		'plugins'      => array(
			'products'        => (int) $response['stats']['products']['plugins'],
			'vulnerabilities' => (int) $response['stats']['plugins'],
		),
		'themes'       => array(
			'products'        => (int) $response['stats']['products']['themes'],
			'vulnerabilities' => (int) $response['stats']['themes'],
		),
		'php'          => array(
			'vulnerabilities' => (int) $response['stats']['php'],
		),
		'apache'       => array(
			'vulnerabilities' => (int) $response['stats']['apache'],
		),
		'nginx'        => array(
			'vulnerabilities' => (int) $response['stats']['nginx'],
		),
		'mariadb'      => array(
			'vulnerabilities' => (int) $response['stats']['mariadb'],
		),
		'mysql'        => array(
			'vulnerabilities' => (int) $response['stats']['mysql'],
		),
		'imagemagick'  => array(
			'vulnerabilities' => (int) $response['stats']['imagemagick'],
		),
		'curl'         => array(
			'vulnerabilities' => (int) $response['stats']['curl'],
		),
		'memcached'    => array(
			'vulnerabilities' => (int) $response['stats']['memcached'],
		),
		'redis'        => array(
			'vulnerabilities' => (int) $response['stats']['redis'],
		),
		'sqlite'       => array(
			'vulnerabilities' => (int) $response['stats']['sqlite'],
		),
		'sponsors'     => $sponsors,
		'contributors' => $contributors,
		'updated'      => array(
			'unixepoch' => (int) $response['updated'],
			'datetime'  => gmdate( 'Y-m-d H:i:s', (int) $response['updated'] ),
			'iso8601'   => gmdate( 'c', (int) $response['updated'] ),
			'rfc2822'   => gmdate( 'r', (int) $response['updated'] ),
		),
	);
}

/**
 * Retrieves the latest vulnerability statistics.
 *
 * This function calls the wpvulnerability API to get fresh statistics related to vulnerabilities
 * and returns the updated information.
 *
 * @since 3.4.0
 *
 * @return array|false The updated vulnerability statistics, or false on error.
 */
function wpvulnerability_get_fresh_statistics() {
	// Call the function to get the latest vulnerability statistics.
	$statistics_api_response = wpvulnerability_get_statistics();

	// Return the response from the API.
	return $statistics_api_response;
}

/**
 * Retrieves and caches the latest vulnerability statistics.
 *
 * This function retrieves the most recent vulnerability statistics, caches the data,
 * and returns the information as a JSON-encoded array. The cache expiration timestamp is also updated.
 *
 * @since 3.4.0
 *
 * @return string JSON-encoded array containing the vulnerability statistics.
 */
function wpvulnerability_statistics_get() {
	// Retrieve fresh statistics.
	$statistics = wpvulnerability_get_fresh_statistics();

	// Cache the statistics data and the timestamp for cache expiration.
	$encoded_statistics   = wp_json_encode( $statistics );
		$cache_expiration = number_format( time() + ( 3600 * wpvulnerability_cache_hours() ), 0, '.', '' );

	if ( is_multisite() ) {
		update_site_option( 'wpvulnerability-statistics', $encoded_statistics );
		update_site_option( 'wpvulnerability-statistics-cache', $cache_expiration );
	} else {
		update_option( 'wpvulnerability-statistics', $encoded_statistics );
		update_option( 'wpvulnerability-statistics-cache', $cache_expiration );
	}

	// Return the JSON-encoded array of statistics data.
	return $encoded_statistics;
}

/**
 * Get vulnerabilities for a specific product version.
 *
 * This function retrieves vulnerability data for a specified product version.
 * It supports caching to minimize API requests and improve performance.
 *
 * @since 3.5.0
 *
 * @param string $type    The type of product (e.g., 'php', 'apache', 'nginx', 'mariadb', 'mysql').
 * @param string $version The version of the product to check.
 * @param int    $cache   Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false Returns an array of vulnerabilities, or false if there are none.
 */
function wpvulnerability_get_vulnerabilities( $type, $version, $cache = 1 ) {
	$key                = 'wpvulnerability_' . $type;
	$vulnerability_data = null;
	$vulnerability      = array();

	// Get cached statistics if available.
	if ( $cache ) {
		$vulnerability_data = is_multisite() ? get_site_transient( $key ) : get_transient( $key );
	}

	// If cached statistics are not available, retrieve them from the API and store them in cache.
	if ( empty( $vulnerability_data ) ) {
		$url      = WPVULNERABILITY_API_HOST . $type . '/' . $version . '/';
		$response = wp_remote_get( $url, array( 'timeout' => 2500 ) );
		wpvulnerability_maybe_log_api_response( $url, $response );

		if ( ! is_wp_error( $response ) ) {
			$body = wp_remote_retrieve_body( $response );
			if ( $cache ) {
				if ( is_multisite() ) {
						set_site_transient( $key, $body, HOUR_IN_SECONDS * wpvulnerability_cache_hours() );
				} else {
						set_transient( $key, $body, HOUR_IN_SECONDS * wpvulnerability_cache_hours() );
				}
			}
			$vulnerability_data = $body; // Use the fresh data.
		}
	}

	// If the response does not contain vulnerabilities, return false.
	$response = json_decode( $vulnerability_data, true );

	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Process each vulnerability.
	foreach ( $response['data']['vulnerability'] as $v ) {
		// Check if the version falls within the specified min and max operator range.
		if ( isset( $v['operator']['min_operator'], $v['operator']['max_operator'] ) &&
		     ! empty( $v['operator']['min_operator'] ) && ! empty( $v['operator']['max_operator'] ) ) {
			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) &&
				version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);
			}
		} elseif ( isset( $v['operator']['max_operator'] ) ) {
			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['max_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);
			}
		} elseif ( isset( $v['operator']['min_operator'] ) ) {
			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);
			}
		}
	}

	return $vulnerability;
}
