<?php
/**
 * Notifications functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Adds a custom schedule for a weekly cron job.
 *
 * This function adds a new schedule interval of one week (604800 seconds)
 * to the system's available cron schedules. It allows tasks to be scheduled
 * to run every week using the 'weekly' interval.
 *
 * @since 2.0.0
 *
 * @param array $schedules The existing system schedules.
 *
 * @return array The updated list of schedules with the added weekly interval.
 */
function wpvulnerability_add_every_week( $schedules ) {
	// Add a weekly schedule interval of 604800 seconds (1 week).
	$schedules['weekly'] = array(
		'interval' => 604800,
		'display'  => __( 'Every week', 'wpvulnerability' ),
	);

	// Return the modified list of schedules.
	return $schedules;
}
// Hook the function to the 'cron_schedules' filter to add the custom schedule.
add_filter( 'cron_schedules', 'wpvulnerability_add_every_week' );

/**
 * Adds a custom schedule for daily events.
 *
 * This function adds a new schedule interval of one day (86400 seconds)
 * to the system's available cron schedules. It allows tasks to be scheduled
 * to run every day using the 'daily' interval.
 *
 * @since 2.0.0
 *
 * @param array $schedules List of available schedules.
 *
 * @return array Modified list of available schedules with the added daily interval.
 */
function wpvulnerability_add_every_day( $schedules ) {
	// Define a new schedule with a 24 hour interval.
	$schedules['daily'] = array(
		'interval' => 86400,
		'display'  => __( 'Every day', 'wpvulnerability' ),
	);

	// Return the modified list of schedules.
	return $schedules;
}
// Hook the function to the 'cron_schedules' filter to add the custom schedule.
add_filter( 'cron_schedules', 'wpvulnerability_add_every_day' );

/**
 * Handles disabling email notifications via a front-end link.
 *
 * When the `wpvulnerability_disable_email` query parameter is present and the nonce is
 * valid, this function disables email notifications for the plugin and outputs
 * a confirmation message.
 *
 * @since 2.0.0
 */
function wpvulnerability_disable_email_handler() {
	if ( isset( $_GET['wpvulnerability_disable_email'] ) ) {
			$nonce = isset( $_GET['nonce'] ) ? sanitize_text_field( wp_unslash( $_GET['nonce'] ) ) : '';

		if ( ! wp_verify_nonce( $nonce, 'wpvulnerability-disable-email' ) ) {
				wp_die( esc_html__( 'Invalid request.', 'wpvulnerability' ) );
		}

			$settings = is_multisite() ? get_site_option( 'wpvulnerability-config' ) : get_option( 'wpvulnerability-config' );

		if ( ! is_array( $settings ) ) {
				$settings = array();
		}

						$settings['notify']          = wpvulnerability_normalize_notify_settings( isset( $settings['notify'] ) ? $settings['notify'] : array() );
						$settings['notify']['email'] = 'n';

		if ( is_multisite() ) {
				update_site_option( 'wpvulnerability-config', $settings );
		} else {
				update_option( 'wpvulnerability-config', $settings );
		}
			wp_die(
				esc_html__( 'You have unsubscribed from WPVulnerability notifications.', 'wpvulnerability' ),
				esc_html__( 'WPVulnerability', 'wpvulnerability' ),
				array( 'response' => 200 )
			);
	}
}

/**
 * Prepares the HTML email message.
 *
 * This function generates an HTML email message with the given title and content.
 * It includes basic styling and structure to ensure compatibility with most email clients.
 *
 * @since 2.0.0
 *
 * @param string $title   The title of the email message.
 * @param string $content The content of the email message.
 *
 * @return string The prepared HTML email message.
 */
function wpvulnerability_email_prepare( $title, $content ) {

	$message  = '';
	$message .= '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">' . "\n";
	$message .= '<html xmlns="http://www.w3.org/1999/xhtml" style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '<head>' . "\n";
	$message .= '	<meta name="viewport" content="width=device-width">' . "\n";
	$message .= '	<meta http-equiv="Content-Type" content="text/html; charset=UTF-8">' . "\n";
	$message .= '	<title>WPVulnerability</title>' . "\n";
	$message .= '	<style type="text/css">' . "\n";
	$message .= '	img { max-width: 100%; }' . "\n";
	$message .= '  body { -webkit-font-smoothing: antialiased; -webkit-text-size-adjust: none; width: 100% !important; height: 100%; line-height: 1.2em; }' . "\n";
	$message .= '  body { background-color: #f6f6f6; }' . "\n";
	$message .= '  @media only screen and (max-width: 640px) {' . "\n";
	$message .= '		body { padding: 0 !important; }' . "\n";
	$message .= '		h1, h2, h3, h4 { margin: 20px 0 5px 0 !important; }' . "\n";
	$message .= '		.container { padding: 0 !important; width: 100% !important; }' . "\n";
	$message .= '		.content { padding: 0 !important; }' . "\n";
	$message .= '		.content-wrap { padding: 10px !important; }' . "\n";
	$message .= '		.invoice { width: 100% !important; }' . "\n";
	$message .= '  }' . "\n";
	$message .= '	</style>' . "\n";
	$message .= '</head>' . "\n";
	$message .= '<body itemscope itemtype="http://schema.org/EmailMessage" style="box-sizing: border-box; -webkit-font-smoothing: antialiased; -webkit-text-size-adjust: none; width: 100% !important; height: 100%; line-height: 1.6em; background-color: #f6f6f6; margin: 0;" bgcolor="#f6f6f6">' . "\n";
	$message .= '	<table class="body-wrap" style="box-sizing: border-box; width: 100%; background-color: #f6f6f6; margin: 0;" bgcolor="#f6f6f6">' . "\n";
	$message .= '		<tr style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '			<td style="box-sizing: border-box; vertical-align: top; margin: 0;" valign="top"></td>' . "\n";
	$message .= '			<td class="container" width="600" style="box-sizing: border-box; vertical-align: top; display: block !important; max-width: 600px !important; clear: both !important; margin: 0 auto;" valign="top">' . "\n";
	$message .= '				<div class="content" style="box-sizing: border-box; max-width: 600px; display: block; margin: 0 auto; padding: 20px;">' . "\n";
	$message .= '					<table class="main" width="100%" cellpadding="0" cellspacing="0" style="box-sizing: border-box; border-radius: 3px; background-color: #fff; margin: 0; border: 1px solid #e9e9e9;" bgcolor="#fff">' . "\n";
	$message .= '						<tr style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '							<td class="content-wrap aligncenter" style="box-sizing: border-box; vertical-align: top; text-align: center; margin: 0; padding: 20px;" align="center" valign="top">' . "\n";
	$message .= '								<table width="100%" cellpadding="0" cellspacing="0" style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '									<tr style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '										<td class="content-block" style="box-sizing: border-box; vertical-align: top; margin: 0; padding: 0 0 20px; text-align: center;" valign="top">' . "\n";
	$message .= '											<img class="aligncenter" src="' . WPVULNERABILITY_PLUGIN_URL . 'assets/logo64.png" width="64" height="64" alt="WPVulnerability">' . "\n";
	$message .= '											<h1 class="aligncenter" style="box-sizing: border-box; color: #000; line-height: 1.2em; text-align: center; margin: 40px 0 0;" align="center">' . esc_html( $title ) . '</h1>' . "\n";

	// Add the site URL based on the multisite configuration.
	if ( is_multisite() ) {
		$message .= '											<p class="aligncenter" style="box-sizing: border-box; color: #000; line-height: 1.2em; text-align: center; margin: 5px 0 0;" align="center"><a href="' . esc_url( network_site_url() ) . '" target="_blank" rel="noopener noreferrer">' . esc_html( network_site_url() ) . '</a></p>' . "\n";
	} else {
		$message .= '											<p class="aligncenter" style="box-sizing: border-box; color: #000; line-height: 1.2em; text-align: center; margin: 5px 0 0;" align="center"><a href="' . esc_url( site_url() ) . '" target="_blank" rel="noopener noreferrer">' . esc_html( site_url() ) . '</a></p>' . "\n";
	}

	$message .= '										</td>' . "\n";
	$message .= '									</tr>' . "\n";
	$message .= '									<tr style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '										<td class="content-block alignleft" style="box-sizing: border-box; vertical-align: top; text-align: left; margin: 0; padding: 0 0 20px;" valign="top">' . "\n";
	$message .= $content; // Add the main content of the email.
	$message .= '									</td>' . "\n";
	$message .= '									</tr>' . "\n";
	$message .= '								</table>' . "\n";
	$message .= '								<div class="footer" style="box-sizing: border-box; width: 100%; clear: both; color: #999; margin: 0; padding: 20px;">' . "\n";
	$message .= '									<table width="100%" style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '										<tr style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '											<td class="aligncenter content-block" style="box-sizing: border-box; vertical-align: top; color: #999; text-align: center; margin: 0; padding: 0 0 20px;" align="center" valign="top">' . "\n";
	$message .= sprintf(
		// translators: %1$s the website of Database, %2$s database site name.
		__( 'Learn more about the WordPress Vulnerability Database API at <a href="%1$s">%2$s</a>', 'wpvulnerability' ),
		'https://www.wpvulnerability.com/',
		'WPVulnerability'
	);
	$message .= '											</td>' . "\n";
	$message .= '										</tr>' . "\n";

	$nonce = wp_create_nonce( 'wpvulnerability-disable-email' );
	if ( is_multisite() ) {
		$disable_url = network_home_url( '?wpvulnerability_disable_email=1&nonce=' . $nonce );
		$manage_url  = network_admin_url( 'settings.php?page=wpvulnerability-options' );
	} else {
		$disable_url = home_url( '?wpvulnerability_disable_email=1&nonce=' . $nonce );
		$manage_url  = admin_url( 'options-general.php?page=wpvulnerability-options' );
	}
	$message .= '									  <tr style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '										  <td class="aligncenter content-block" style="box-sizing: border-box; vertical-align: top; color: #999; text-align: center; margin: 0; padding: 0 0 20px;" align="center" valign="top"><a href="' . esc_url( $disable_url ) . '">' . esc_html__( 'Disable email notifications', 'wpvulnerability' ) . '</a> | <a href="' . esc_url( $manage_url ) . '">' . esc_html__( 'Manage notifications', 'wpvulnerability' ) . '</a></td>' . "\n";
	$message .= '										</tr>' . "\n";

	// Add the site URL in the footer based on the multisite configuration.
	if ( is_multisite() ) {
		$message .= '											<td class="aligncenter content-block" style="box-sizing: border-box; vertical-align: top; color: #999; text-align: center; margin: 0; padding: 0 0 20px;" align="center" valign="top"><a href="' . esc_url( network_site_url() ) . '" target="_blank" rel="noopener noreferrer">' . esc_html( network_site_url() ) . '</a></td>' . "\n";
	} else {
		$message .= '											<td class="aligncenter content-block" style="box-sizing: border-box; vertical-align: top; color: #999; text-align: center; margin: 0; padding: 0 0 20px;" align="center" valign="top"><a href="' . esc_url( site_url() ) . '" target="_blank" rel="noopener noreferrer">' . esc_html( site_url() ) . '</a></td>' . "\n";
	}

	$message .= '										</tr>' . "\n";
	$message .= '									</table>' . "\n";
	$message .= '								</div>' . "\n";
	$message .= '							</td>' . "\n";
	$message .= '							<td style="box-sizing: border-box; vertical-align: top; margin: 0;" valign="top"></td>' . "\n";
	$message .= '						</tr>' . "\n";
	$message .= '					</table>' . "\n";
	$message .= '				</div>' . "\n";
	$message .= '			</td>' . "\n";
	$message .= '		</tr>' . "\n";
	$message .= '	</table>' . "\n";
	$message .= '</body>' . "\n";
	$message .= '</html>';

	// Return the prepared HTML email message.
	return $message;
}

/**
 * Executes the vulnerability notification process for a WordPress site.
 *
 * This function checks for vulnerabilities in the WordPress core, plugins, themes, PHP environment, and web server components.
 * It generates an HTML email report detailing any vulnerabilities found. If the function is called with
 * the $forced parameter set to true, it will send an email even if no vulnerabilities are found, which is useful for testing purposes.
 *
 * @since 2.0.0
 *
 * @param bool $forced Optional. If set to true, forces the sending of a notification email regardless of whether vulnerabilities are found. Default false.
 * @return string|false Returns the email content if the email was successfully sent, false otherwise.
 */
function wpvulnerability_execute_notification( $forced = false ) {
				$email_content                      = '';
				$wpvulnerability_settings           = is_multisite() ? get_site_option( 'wpvulnerability-config' ) : get_option( 'wpvulnerability-config' );
				$wpvulnerability_settings           = is_array( $wpvulnerability_settings ) ? $wpvulnerability_settings : array();
				$wpvulnerability_settings['notify'] = wpvulnerability_normalize_notify_settings( isset( $wpvulnerability_settings['notify'] ) ? $wpvulnerability_settings['notify'] : array() );

		// Check if forced sending is not required and no channels configured.
	if ( ! $forced ) {
			$available = false;

		if ( wpvulnerability_is_yes( $wpvulnerability_settings['notify']['email'] ) && ! empty( $wpvulnerability_settings['emails'] ) ) {
						$available = true;
		}

		if ( wpvulnerability_is_yes( $wpvulnerability_settings['notify']['slack'] ) && ! empty( $wpvulnerability_settings['slack_webhook'] ) ) {
								$available = true;
		}

		if ( wpvulnerability_is_yes( $wpvulnerability_settings['notify']['teams'] ) && ! empty( $wpvulnerability_settings['teams_webhook'] ) ) {
								$available = true;
		}

		if ( ! $available || empty( $wpvulnerability_settings['period'] ) ) {
				return false;
		}
	}

	// Generate HTML for core, plugins, and themes vulnerabilities.
	$html_core = wpvulnerability_analyze_filter( 'core' ) && ( is_multisite() ? json_decode( get_site_option( 'wpvulnerability-core-vulnerable' ) ) : json_decode( get_option( 'wpvulnerability-core-vulnerable' ) ) ) ? wpvulnerability_html_core() : null;

	$html_plugins = wpvulnerability_analyze_filter( 'plugins' ) && ( is_multisite() ? json_decode( get_site_option( 'wpvulnerability-plugins-vulnerable' ) ) : json_decode( get_option( 'wpvulnerability-plugins-vulnerable' ) ) ) ? wpvulnerability_html_plugins() : null;

	$html_themes = wpvulnerability_analyze_filter( 'themes' ) && ( is_multisite() ? json_decode( get_site_option( 'wpvulnerability-themes-vulnerable' ) ) : json_decode( get_option( 'wpvulnerability-themes-vulnerable' ) ) ) ? wpvulnerability_html_themes() : null;

	// Generate HTML for PHP, Apache, Nginx, MariaDB, MySQL... vulnerabilities.
	$html_php = wpvulnerability_analyze_filter( 'php' ) && ( is_multisite() ? json_decode( get_site_option( 'wpvulnerability-php-vulnerable' ) ) : json_decode( get_option( 'wpvulnerability-php-vulnerable' ) ) ) ? wpvulnerability_html_software( 'php' ) : null;

	$html_apache = wpvulnerability_analyze_filter( 'apache' ) && ( is_multisite() ? json_decode( get_site_option( 'wpvulnerability-apache-vulnerable' ) ) : json_decode( get_option( 'wpvulnerability-apache-vulnerable' ) ) ) ? wpvulnerability_html_software( 'apache' ) : null;

	$html_nginx = wpvulnerability_analyze_filter( 'nginx' ) && ( is_multisite() ? json_decode( get_site_option( 'wpvulnerability-nginx-vulnerable' ) ) : json_decode( get_option( 'wpvulnerability-nginx-vulnerable' ) ) ) ? wpvulnerability_html_software( 'nginx' ) : null;

	$html_mariadb = wpvulnerability_analyze_filter( 'mariadb' ) && ( is_multisite() ? json_decode( get_site_option( 'wpvulnerability-mariadb-vulnerable' ) ) : json_decode( get_option( 'wpvulnerability-mariadb-vulnerable' ) ) ) ? wpvulnerability_html_software( 'mariadb' ) : null;

	$html_mysql = wpvulnerability_analyze_filter( 'mysql' ) && ( is_multisite() ? json_decode( get_site_option( 'wpvulnerability-mysql-vulnerable' ) ) : json_decode( get_option( 'wpvulnerability-mysql-vulnerable' ) ) ) ? wpvulnerability_html_software( 'mysql' ) : null;

	$html_imagemagick = wpvulnerability_analyze_filter( 'imagemagick' ) && ( is_multisite() ? json_decode( get_site_option( 'wpvulnerability-imagemagick-vulnerable' ) ) : json_decode( get_option( 'wpvulnerability-imagemagick-vulnerable' ) ) ) ? wpvulnerability_html_software( 'imagemagick' ) : null;

	$html_curl = wpvulnerability_analyze_filter( 'curl' ) && ( is_multisite() ? json_decode( get_site_option( 'wpvulnerability-curl-vulnerable' ) ) : json_decode( get_option( 'wpvulnerability-curl-vulnerable' ) ) ) ? wpvulnerability_html_software( 'curl' ) : null;

	$html_memcached = wpvulnerability_analyze_filter( 'memcached' ) && ( is_multisite() ? json_decode( get_site_option( 'wpvulnerability-memcached-vulnerable' ) ) : json_decode( get_option( 'wpvulnerability-memcached-vulnerable' ) ) ) ? wpvulnerability_html_software( 'memcached' ) : null;

	$html_redis = wpvulnerability_analyze_filter( 'redis' ) && ( is_multisite() ? json_decode( get_site_option( 'wpvulnerability-redis-vulnerable' ) ) : json_decode( get_option( 'wpvulnerability-redis-vulnerable' ) ) ) ? wpvulnerability_html_software( 'redis' ) : null;

	$html_sqlite = wpvulnerability_analyze_filter( 'sqlite' ) && ( is_multisite() ? json_decode( get_site_option( 'wpvulnerability-sqlite-vulnerable' ) ) : json_decode( get_option( 'wpvulnerability-sqlite-vulnerable' ) ) ) ? wpvulnerability_html_software( 'sqlite' ) : null;

	$all_empty = ( empty( $html_core ) && empty( $html_plugins ) && empty( $html_themes ) && empty( $html_php ) && empty( $html_apache ) && empty( $html_nginx ) && empty( $html_mariadb ) && empty( $html_mysql ) && empty( $html_imagemagick ) && empty( $html_curl ) && empty( $html_memcached ) && empty( $html_redis ) && empty( $html_sqlite ) );

	// If forced email sending is not enabled and no vulnerabilities were found, exit the function.
	if ( ! $forced && $all_empty ) {
		return false;
	} elseif ( $forced && $all_empty ) {
		$email_content .= '<h2>' . esc_html__( 'No vulnerabilities found', 'wpvulnerability' ) . '</h2>';
		$email_content .= '<p>' . esc_html__( 'This is likely a test. The site does not have vulnerabilities.', 'wpvulnerability' ) . '</p>';
	}

	// Append core vulnerabilities HTML to the email content.
	if ( ! empty( $html_core ) ) {
		$email_content .= '<h2>' . esc_html__( 'Core vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_core;
	}

	// Append plugins vulnerabilities HTML to the email content.
	if ( ! empty( $html_plugins ) ) {
		$email_content .= '<h2>' . esc_html__( 'Plugins vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_plugins;
	}

	// Append themes vulnerabilities HTML to the email content.
	if ( ! empty( $html_themes ) ) {
		$email_content .= '<h2>' . esc_html__( 'Themes vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_themes;
	}

	// Append PHP vulnerabilities HTML to the email content.
	if ( ! empty( $html_php ) ) {
		$email_content .= '<h2>' . esc_html__( 'PHP vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_php;
	}

	// Append Apache vulnerabilities HTML to the email content.
	if ( ! empty( $html_apache ) ) {
		$email_content .= '<h2>' . esc_html__( 'Apache HTTPD vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_apache;
	}

	// Append Nginx vulnerabilities HTML to the email content.
	if ( ! empty( $html_nginx ) ) {
		$email_content .= '<h2>' . esc_html__( 'Nginx vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_nginx;
	}

	// Append MariaDB vulnerabilities HTML to the email content.
	if ( ! empty( $html_mariadb ) ) {
		$email_content .= '<h2>' . esc_html__( 'MariaDB vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_mariadb;
	}

	// Append MySQL vulnerabilities HTML to the email content.
	if ( ! empty( $html_mysql ) ) {
		$email_content .= '<h2>' . esc_html__( 'MySQL vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_mysql;
	}

	// Append ImageMagick vulnerabilities HTML to the email content.
	if ( ! empty( $html_imagemagick ) ) {
		$email_content .= '<h2>' . esc_html__( 'ImageMagick vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_imagemagick;
	}

	// Append curl vulnerabilities HTML to the email content.
	if ( ! empty( $html_curl ) ) {
		$email_content .= '<h2>' . esc_html__( 'curl vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_curl;
	}

	// Append memcached vulnerabilities HTML to the email content.
	if ( ! empty( $html_memcached ) ) {
		$email_content .= '<h2>' . esc_html__( 'memcached vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_memcached;
	}

	// Append Redis vulnerabilities HTML to the email content.
	if ( ! empty( $html_redis ) ) {
		$email_content .= '<h2>' . esc_html__( 'Redis vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_redis;
	}

	// Append SQLite vulnerabilities HTML to the email content.
	if ( ! empty( $html_sqlite ) ) {
		$email_content .= '<h2>' . esc_html__( 'SQLite vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_sqlite;
	}

	// Get the site name.
	$admin_site = is_multisite() ? get_site_option( 'network_name_option' ) : get_bloginfo( 'name' );

	// Get the admin email.
	$admin_email = is_multisite() ? get_site_option( 'admin_email' ) : get_bloginfo( 'admin_email' );
	$from_email  = $admin_email;

	// Check if WPVULNERABILITY_MAIL is defined and valid, and use it if available.
	if ( defined( 'WPVULNERABILITY_MAIL' ) ) {
		$wpvulnerability_sender_email = sanitize_email( trim( WPVULNERABILITY_MAIL ) );
		if ( is_email( $wpvulnerability_sender_email ) ) {
			$from_email = $wpvulnerability_sender_email;
		}
		unset( $wpvulnerability_sender_email );
	}

	// Prepare email subject and content.
	$email_subject = sprintf(
		// translators: Site name.
		__( 'Vulnerability found: %s', 'wpvulnerability' ),
		$admin_site
	);

		$email_prepared = wpvulnerability_email_prepare( esc_html__( 'Vulnerability found', 'wpvulnerability' ), $email_content );

		// Prepare email headers.
		$email_headers   = array();
		$email_headers[] = 'From: WPVulnerability <' . $from_email . '>';
		$email_headers[] = 'Content-Type: text/html; charset=UTF-8';

		$text_message = $email_subject . "\n\n" . wp_strip_all_tags( $email_content );

	if ( $forced && wpvulnerability_is_yes( $wpvulnerability_settings['notify']['email'] ) && empty( $wpvulnerability_settings['emails'] ) ) {
						// Determine the recipient email.
						$wpvulnerability_settings['emails'][] = $admin_email;
	}

		$sent = false;

	if ( wpvulnerability_is_yes( $wpvulnerability_settings['notify']['email'] ) && ! empty( $wpvulnerability_settings['emails'] ) ) {
						$wpmail = wp_mail( $wpvulnerability_settings['emails'], $email_subject, $email_prepared, $email_headers );
						$sent   = $sent || $wpmail;
	}

	if ( wpvulnerability_is_yes( $wpvulnerability_settings['notify']['slack'] ) && ! empty( $wpvulnerability_settings['slack_webhook'] ) ) {
						wp_remote_post(
							$wpvulnerability_settings['slack_webhook'],
							array(
								'body'    => wp_json_encode( array( 'text' => $text_message ) ),
								'headers' => array( 'Content-Type' => 'application/json' ),
							)
						);
						$sent = true;
	}

	if ( wpvulnerability_is_yes( $wpvulnerability_settings['notify']['teams'] ) && ! empty( $wpvulnerability_settings['teams_webhook'] ) ) {
						wp_remote_post(
							$wpvulnerability_settings['teams_webhook'],
							array(
								'body'    => wp_json_encode( array( 'text' => $text_message ) ),
								'headers' => array( 'Content-Type' => 'application/json' ),
							)
						);
						$sent = true;
	}

		return $sent;
}
