<?php
/**
 * CLI functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Registers a WP-CLI command for WPVulnerability plugin.
 *
 * @since 2.0.0
 *
 * @return void
 */
if ( defined( 'WP_CLI' ) && WP_CLI ) {

	/**
	 * Core section in WP-CLI command
	 *
	 * @since 2.0.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output, either 'table' or 'json'.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_core( $args, $format ) {
		// Validate the format.
		switch ( $format ) {
			case 'table':
				$format = 'table';
				break;
			case 'json':
				$format = 'json';
				break;
			default:
				WP_CLI::error( "'$format' is not a valid format.\nAvailable formats: table, json" );
				break;
		}

		$core_vulnerabilities = array();
		if ( wpvulnerability_analyze_filter( 'core' ) ) {
			// Get core vulnerabilities.
			$core_vulnerabilities = wpvulnerability_core_get_vulnerabilities();
		}

		$core_complete   = array();
		$vulnerabilities = array();

		if ( $core_vulnerabilities && is_array( $core_vulnerabilities ) ) {
			// Loop through each core vulnerability.
			foreach ( $core_vulnerabilities as $vulnerability ) {
				$core_complete_temp = array();

				// Process version.
				$core_complete_temp['version'] = trim( html_entity_decode( wp_kses( (string) $vulnerability['name'], 'strip' ) ) );

				// Process severity.
				$core_complete_temp['severity'] = null;
				if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
					$core_complete_temp['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
				}

				// Process CWE details.
				$core_complete_temp['cwe'] = array();
				if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {
					foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
						$core_complete_temp['cwe'][] = array(
							'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) ) ),
							'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['description'], 'strip' ) ) ),
						);
					}
				}

				// Process CVSS score.
				$core_complete_temp['score'] = null;
				if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
					$core_complete_temp['score'] = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
				}

				// Process sources.
				$core_complete_temp['source'] = array();
				if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {
					foreach ( $vulnerability['source'] as $vulnerability_source ) {
						$core_complete_temp['source'][] = array(
							'name' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['name'], 'strip' ) ) ),
							'link' => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
						);
					}
				}

				$core_complete[] = $core_complete_temp;
				unset( $core_complete_temp );
			}
		}

		// Format output based on the selected format.
		if ( 'table' === $format ) {
			foreach ( $core_complete as $c_vuln ) {
				$v_version  = $c_vuln['version'];
				$v_score    = $c_vuln['score'];
				$v_severity = $c_vuln['severity'];

				// Compile CWE descriptions.
				$v_description_array = array();
				foreach ( $c_vuln['cwe'] as $c_cwe ) {
					if ( isset( $c_cwe['name'] ) ) {
						$v_description_array[] = $c_cwe['name'];
					}
				}
				$v_description = trim( implode( ' + ', $v_description_array ) );

				// Add to vulnerabilities array for table output.
				array_push(
					$vulnerabilities,
					array(
						'version'     => $v_version,
						'score'       => $v_score,
						'severity'    => $v_severity,
						'description' => $v_description,
					)
				);
			}

			// Format and output the vulnerabilities in a table.
			WP_CLI\Utils\format_items(
				'table',
				$vulnerabilities,
				array( 'version', 'score', 'severity', 'description' )
			);

		} elseif ( 'json' === $format ) {
			// Format and output the vulnerabilities in JSON.
			echo wp_json_encode( $core_complete );
		}
	}

	/**
	 * Plugin section in WP-CLI command
	 *
	 * This function retrieves and displays plugin vulnerabilities
	 * based on the specified format (table or JSON).
	 *
	 * @since 2.0.0
	 *
	 * @param array  $args   The arguments passed from the command line.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_plugins( $args, $format ) {
		// Validate the format.
		switch ( $format ) {
			case 'table':
				$format = 'table';
				break;
			case 'json':
				$format = 'json';
				break;
			default:
				WP_CLI::error( "'$format' is not a valid format.\nAvailable formats: table, json" );
				break;
		}

		$plugin_vulnerabilities = array();
		if ( wpvulnerability_analyze_filter( 'plugins' ) ) {
			// Get plugin vulnerabilities.
			$plugin_vulnerabilities = wpvulnerability_plugin_get_vulnerabilities();
		}

		$plugins_complete = array();
		$vulnerabilities  = array();

		// Loop through each plugin vulnerability.
		foreach ( $plugin_vulnerabilities as $plugin ) {
			if ( 1 === $plugin['vulnerable'] ) {
				$plugins_complete_temp                 = array();
				$plugins_complete_temp_vulnerabilities = array();

				// Process plugin name and slug.
				$plugins_complete_temp['name'] = trim( html_entity_decode( wp_kses( (string) $plugin['Name'], 'strip' ) ) );
				$plugins_complete_temp['slug'] = trim( html_entity_decode( wp_kses( (string) $plugin['slug'], 'strip' ) ) );

				// Prepare the vulnerabilities array for table format output.
				foreach ( $plugin['vulnerabilities'] as $vulnerability ) {
					$plugins_complete_temp_vulnerabilities['severity'] = null;
					if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
						$plugins_complete_temp_vulnerabilities['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
					}

					// Process vulnerability details.
					$plugins_complete_temp_vulnerabilities['version'] = trim( html_entity_decode( wp_kses( (string) $vulnerability['versions'], 'strip' ) ) );
					$plugins_complete_temp_vulnerabilities['name']    = trim( html_entity_decode( wp_kses( (string) $vulnerability['name'], 'strip' ) ) );
					$plugins_complete_temp_vulnerabilities['closed']  = (int) $vulnerability['closed'];
					$plugins_complete_temp_vulnerabilities['unfixed'] = (int) $vulnerability['unfixed'];

					// Process CWE details.
					$plugins_complete_temp_vulnerabilities['cwe'] = array();
					if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {
						foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
							$plugins_complete_temp_vulnerabilities['cwe'][] = array(
								'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) ) ),
								'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['description'], 'strip' ) ) ),
							);
						}
					}

					// Process CVSS score.
					$plugins_complete_temp_vulnerabilities['score'] = null;
					if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
						$plugins_complete_temp_vulnerabilities['score'] = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
					}

					// Process sources.
					$plugins_complete_temp_vulnerabilities['source'] = array();
					if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {
						foreach ( $vulnerability['source'] as $vulnerability_source ) {
							$plugins_complete_temp_vulnerabilities['source'][] = array(
								'name' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['name'], 'strip' ) ) ),
								'link' => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
							);
						}
					}

					$plugins_complete_temp['vulnerabilities'][] = $plugins_complete_temp_vulnerabilities;
					unset( $plugins_complete_temp_vulnerabilities, $vulnerability );
				}

				$plugins_complete[] = $plugins_complete_temp;
				unset( $plugins_complete_temp );
			}
			unset( $plugin );
		}

		// Format output based on the selected format.
		if ( 'table' === $format ) {
			foreach ( $plugins_complete as $p_vuln ) {
				$v_name = $p_vuln['slug'];

				foreach ( $p_vuln['vulnerabilities'] as $p_vul ) {
					$v_version = $p_vul['version'];

					// Determine if vulnerability is fixed.
					switch ( $p_vul['unfixed'] ) {
						default:
						case 0:
							$v_fixed = 'yes';
							break;
						case 1:
							$v_fixed = 'no';
							break;
					}

					// Determine if plugin is closed.
					switch ( $p_vul['closed'] ) {
						default:
						case 0:
							$v_closed = 'no';
							break;
						case 1:
							$v_closed = 'yes';
							break;
					}
					$v_score    = $p_vul['score'];
					$v_severity = $p_vul['severity'];

					// Compile CWE descriptions.
					$v_description_array = array();
					foreach ( $p_vul['cwe'] as $p_cwe ) {
						if ( isset( $p_cwe['name'] ) ) {
							$v_description_array[] = $p_cwe['name'];
						}
					}
					$v_description = trim( implode( ' + ', $v_description_array ) );

					// Add to vulnerabilities array for table output.
					array_push(
						$vulnerabilities,
						array(
							'name'        => $v_name,
							'version'     => $v_version,
							'fixed'       => $v_fixed,
							'closed'      => $v_closed,
							'score'       => $v_score,
							'severity'    => $v_severity,
							'description' => $v_description,
						)
					);
				}
			}

			// Format and output the vulnerabilities in a table.
			WP_CLI\Utils\format_items(
				'table',
				$vulnerabilities,
				array( 'name', 'version', 'fixed', 'closed', 'score', 'severity', 'description' )
			);

		} elseif ( 'json' === $format ) {
			// Format and output the vulnerabilities in JSON.
			echo wp_json_encode( $plugins_complete );
		}
	}


	/**
	 * Theme section in WP-CLI command
	 *
	 * This function retrieves and displays theme vulnerabilities
	 * based on the specified format (table or JSON).
	 *
	 * @since 2.0.0
	 *
	 * @param array  $args   The arguments passed from the command line.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_themes( $args, $format ) {
		// Validate the format.
		switch ( $format ) {
			case 'table':
				$format = 'table';
				break;
			case 'json':
				$format = 'json';
				break;
			default:
				WP_CLI::error( "'$format' is not a valid format.\nAvailable formats: table, json" );
				break;
		}

		$theme_vulnerabilities = array();
		if ( wpvulnerability_analyze_filter( 'themes' ) ) {
			// Get theme vulnerabilities.
			$theme_vulnerabilities = wpvulnerability_theme_get_vulnerabilities();
		}

		$themes_complete = array();
		$vulnerabilities = array();

		// Loop through each theme vulnerability.
		foreach ( $theme_vulnerabilities as $theme ) {
			if ( 1 === $theme['wpvulnerability']['vulnerable'] ) {
				$themes_complete_temp                 = array();
				$themes_complete_temp_vulnerabilities = array();

				// Process theme name and slug.
				$themes_complete_temp['name'] = trim( html_entity_decode( wp_kses( (string) $theme['wpvulnerability']['name'], 'strip' ) ) );
				$themes_complete_temp['slug'] = trim( html_entity_decode( wp_kses( (string) $theme['wpvulnerability']['slug'], 'strip' ) ) );

				// Prepare the vulnerabilities array for table format output.
				foreach ( $theme['wpvulnerability']['vulnerabilities'] as $vulnerability ) {
					$themes_complete_temp_vulnerabilities['severity'] = null;
					if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
						$themes_complete_temp_vulnerabilities['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
					}

					// Process vulnerability details.
					$themes_complete_temp_vulnerabilities['version'] = trim( html_entity_decode( wp_kses( (string) $vulnerability['versions'], 'strip' ) ) );
					$themes_complete_temp_vulnerabilities['name']    = trim( html_entity_decode( wp_kses( (string) $vulnerability['name'], 'strip' ) ) );
					$themes_complete_temp_vulnerabilities['closed']  = (int) $vulnerability['closed'];
					$themes_complete_temp_vulnerabilities['unfixed'] = (int) $vulnerability['unfixed'];

					// Process CWE details.
					$themes_complete_temp_vulnerabilities['cwe'] = array();
					if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {
						foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
							$themes_complete_temp_vulnerabilities['cwe'][] = array(
								'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) ) ),
								'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['description'], 'strip' ) ) ),
							);
						}
					}

					// Process CVSS score.
					$themes_complete_temp_vulnerabilities['score'] = null;
					if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
						$themes_complete_temp_vulnerabilities['score'] = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
					}

					// Process sources.
					$themes_complete_temp_vulnerabilities['source'] = array();
					if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {
						foreach ( $vulnerability['source'] as $vulnerability_source ) {
							$themes_complete_temp_vulnerabilities['source'][] = array(
								'name' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['name'], 'strip' ) ) ),
								'link' => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
							);
						}
					}

					$themes_complete_temp['vulnerabilities'][] = $themes_complete_temp_vulnerabilities;
					unset( $themes_complete_temp_vulnerabilities, $vulnerability );
				}

				$themes_complete[] = $themes_complete_temp;
				unset( $themes_complete_temp );
			}
			unset( $theme );
		}

		// Format output based on the selected format.
		if ( 'table' === $format ) {
			foreach ( $themes_complete as $t_vuln ) {
				$v_name = $t_vuln['slug'];

				foreach ( $t_vuln['vulnerabilities'] as $t_vul ) {
					$v_version = $t_vul['version'];

					// Determine if vulnerability is fixed.
					switch ( $t_vul['unfixed'] ) {
						default:
						case 0:
							$v_fixed = 'yes';
							break;
						case 1:
							$v_fixed = 'no';
							break;
					}

					// Determine if theme is closed.
					switch ( $t_vul['closed'] ) {
						default:
						case 0:
							$v_closed = 'no';
							break;
						case 1:
							$v_closed = 'yes';
							break;
					}
					$v_score    = $t_vul['score'];
					$v_severity = $t_vul['severity'];

					// Compile CWE descriptions.
					$v_description_array = array();
					foreach ( $t_vul['cwe'] as $t_cwe ) {
						if ( isset( $t_cwe['name'] ) ) {
							$v_description_array[] = $t_cwe['name'];
						}
					}
					$v_description = trim( implode( ' + ', $v_description_array ) );

					// Add to vulnerabilities array for table output.
					array_push(
						$vulnerabilities,
						array(
							'name'        => $v_name,
							'version'     => $v_version,
							'fixed'       => $v_fixed,
							'closed'      => $v_closed,
							'score'       => $v_score,
							'severity'    => $v_severity,
							'description' => $v_description,
						)
					);
				}
			}

			// Format and output the vulnerabilities in a table.
			WP_CLI\Utils\format_items(
				'table',
				$vulnerabilities,
				array( 'name', 'version', 'fixed', 'closed', 'score', 'severity', 'description' )
			);

		} elseif ( 'json' === $format ) {
			// Format and output the vulnerabilities in JSON.
			echo wp_json_encode( $themes_complete );
		}
	}


	/**
	 * Handles vulnerabilities for specified software in WP-CLI command.
	 *
	 * This function retrieves and displays vulnerabilities based on the specified format (table or JSON).
	 *
	 * @since 3.5.0
	 *
	 * @param string $software The software type (php, apache, nginx, mariadb, mysql).
	 * @param string $format   The format for the output, either 'table' or 'json'.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_software( $software, $format ) {
		// Validate the format.
		switch ( $format ) {
			case 'table':
				$format = 'table';
				break;
			case 'json':
				$format = 'json';
				break;
			default:
				WP_CLI::error( "'$format' is not a valid format.\nAvailable formats: table, json" );
				break;
		}

		$vulnerabilities = array();

		switch ( $software ) {
			case 'php':
			case 'apache':
			case 'nginx':
			case 'mariadb':
			case 'mysql':
			case 'imagemagick':
			case 'curl':
			case 'memcached':
			case 'redis':
			case 'sqlite':
				$software_vulnerabilities = wpvulnerability_get_vulnerabilities( $software, wpvulnerability_get_software_version( $software ) );
				break;
			default:
				$software_vulnerabilities = null;
		}

		$software_complete = array();

		if ( isset( $software_vulnerabilities ) && is_array( $software_vulnerabilities ) ) {
			// Loop through each vulnerability.
			foreach ( $software_vulnerabilities as $item ) {
				$complete_temp = array();

				$complete_temp['version']  = trim( html_entity_decode( wp_kses( (string) $item['version'], 'strip' ) ) );
				$complete_temp['affected'] = trim( html_entity_decode( wp_kses( (string) $item['versions'], 'strip' ) ) );
				$complete_temp['unfixed']  = (int) $item['unfixed'];

				// Process vulnerability sources.
				$complete_temp['source'] = array();
				if ( isset( $item['source'] ) && count( $item['source'] ) ) {
					foreach ( $item['source'] as $source ) {
						$complete_temp['source'][] = array(
							'name'        => trim( html_entity_decode( wp_kses( (string) $source['id'], 'strip' ) ) ),
							'description' => trim( html_entity_decode( wp_kses( (string) $source['description'], 'strip' ) ) ),
							'link'        => esc_url_raw( (string) $source['link'], 'strip' ),
						);
					}
				}

				$software_complete[] = $complete_temp;
			}
		}

		// Format output based on the selected format.
		if ( 'table' === $format ) {
			$vulnerabilities = array();

			foreach ( $software_complete as $vuln ) {
				$v_fixed = $vuln['unfixed'] ? 'no' : 'yes';

				// Compile source descriptions.
				$v_description_array = array();
				foreach ( $vuln['source'] as $source ) {
					$v_description_array[] = $source['name'] . ': ' . $source['description'];
				}
				$v_description = trim( implode( ' + ', $v_description_array ) );

				// Add to vulnerabilities array for table output.
				array_push(
					$vulnerabilities,
					array(
						'version'     => $vuln['version'],
						'affected'    => $vuln['affected'],
						'fixed'       => $v_fixed,
						'description' => $v_description,
					)
				);
			}

			// Format and output the vulnerabilities in a table.
			WP_CLI\Utils\format_items(
				'table',
				$vulnerabilities,
				array( 'version', 'affected', 'fixed', 'description' )
			);
		} elseif ( 'json' === $format ) {
			// Format and output the vulnerabilities in JSON.
			echo wp_json_encode( $software_complete );
		}
	}

	/**
	 * PHP section in WP-CLI command.
	 *
	 * @since 3.5.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_php( $args, $format ) {
		wpvulnerability_cli_software( 'php', $format );
	}

	/**
	 * Apache HTTPD section in WP-CLI command.
	 *
	 * @since 3.5.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_apache( $args, $format ) {
		wpvulnerability_cli_software( 'apache', $format );
	}

	/**
	 * Nginx section in WP-CLI command.
	 *
	 * @since 3.5.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_nginx( $args, $format ) {
		wpvulnerability_cli_software( 'nginx', $format );
	}

	/**
	 * Handles the MariaDB section in the WP-CLI command.
	 *
	 * @since 3.5.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_mariadb( $args, $format ) {
		wpvulnerability_cli_software( 'mariadb', $format );
	}

	/**
	 * Handles the MySQL section in the WP-CLI command.
	 *
	 * @since 3.5.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_mysql( $args, $format ) {
		wpvulnerability_cli_software( 'mysql', $format );
	}

	/**
	 * Handles the ImageMagick section in the WP-CLI command.
	 *
	 * @since 3.5.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_imagemagick( $args, $format ) {
		wpvulnerability_cli_software( 'imagemagick', $format );
	}

	/**
	 * Handles the curl section in the WP-CLI command.
	 *
	 * @since 3.5.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_curl( $args, $format ) {
		wpvulnerability_cli_software( 'curl', $format );
	}

	/**
	 * Handles the memcached section in the WP-CLI command.
	 *
	 * @since 3.5.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_memcached( $args, $format ) {
		wpvulnerability_cli_software( 'memcached', $format );
	}

	/**
	 * Handles the Redis section in the WP-CLI command.
	 *
	 * @since 3.5.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_redis( $args, $format ) {
		wpvulnerability_cli_software( 'redis', $format );
	}

	/**
	 * Handles the SQLite section in the WP-CLI command.
	 *
	 * @since 3.5.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_sqlite( $args, $format ) {
		wpvulnerability_cli_software( 'sqlite', $format );
	}

	/**
	 * Dispatches the WPVulnerability CLI command to show the list of detected vulnerabilities on the site.
	 *
	 * This function selects and executes the appropriate function based on the provided subcommand.
	 * It supports different output formats, such as 'table' and 'json', to display the vulnerabilities.
	 *
	 * @since 2.0.0
	 *
	 * @param array $args       The subcommand to execute.
	 *                          Accepted values: 'core', 'plugins', 'themes', 'php', 'apache', 'nginx', 'mariadb', 'mysql'.
	 * @param array $assoc_args Associative arguments passed from the command line.
	 *                          'format' (optional) - The format for the output. Defaults to 'table'.
	 *                          Accepted values: 'table', 'json'.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_command( $args, $assoc_args ) {
		$subcommand = $args[0];
		$format     = isset( $assoc_args['format'] ) ? $assoc_args['format'] : 'table';

		// Select the correct function to execute based on the subcommand.
		switch ( $subcommand ) {
			case 'core':
				wpvulnerability_cli_core( $args, $format );
				break;
			case 'plugins':
				wpvulnerability_cli_plugins( $args, $format );
				break;
			case 'themes':
				wpvulnerability_cli_themes( $args, $format );
				break;
			case 'php':
				wpvulnerability_cli_php( $args, $format );
				break;
			case 'apache':
				wpvulnerability_cli_apache( $args, $format );
				break;
			case 'nginx':
				wpvulnerability_cli_nginx( $args, $format );
				break;
			case 'mariadb':
				wpvulnerability_cli_mariadb( $args, $format );
				break;
			case 'mysql':
				wpvulnerability_cli_mysql( $args, $format );
				break;
			case 'imagemagick':
				wpvulnerability_cli_imagemagick( $args, $format );
				break;
			case 'curl':
				wpvulnerability_cli_curl( $args, $format );
				break;
			case 'memcached':
				wpvulnerability_cli_memcached( $args, $format );
				break;
			case 'redis':
				wpvulnerability_cli_redis( $args, $format );
				break;
			case 'sqlite':
				wpvulnerability_cli_sqlite( $args, $format );
				break;
			default:
				// Display an error message for an invalid subcommand.
				WP_CLI::error( "'$subcommand' is not a valid subcommand of 'wpvulnerability'.\nAvailable subcommands: core, plugins, themes, php, apache, nginx, mariadb, mysql, imagemagick, curl, memcached, redis, sqlite" );
				break;
		}
	}

	/**
	 * Registers a WP-CLI command to show the list of vulnerabilities detected on your site.
	 *
	 * EXAMPLES:
	 *
	 *  - wp wpvulnerability core
	 *  - wp wpvulnerability plugins
	 *  - wp wpvulnerability themes
	 *  - wp wpvulnerability php
	 *  - wp wpvulnerability apache
	 *  - wp wpvulnerability nginx
	 *  - wp wpvulnerability mariadb
	 *  - wp wpvulnerability mysql
	 *  - wp wpvulnerability imagemagick
	 *  - wp wpvulnerability curl
	 *  - wp wpvulnerability memcached
	 *  - wp wpvulnerability redis
	 *  - wp wpvulnerability sqlite
	 *
	 * @param object $args Arguments passed from the command line.
	 *
	 * @return void
	 */
	WP_CLI::add_command(
		'wpvulnerability',
		'wpvulnerability_cli_command',
		array(
			'shortdesc' => 'Show the list of vulnerabilities detected on your site.',
			'synopsis'  => array(
				array(
					'type'        => 'positional',
					'name'        => 'subcommand',
					'description' => 'Type of vulnerability [ core | plugins | themes | php | apache | nginx | mariadb | mysql | imagemagick | curl ].',
					'optional'    => false,
				),
				array(
					'type'        => 'assoc',
					'name'        => 'format',
					'description' => 'Format for the output [ table | json ].',
					'optional'    => true,
					'default'     => 'table',
				),
			),
			'when'      => 'after_wp_load',
			'longdesc'  => "EXAMPLES:\n\n - wp wpvulnerability core\n - wp wpvulnerability plugins\n - wp wpvulnerability themes\n - wp wpvulnerability php\n - wp wpvulnerability apache\n - wp wpvulnerability nginx\n - wp wpvulnerability mariadb\n - wp wpvulnerability mysql\n - wp wpvulnerability imagemagick\n - wp wpvulnerability curl\n - wp wpvulnerability memcached\n - wp wpvulnerability redis\n - wp wpvulnerability sqlite",
		)
	);

}
