<?php
/**
 * Running functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Conditionally adds a settings link to the plugin row in the plugins list for a multisite network or single site installation.
 *
 * In a multisite environment, it adds a network admin settings link that appears in the network admin plugin row.
 * For a single site installation, it adds a standard settings link that appears in the plugin row.
 *
 * @since 3.0.0
 *
 * @param array $links The links that appear in the plugin row.
 *
 * @return array The modified array of links.
 */
if ( is_multisite() && is_network_admin() ) {

	/**
	 * Adds a network admin settings link to the plugin row in the network admin plugins list.
	 *
	 * @since 3.0.0
	 *
	 * @param array $links The links that appear in the network admin plugin row.
	 *
	 * @return array The modified array of links.
	 */
	function wpvulnerability_add_network_settings_link( $links ) {
		// Check if the user has the required capabilities to view the settings link.
		if ( wpvulnerability_capabilities() ) {
			// Add the network settings link to the plugin row.
			$links[] = '<a href="' . network_admin_url( 'settings.php?page=wpvulnerability-options' ) . '">' . __( 'Network Settings', 'wpvulnerability' ) . '</a>';
		}
		return $links;
	}
	add_filter( 'network_admin_plugin_action_links_' . WPVULNERABILITY_PLUGIN_BASE, 'wpvulnerability_add_network_settings_link' );

} elseif ( ! is_multisite() && is_admin() ) {

	/**
	 * Adds a standard settings link to the plugin row in the plugins list for a single site installation.
	 *
	 * @since 2.0.0
	 *
	 * @param array $links The links that appear in the plugin row.
	 *
	 * @return array The modified array of links.
	 */
	function wpvulnerability_add_settings_link( $links ) {
		// Check if the user has the required capabilities to view the settings link.
		if ( wpvulnerability_capabilities() ) {
			// Add the settings link to the plugin row.
			$links[] = '<a href="' . get_admin_url( null, 'options-general.php?page=wpvulnerability-options' ) . '">' . __( 'Settings', 'wpvulnerability' ) . '</a>';
		}
		return $links;
	}
	add_filter( 'plugin_action_links_' . WPVULNERABILITY_PLUGIN_BASE, 'wpvulnerability_add_settings_link' );
}

/**
 * Updates the plugin's vulnerability data.
 *
 * This function updates the vulnerability data for WordPress core, plugins, themes, PHP, Apache, nginx, MariaDB, and MySQL.
 * It ensures that the required functions are available by including the necessary files.
 * After updating the vulnerabilities, it flushes the WordPress cache.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_update_database_data() {

	// Ensure the core vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_core_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-core.php';
	}
	// Update core vulnerabilities.
	wpvulnerability_core_get_vulnerabilities_clean();

	// Ensure the plugin vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_plugin_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-plugins.php';
	}
	// Update plugin vulnerabilities.
	wpvulnerability_plugin_get_vulnerabilities_clean();

	// Ensure the theme vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_theme_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-themes.php';
	}
	// Update theme vulnerabilities.
	wpvulnerability_theme_get_vulnerabilities_clean();

	// Ensure the PHP vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_php_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-php.php';
	}
	// Update PHP vulnerabilities.
	wpvulnerability_php_get_vulnerabilities_clean();

	// Ensure the Apache vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_apache_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-apache.php';
	}
	// Update Apache vulnerabilities.
	wpvulnerability_apache_get_vulnerabilities_clean();

	// Ensure the nginx vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_nginx_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-nginx.php';
	}
	// Update nginx vulnerabilities.
	wpvulnerability_nginx_get_vulnerabilities_clean();

	// Ensure the MariaDB vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_mariadb_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-mariadb.php';
	}
	// Update MariaDB vulnerabilities.
	wpvulnerability_mariadb_get_vulnerabilities_clean();

	// Ensure the MySQL vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_mysql_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-mysql.php';
	}
	// Update MySQL vulnerabilities.
	wpvulnerability_mysql_get_vulnerabilities_clean();

	wpvulnerability_statistics_get();

	// Clean the WordPress cache.
	wp_cache_flush();
}

/**
 * Updates the plugin's vulnerability data if the cache has expired.
 *
 * This function checks if the cached vulnerability data for various components (core, plugins, themes, PHP, Apache, nginx, MariaDB, MySQL) has expired and updates it accordingly.
 * It ensures that the required functions are available by including the necessary files.
 * The function handles both multisite and single site installations.
 *
 * @since 3.0.0
 *
 * @return void
 */
function wpvulnerability_expired_database_data() {

	// Ensure the core vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_core_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-core.php';
	}
	// Ensure the plugin vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_plugin_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-plugins.php';
	}
	// Ensure the theme vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_theme_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-themes.php';
	}
	// Ensure the PHP vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_php_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-php.php';
	}
	// Ensure the Apache vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_apache_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-apache.php';
	}
	// Ensure the nginx vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_nginx_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-nginx.php';
	}
	// Ensure the MariaDB vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_mariadb_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-mariadb.php';
	}
	// Ensure the MySQL vulnerabilities function is available.
	if ( ! function_exists( 'wpvulnerability_mysql_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-mysql.php';
	}

	// Current time for cache expiration comparison.
	$cache_time = time();

	if ( is_multisite() ) {
		// Check and update core vulnerabilities if cache has expired.
		if ( json_decode( get_site_option( 'wpvulnerability-core-cache' ) ) < $cache_time ) {
			wpvulnerability_core_get_vulnerabilities_clean();
		}
		// Check and update plugin vulnerabilities if cache has expired.
		if ( json_decode( get_site_option( 'wpvulnerability-plugins-cache' ) ) < $cache_time ) {
			wpvulnerability_plugin_get_vulnerabilities_clean();
		}
		// Check and update theme vulnerabilities if cache has expired.
		if ( json_decode( get_site_option( 'wpvulnerability-themes-cache' ) ) < $cache_time ) {
			wpvulnerability_theme_get_vulnerabilities_clean();
		}
		// Check and update PHP vulnerabilities if cache has expired.
		if ( json_decode( get_site_option( 'wpvulnerability-php-cache' ) ) < $cache_time ) {
			wpvulnerability_php_get_vulnerabilities_clean();
		}
		// Check and update Apache vulnerabilities if cache has expired.
		if ( json_decode( get_site_option( 'wpvulnerability-apache-cache' ) ) < $cache_time ) {
			wpvulnerability_apache_get_vulnerabilities_clean();
		}
		// Check and update nginx vulnerabilities if cache has expired.
		if ( json_decode( get_site_option( 'wpvulnerability-nginx-cache' ) ) < $cache_time ) {
			wpvulnerability_nginx_get_vulnerabilities_clean();
		}
		// Check and update MariaDB vulnerabilities if cache has expired.
		if ( json_decode( get_site_option( 'wpvulnerability-mariadb-cache' ) ) < $cache_time ) {
			wpvulnerability_mariadb_get_vulnerabilities_clean();
		}
		// Check and update MySQL vulnerabilities if cache has expired.
		if ( json_decode( get_site_option( 'wpvulnerability-mysql-cache' ) ) < $cache_time ) {
			wpvulnerability_mysql_get_vulnerabilities_clean();
		}
	} else {
		// Check and update core vulnerabilities if cache has expired.
		if ( json_decode( get_option( 'wpvulnerability-core-cache' ) ) < $cache_time ) {
			wpvulnerability_core_get_vulnerabilities_clean();
		}
		// Check and update plugin vulnerabilities if cache has expired.
		if ( json_decode( get_option( 'wpvulnerability-plugins-cache' ) ) < $cache_time ) {
			wpvulnerability_plugin_get_vulnerabilities_clean();
		}
		// Check and update theme vulnerabilities if cache has expired.
		if ( json_decode( get_option( 'wpvulnerability-themes-cache' ) ) < $cache_time ) {
			wpvulnerability_theme_get_vulnerabilities_clean();
		}
		// Check and update PHP vulnerabilities if cache has expired.
		if ( json_decode( get_option( 'wpvulnerability-php-cache' ) ) < $cache_time ) {
			wpvulnerability_php_get_vulnerabilities_clean();
		}
		// Check and update Apache vulnerabilities if cache has expired.
		if ( json_decode( get_option( 'wpvulnerability-apache-cache' ) ) < $cache_time ) {
			wpvulnerability_apache_get_vulnerabilities_clean();
		}
		// Check and update nginx vulnerabilities if cache has expired.
		if ( json_decode( get_option( 'wpvulnerability-nginx-cache' ) ) < $cache_time ) {
			wpvulnerability_nginx_get_vulnerabilities_clean();
		}
		// Check and update MariaDB vulnerabilities if cache has expired.
		if ( json_decode( get_option( 'wpvulnerability-mariadb-cache' ) ) < $cache_time ) {
			wpvulnerability_mariadb_get_vulnerabilities_clean();
		}
		// Check and update MySQL vulnerabilities if cache has expired.
		if ( json_decode( get_option( 'wpvulnerability-mysql-cache' ) ) < $cache_time ) {
			wpvulnerability_mysql_get_vulnerabilities_clean();
		}
	}

	wpvulnerability_statistics_get();

	unset( $cache_time );
}

/**
 * Callback function for when the plugin is activated.
 * Adds plugin data options if they are not already created.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_activation() {

	if ( is_multisite() ) {

		// Add wpvulnerability-config option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-config' ) ) {
			$config = get_option( 'wpvulnerability-config' );

			// Check if emails and period are set in the existing config.
			if ( isset( $config['emails'] ) && $config['emails'] && isset( $config['period'] ) && $config['period'] ) {
				add_site_option(
					'wpvulnerability-config',
					array(
						'emails' => $config['emails'],
						'period' => $config['period'],
					)
				);
				unset( $config );
			} else {
				add_site_option(
					'wpvulnerability-config',
					array(
						'emails' => get_bloginfo( 'admin_email' ),
						'period' => 'weekly',
					)
				);
			}
		}

		// Add other site options if they do not exist.
		$options = array(
			'wpvulnerability-plugins'            => '',
			'wpvulnerability-plugins-cache'      => 0,
			'wpvulnerability-plugins-vulnerable' => 0,
			'wpvulnerability-plugins-data'       => '',
			'wpvulnerability-plugins-data-cache' => 0,
			'wpvulnerability-themes'             => '',
			'wpvulnerability-themes-cache'       => 0,
			'wpvulnerability-themes-vulnerable'  => 0,
			'wpvulnerability-core'               => '',
			'wpvulnerability-core-cache'         => 0,
			'wpvulnerability-core-vulnerable'    => 0,
			'wpvulnerability-php'                => '',
			'wpvulnerability-php-cache'          => 0,
			'wpvulnerability-php-vulnerable'     => 0,
			'wpvulnerability-apache'             => '',
			'wpvulnerability-apache-cache'       => 0,
			'wpvulnerability-apache-vulnerable'  => 0,
			'wpvulnerability-nginx'              => '',
			'wpvulnerability-nginx-cache'        => 0,
			'wpvulnerability-nginx-vulnerable'   => 0,
			'wpvulnerability-mariadb'            => '',
			'wpvulnerability-mariadb-cache'      => 0,
			'wpvulnerability-mariadb-vulnerable' => 0,
			'wpvulnerability-mysql'              => '',
			'wpvulnerability-mysql-cache'        => 0,
			'wpvulnerability-mysql-vulnerable'   => 0,
			'wpvulnerability-statistics'         => '',
			'wpvulnerability-statistics-cache'   => 0,
		);

		foreach ( $options as $key => $value ) {
			if ( ! get_site_option( $key ) ) {
				add_site_option( $key, $value );
			}
		}

		// Add wpvulnerability-analyze option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-analyze' ) ) {
			$analyze = get_option( 'wpvulnerability-analyze' );

			if ( isset( $analyze['core'] ) && isset( $analyze['plugins'] ) && isset( $analyze['themes'] ) && isset( $analyze['php'] ) && isset( $analyze['apache'] ) && isset( $analyze['nginx'] ) && isset( $analyze['mariadb'] ) && isset( $analyze['mysql'] ) ) {
				add_site_option(
					'wpvulnerability-analyze',
					array(
						'core'    => $analyze['core'],
						'plugins' => $analyze['plugins'],
						'themes'  => $analyze['themes'],
						'php'     => $analyze['php'],
						'apache'  => $analyze['apache'],
						'nginx'   => $analyze['nginx'],
						'mariadb' => $analyze['mariadb'],
						'mysql'   => $analyze['mysql'],
					)
				);
				unset( $analyze );
			} else {
				$default_wpvulnerability_analyze = array(
					'core'    => 0,
					'plugins' => 0,
					'themes'  => 0,
					'php'     => 0,
					'apache'  => 0,
					'nginx'   => 0,
					'mariadb' => 0,
					'mysql'   => 0,
				);
				$current_option                  = get_site_option( 'wpvulnerability-analyze' );
				if ( false === $current_option ) {
					add_site_option( 'wpvulnerability-analyze', $default_wpvulnerability_analyze );
				} else {
					$updated_option = $current_option;
					foreach ( $default_wpvulnerability_analyze as $key => $value ) {
						if ( ! array_key_exists( $key, $current_option ) ) {
							$updated_option[ $key ] = $value;
						}
					}
					if ( $updated_option !== $current_option ) {
						update_site_option( $option_name, $updated_option );
					}
				}
			}
		}
	} else {

		// Add wpvulnerability-config option if it does not exist.
		if ( ! get_option( 'wpvulnerability-config' ) ) {
			add_option(
				'wpvulnerability-config',
				array(
					'emails' => get_bloginfo( 'admin_email' ),
					'period' => 'weekly',
				)
			);
		}

		// Add other options if they do not exist.
		$options = array(
			'wpvulnerability-plugins'            => '',
			'wpvulnerability-plugins-cache'      => 0,
			'wpvulnerability-plugins-vulnerable' => 0,
			'wpvulnerability-plugins-data'       => '',
			'wpvulnerability-plugins-data-cache' => 0,
			'wpvulnerability-themes'             => '',
			'wpvulnerability-themes-cache'       => 0,
			'wpvulnerability-themes-vulnerable'  => 0,
			'wpvulnerability-core'               => '',
			'wpvulnerability-core-cache'         => 0,
			'wpvulnerability-core-vulnerable'    => 0,
			'wpvulnerability-php'                => '',
			'wpvulnerability-php-cache'          => 0,
			'wpvulnerability-php-vulnerable'     => 0,
			'wpvulnerability-apache'             => '',
			'wpvulnerability-apache-cache'       => 0,
			'wpvulnerability-apache-vulnerable'  => 0,
			'wpvulnerability-nginx'              => '',
			'wpvulnerability-nginx-cache'        => 0,
			'wpvulnerability-nginx-vulnerable'   => 0,
			'wpvulnerability-mariadb'            => '',
			'wpvulnerability-mariadb-cache'      => 0,
			'wpvulnerability-mariadb-vulnerable' => 0,
			'wpvulnerability-mysql'              => '',
			'wpvulnerability-mysql-cache'        => 0,
			'wpvulnerability-mysql-vulnerable'   => 0,
			'wpvulnerability-statistics'         => '',
			'wpvulnerability-statistics-cache'   => 0,
		);

		foreach ( $options as $key => $value ) {
			if ( ! get_option( $key ) ) {
				add_option( $key, $value );
			}
		}

		// Add wpvulnerability-analyze option if it does not exist.
		if ( ! get_option( 'wpvulnerability-analyze' ) ) {
			$default_wpvulnerability_analyze = array(
				'core'    => 0,
				'plugins' => 0,
				'themes'  => 0,
				'php'     => 0,
				'apache'  => 0,
				'nginx'   => 0,
				'mariadb' => 0,
				'mysql'   => 0,
			);
			$current_option                  = get_option( 'wpvulnerability-analyze' );
			if ( false === $current_option ) {
				add_option( 'wpvulnerability-analyze', $default_wpvulnerability_analyze );
			} else {
				$updated_option = $current_option;
				foreach ( $default_wpvulnerability_analyze as $key => $value ) {
					if ( ! array_key_exists( $key, $current_option ) ) {
						$updated_option[ $key ] = $value;
					}
				}
				if ( $updated_option !== $current_option ) {
					update_option( $option_name, $updated_option );
				}
			}
		}
	}
}

/**
 * Callback function to run when the plugin is deactivated.
 * Deletes options and removes scheduled wp-cron jobs.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_deactivation() {

	if ( is_multisite() ) {

		// Delete all plugin options for multisite.
		$multisite_options = array(
			'wpvulnerability_settings',
			'wpvulnerability-data',
			'wpvulnerability-analyze',
			'wpvulnerability-themes',
			'wpvulnerability-themes-cache',
			'wpvulnerability-themes-vulnerable',
			'wpvulnerability-plugins',
			'wpvulnerability-plugins-cache',
			'wpvulnerability-plugins-vulnerable',
			'wpvulnerability-core',
			'wpvulnerability-core-cache',
			'wpvulnerability-core-vulnerable',
			'wpvulnerability-php',
			'wpvulnerability-php-cache',
			'wpvulnerability-php-vulnerable',
			'wpvulnerability-apache',
			'wpvulnerability-apache-cache',
			'wpvulnerability-apache-vulnerable',
			'wpvulnerability-nginx',
			'wpvulnerability-nginx-cache',
			'wpvulnerability-nginx-vulnerable',
			'wpvulnerability-mariadb',
			'wpvulnerability-mariadb-cache',
			'wpvulnerability-mariadb-vulnerable',
			'wpvulnerability-mysql',
			'wpvulnerability-mysql-cache',
			'wpvulnerability-mysql-vulnerable',
			'wpvulnerability-statistics',
			'wpvulnerability-statistics-cache',
		);

		foreach ( $multisite_options as $option ) {
			delete_site_option( $option );
		}
	} else {

		// Delete all plugin options for single site.
		$single_site_options = array(
			'wpvulnerability_settings',
			'wpvulnerability-data',
			'wpvulnerability-analyze',
			'wpvulnerability-themes',
			'wpvulnerability-themes-cache',
			'wpvulnerability-themes-vulnerable',
			'wpvulnerability-plugins',
			'wpvulnerability-plugins-cache',
			'wpvulnerability-plugins-vulnerable',
			'wpvulnerability-core',
			'wpvulnerability-core-cache',
			'wpvulnerability-core-vulnerable',
			'wpvulnerability-php',
			'wpvulnerability-php-cache',
			'wpvulnerability-php-vulnerable',
			'wpvulnerability-apache',
			'wpvulnerability-apache-cache',
			'wpvulnerability-apache-vulnerable',
			'wpvulnerability-nginx',
			'wpvulnerability-nginx-cache',
			'wpvulnerability-nginx-vulnerable',
			'wpvulnerability-mariadb',
			'wpvulnerability-mariadb-cache',
			'wpvulnerability-mariadb-vulnerable',
			'wpvulnerability-mysql',
			'wpvulnerability-mysql-cache',
			'wpvulnerability-mysql-vulnerable',
			'wpvulnerability-statistics',
			'wpvulnerability-statistics-cache',
		);

		foreach ( $single_site_options as $option ) {
			delete_option( $option );
		}
	}

	// Unschedule and remove scheduled wp-cron jobs.
	wp_unschedule_event( wp_next_scheduled( 'wpvulnerability_notification' ), 'wpvulnerability_notification' );
	wp_clear_scheduled_hook( 'wpvulnerability_notification' );

	wp_unschedule_event( wp_next_scheduled( 'wpvulnerability_update_database' ), 'wpvulnerability_update_database' );
	wp_clear_scheduled_hook( 'wpvulnerability_update_database' );

	wp_unschedule_event( wp_next_scheduled( 'wpvulnerability_pull_db_data_event' ), 'wpvulnerability_pull_db_data_event' );
	wp_clear_scheduled_hook( 'wpvulnerability_pull_db_data_event' );
}

/**
 * Callback function to run when the plugin is uninstalled.
 * Deletes options and removes scheduled wp-cron jobs.
 *
 * @since 3.0.0
 *
 * @return void
 */
function wpvulnerability_uninstall() {

	// Deprecated options.
	delete_option( 'wpvulnerability_settings' );
	delete_option( 'wpvulnerability-data' );

	// Delete all plugin data options.
	$options = array(
		'wpvulnerability-themes',
		'wpvulnerability-themes-cache',
		'wpvulnerability-themes-vulnerable',
		'wpvulnerability-plugins',
		'wpvulnerability-plugins-cache',
		'wpvulnerability-plugins-vulnerable',
		'wpvulnerability-core',
		'wpvulnerability-core-cache',
		'wpvulnerability-core-vulnerable',
		'wpvulnerability-php',
		'wpvulnerability-php-cache',
		'wpvulnerability-php-vulnerable',
		'wpvulnerability-apache',
		'wpvulnerability-apache-cache',
		'wpvulnerability-apache-vulnerable',
		'wpvulnerability-nginx',
		'wpvulnerability-nginx-cache',
		'wpvulnerability-nginx-vulnerable',
		'wpvulnerability-mariadb',
		'wpvulnerability-mariadb-cache',
		'wpvulnerability-mariadb-vulnerable',
		'wpvulnerability-mysql',
		'wpvulnerability-mysql-cache',
		'wpvulnerability-mysql-vulnerable',
		'wpvulnerability-statistics',
		'wpvulnerability-statistics-cache',
		'wpvulnerability-analyze',
	);

	foreach ( $options as $option ) {
		delete_option( $option );
	}

	// Config data, not deleted when deactivated.
	delete_option( 'wpvulnerability-config' );

	// Delete all multisite options.
	$multisite_options = array(
		'wpvulnerability-themes',
		'wpvulnerability-themes-cache',
		'wpvulnerability-themes-vulnerable',
		'wpvulnerability-plugins',
		'wpvulnerability-plugins-cache',
		'wpvulnerability-plugins-vulnerable',
		'wpvulnerability-core',
		'wpvulnerability-core-cache',
		'wpvulnerability-core-vulnerable',
		'wpvulnerability-php',
		'wpvulnerability-php-cache',
		'wpvulnerability-php-vulnerable',
		'wpvulnerability-apache',
		'wpvulnerability-apache-cache',
		'wpvulnerability-apache-vulnerable',
		'wpvulnerability-nginx',
		'wpvulnerability-nginx-cache',
		'wpvulnerability-nginx-vulnerable',
		'wpvulnerability-mariadb',
		'wpvulnerability-mariadb-cache',
		'wpvulnerability-mariadb-vulnerable',
		'wpvulnerability-mysql',
		'wpvulnerability-mysql-cache',
		'wpvulnerability-mysql-vulnerable',
		'wpvulnerability-statistics',
		'wpvulnerability-statistics-cache',
		'wpvulnerability-analyze',
	);

	foreach ( $multisite_options as $option ) {
		delete_site_option( $option );
	}

	// Config data (Multisite), not deleted when deactivated.
	delete_site_option( 'wpvulnerability-config' );

	// Unschedule and remove scheduled wp-cron jobs.
	wp_unschedule_event( wp_next_scheduled( 'wpvulnerability_notification' ), 'wpvulnerability_notification' );
	wp_clear_scheduled_hook( 'wpvulnerability_notification' );

	wp_unschedule_event( wp_next_scheduled( 'wpvulnerability_update_database' ), 'wpvulnerability_update_database' );
	wp_clear_scheduled_hook( 'wpvulnerability_update_database' );
}

/**
 * Filters and returns the WPVulnerability analysis setting for a given type.
 *
 * This function retrieves the WPVulnerability analysis settings, either from
 * the single site or the multisite network, depending on the WordPress setup.
 * It returns false if the specified type ('core', 'plugins', 'themes',
 * 'php', 'apache', 'nginx', 'mariadb', 'mysql') is set. If the type is not set or is invalid, it returns true.
 *
 * @since 3.3.0
 *
 * @param string $type The type of analysis setting to retrieve ('core', 'plugins', 'themes', 'php', 'apache', 'nginx', 'mariadb', 'mysql').
 *
 * @return bool False if the specified type is set, true if not set or invalid.
 */
function wpvulnerability_analyze_filter( $type ) {

	// Retrieve the analysis settings based on the WordPress setup.
	$wpvulnerability_analyze = is_multisite() ? get_site_option( 'wpvulnerability-analyze', array() ) : get_option( 'wpvulnerability-analyze', array() );

	// Check the specified type and return the appropriate value.
	switch ( $type ) {
		case 'core':
			return ! ( isset( $wpvulnerability_analyze['core'] ) && (int) $wpvulnerability_analyze['core'] );
		case 'plugins':
			return ! ( isset( $wpvulnerability_analyze['plugins'] ) && (int) $wpvulnerability_analyze['plugins'] );
		case 'themes':
			return ! ( isset( $wpvulnerability_analyze['themes'] ) && (int) $wpvulnerability_analyze['themes'] );
		case 'php':
			return ! ( isset( $wpvulnerability_analyze['php'] ) && (int) $wpvulnerability_analyze['php'] );
		case 'apache':
			return ! ( isset( $wpvulnerability_analyze['apache'] ) && (int) $wpvulnerability_analyze['apache'] );
		case 'nginx':
			return ! ( isset( $wpvulnerability_analyze['nginx'] ) && (int) $wpvulnerability_analyze['nginx'] );
		case 'mariadb':
			return ! ( isset( $wpvulnerability_analyze['mariadb'] ) && (int) $wpvulnerability_analyze['mariadb'] );
		case 'mysql':
			return ! ( isset( $wpvulnerability_analyze['mysql'] ) && (int) $wpvulnerability_analyze['mysql'] );
		default:
			return true;
	}
}

/**
 * Clean the cache after an update
 *
 * @since 2.0.0
 *
 * @return void
 */
add_action( 'upgrader_process_complete', 'wpvulnerability_update_database_data', 10, 2 );

/**
 * Adds a notification count to the Plugins menu item in the WordPress admin if there are vulnerable plugins.
 *
 * This function retrieves the number of vulnerable plugins from the cache, either from a single site
 * or a multisite setup, and then displays the count in the WordPress admin menu next to the Plugins menu item.
 * The count is displayed with a gold background (#FFD700) and black text.
 *
 * @since 3.3.5
 *
 * @return void
 */
function wpvulnerability_counter_plugins() {

	if ( ! wpvulnerability_analyze_filter( 'plugins' ) ) {
		return;
	}

	// Retrieve the number of plugins vulnerabilities from cache.
	if ( ! is_multisite() ) {
		$wpvulnerability_plugins_count = get_option( 'wpvulnerability-plugins-vulnerable' );
	} elseif ( is_multisite() && is_network_admin() ) {
		$wpvulnerability_plugins_count = get_site_option( 'wpvulnerability-plugins-vulnerable' );
	}

	$wpvulnerability_plugins_total = 0;
	if ( $wpvulnerability_plugins_count ) {
		$wpvulnerability_plugins_total = json_decode( $wpvulnerability_plugins_count );
	}

	if ( $wpvulnerability_plugins_total > 0 ) {
		global $menu;
		foreach ( $menu as $key => $value ) {
			if ( 'plugins.php' === $menu[ $key ][2] ) {
				$menu[ $key ][0] .= ' <span class="update-plugins" style="background-color: #FFD700; color: #000000;"><span class="update-count" title="' . __( 'Vulnerabilities', 'wpvulnerability' ) . '">' . $wpvulnerability_plugins_total . '</span></span>'; // phpcs:ignore
				break;
			}
		}
	}
}
if ( is_multisite() && is_network_admin() ) {
	add_action( 'network_admin_menu', 'wpvulnerability_counter_plugins' );
} elseif ( ! is_multisite() ) {
	add_action( 'admin_menu', 'wpvulnerability_counter_plugins' );
}

/**
 * Adds a notification count to the Themes menu item in the WordPress admin if there are vulnerable themes.
 *
 * This function retrieves the number of vulnerable themes from the cache, either from a single site
 * or a multisite setup, and then displays the count in the WordPress admin menu next to the Themes menu item.
 * The count is displayed with a gold background (#FFD700) and black text.
 *
 * @since 3.3.5
 *
 * @return void
 */
function wpvulnerability_counter_themes() {

	if ( ! wpvulnerability_analyze_filter( 'themes' ) ) {
		return;
	}

	// Retrieve the number of theme vulnerabilities from cache.
	if ( ! is_multisite() ) {
		$wpvulnerability_themes_count = get_option( 'wpvulnerability-themes-vulnerable' );
	} elseif ( is_multisite() && is_network_admin() ) {
		$wpvulnerability_themes_count = get_site_option( 'wpvulnerability-themes-vulnerable' );
	}

	$wpvulnerability_themes_total = 0;
	if ( $wpvulnerability_themes_count ) {
		$wpvulnerability_themes_total = json_decode( $wpvulnerability_themes_count );
	}

	if ( $wpvulnerability_themes_total > 0 ) {

		if ( ! is_multisite() ) {
			global $submenu;
			if ( isset( $submenu['themes.php'] ) ) {
				foreach ( $submenu['themes.php'] as $key => $value ) {
					if ( 'themes.php' === $submenu['themes.php'][ $key ][2] ) {
						$submenu['themes.php'][ $key ][0] .= ' <span class="update-plugins" style="background-color: #FFD700; color: #000000;"><span class="update-count" title="' . __( 'Vulnerabilities', 'wpvulnerability' ) . '">' . $wpvulnerability_themes_total . '</span></span>'; // phpcs:ignore
						break;
					}
				}
			}
		} elseif ( is_multisite() && is_network_admin() ) {
			global $menu;
			foreach ( $menu as $key => $value ) {
				if ( 'themes.php' === $menu[ $key ][2] ) {
					$menu[ $key ][0] .= ' <span class="update-plugins" style="background-color: #FFD700; color: #000000;"><span class="update-count" title="' . __( 'Vulnerabilities', 'wpvulnerability' ) . '">' . $wpvulnerability_themes_total . '</span></span>'; // phpcs:ignore
					break;
				}
			}
		}
	}
}
if ( is_multisite() && is_network_admin() ) {
	add_action( 'network_admin_menu', 'wpvulnerability_counter_themes' );
} elseif ( ! is_multisite() ) {
	add_action( 'admin_menu', 'wpvulnerability_counter_themes' );
}

/**
 * Adds a notification count to the Updates submenu item under Dashboard in the WordPress admin if there are core updates.
 *
 * This function checks for core updates and then displays the count in the WordPress admin submenu
 * next to the Updates menu item with a gold background (#FFD700) and black text.
 *
 * @since 3.3.5
 *
 * @return void
 */
function wpvulnerability_counter_core() {

	if ( ! wpvulnerability_analyze_filter( 'core' ) ) {
		return;
	}

	// Retrieve the number of core vulnerabilities from cache.
	if ( ! is_multisite() ) {
		$wpvulnerability_core_count = get_option( 'wpvulnerability-core-vulnerable' );
	} elseif ( is_multisite() && is_network_admin() ) {
		$wpvulnerability_core_count = get_site_option( 'wpvulnerability-core-vulnerable' );
	}

	$wpvulnerability_core_total = 0;
	if ( $wpvulnerability_core_count ) {
		$wpvulnerability_core_total = json_decode( $wpvulnerability_core_count );
	}

	if ( $wpvulnerability_core_total > 0 ) {
		global $submenu;
		if ( isset( $submenu['index.php'] ) ) {
			foreach ( $submenu['index.php'] as $key => $value ) {
				if ( 'update-core.php' === $submenu['index.php'][ $key ][2] ) {
					$update_count                     = $core_updates_count + $wpvulnerability_core_count;
					$submenu['index.php'][ $key ][0] .= ' <span class="update-plugins" style="background-color: #FFD700; color: #000000;"><span class="update-count" title="' . __( 'Vulnerabilities', 'wpvulnerability' ) . '">' . $update_count . '</span></span>'; // phpcs:ignore
					break;
				}
			}
		}
	}
}
if ( is_multisite() && is_network_admin() ) {
	add_action( 'network_admin_menu', 'wpvulnerability_counter_core' );
} elseif ( ! is_multisite() ) {
	add_action( 'admin_menu', 'wpvulnerability_counter_core' );
}
