<?php
/**
 * MySQL functions
 *
 * @package WPVulnerability
 *
 * @version 3.4.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Retrieves vulnerabilities for a MySQL version and updates its data.
 *
 * This function detects the installed SQL server type and version. If MySQL is detected,
 * it checks for vulnerabilities using an external API and updates the MySQL data array
 * with the vulnerabilities found.
 *
 * @since 3.4.0
 *
 * @return array The updated MySQL data array containing vulnerability information.
 */
function wpvulnerability_get_fresh_mysql_vulnerabilities() {

	$mysql_version = null;
	$sqlserver     = wpvulnerability_detect_sqlserver();

	// Check if the detected SQL server is MySQL and has a valid version.
	if ( isset( $sqlserver['id'] ) && 'mysql' === $sqlserver['id'] && ! empty( $sqlserver['version'] ) ) {
		// Get the MySQL version and sanitize it.
		$mysql_version = wp_kses( (string) $sqlserver['version'], 'strip' );
	}

	// Initialize vulnerability-related fields in the MySQL data array.
	$mysql_data['vulnerabilities'] = null;
	$mysql_data['vulnerable']      = 0;

	// If the MySQL version is available, retrieve vulnerabilities for it.
	if ( $mysql_version ) {

		$mysql_api_response = wpvulnerability_get_mysql( $mysql_version, 0 );

		// If vulnerabilities are found, update the MySQL data accordingly.
		if ( ! empty( $mysql_api_response ) ) {

			$mysql_data['vulnerabilities'] = $mysql_api_response;
			$mysql_data['vulnerable']      = 1;

		}
	}

	return $mysql_data;
}

/**
 * Retrieve and Cache Installed MySQL Vulnerabilities
 *
 * This function retrieves the list of installed MySQL versions, checks for vulnerabilities,
 * caches the data, and sends an email notification if vulnerabilities are detected.
 *
 * @since 3.4.0
 *
 * @return string JSON-encoded array of MySQL data with vulnerabilities and vulnerable status.
 */
function wpvulnerability_mysql_get_installed() {

	$wpvulnerability_mysql_vulnerable = 0;

	// Retrieve fresh vulnerabilities for the installed MySQL version.
	$mysql = wpvulnerability_get_fresh_mysql_vulnerabilities();

	// Check if the MySQL version is vulnerable and count the vulnerabilities.
	if ( isset( $mysql['vulnerable'] ) && (int) $mysql['vulnerable'] ) {
		$wpvulnerability_mysql_vulnerable = count( $mysql['vulnerabilities'] );
	}

	// Cache the vulnerability data and the timestamp for cache expiration.
	if ( is_multisite() ) {
		update_site_option( 'wpvulnerability-mysql', wp_json_encode( $mysql ) );
		update_site_option( 'wpvulnerability-mysql-vulnerable', wp_json_encode( number_format( $wpvulnerability_mysql_vulnerable, 0, '.', '' ) ) );
		update_site_option( 'wpvulnerability-mysql-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );
	} else {
		update_option( 'wpvulnerability-mysql', wp_json_encode( $mysql ) );
		update_option( 'wpvulnerability-mysql-vulnerable', wp_json_encode( number_format( $wpvulnerability_mysql_vulnerable, 0, '.', '' ) ) );
		update_option( 'wpvulnerability-mysql-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );
	}

	// Return the JSON-encoded array of MySQL data.
	return wp_json_encode( $mysql );
}

/**
 * Retrieve Cached MySQL Vulnerabilities or Update the Cache if Stale.
 *
 * This function checks the cache for stored MySQL vulnerabilities. If the cache is stale or missing,
 * it updates the cache with the latest vulnerabilities data.
 *
 * @since 3.4.0
 *
 * @return array Array of MySQL data with their vulnerabilities.
 */
function wpvulnerability_mysql_get_vulnerabilities() {

	if ( is_multisite() ) {

		// Get the cached MySQL data expiration timestamp.
		$mysql_data_cache = json_decode( get_site_option( 'wpvulnerability-mysql-cache' ) );

		// Get the cached MySQL data.
		$mysql_data = json_decode( get_site_option( 'wpvulnerability-mysql' ), true );

	} else {

		// Get the cached MySQL data expiration timestamp.
		$mysql_data_cache = json_decode( get_option( 'wpvulnerability-mysql-cache' ) );

		// Get the cached MySQL data.
		$mysql_data = json_decode( get_option( 'wpvulnerability-mysql' ), true );

	}

	// If the cache is stale or the MySQL data is missing, update the cache.
	if ( $mysql_data_cache < time() || empty( $mysql_data ) ) {

		// Retrieve the latest MySQL vulnerabilities and update the cache.
		$mysql_data = json_decode( wpvulnerability_mysql_get_installed(), true );

	}

	return $mysql_data;
}

/**
 * Refresh the MySQL Cache and Remove Stale Data.
 *
 * This function updates the cache for MySQL vulnerabilities by retrieving the latest
 * installed MySQL data. It ensures that the cache is refreshed with the most current
 * vulnerability information, effectively removing any outdated data.
 *
 * @since 3.4.0
 *
 * @return void
 */
function wpvulnerability_mysql_get_vulnerabilities_clean() {

	// Refresh the cache by updating the installed MySQL data.
	wpvulnerability_mysql_get_installed();
}
