<?php
/**
 * General functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Checks and validates user capabilities for managing vulnerability settings in a WordPress environment.
 *
 * This function verifies if the current user has the appropriate permissions to manage network settings
 * in a multisite installation or manage options in a single site installation. It is designed to be used
 * in the context of a plugin or theme that interacts with WordPress vulnerability settings.
 *
 * @since 3.0.0
 *
 * @return bool Returns false if the current user does not have the required capabilities, true otherwise.
 */
function wpvulnerability_capabilities() {

	$user = wp_get_current_user();
	if ( is_multisite() && is_super_admin( $user->ID ) && ( is_network_admin() || is_main_site() ) ) {
		// In a WordPress Multisite, the user must be SuperAdmin.
		return true;
	} elseif ( ! is_multisite() && is_admin() ) {
		// In a WordPress simple, the user must be Admin.
		return true;
	}

	return false;
}

/**
 * Checks if the `shell_exec` function can be used.
 *
 * This function verifies if the `shell_exec` function is not disabled in the server's
 * configuration and is able to execute a basic shell command. It also checks for
 * safe mode, which is relevant for older PHP versions before 5.4.
 *
 * @since 3.4.0
 *
 * @return bool True if `shell_exec` is available and working, false otherwise.
 */
function wpvulnerability_can_shell_exec() {
	// Check if `shell_exec` is disabled.
	if ( in_array( 'shell_exec', array_map( 'trim', explode( ',', ini_get( 'disable_functions' ) ) ), true ) ) {
		return false;
	}

	// Try to execute a simple command to confirm functionality.
	$test = @shell_exec( 'echo test' ); // phpcs:ignore

	// If the command execution failed or returned null, shell_exec is not working.
	if ( null === $test ) {
		return false;
	}

	return true;
}

/**
 * Sanitize a version string.
 *
 * This function removes any leading or trailing whitespace from the version string
 * and strips out any non-alphanumeric characters except for hyphens, underscores, and dots.
 *
 * @version 2.0.0
 *
 * @param string $version The version string to sanitize.
 *
 * @return string The sanitized version string.
 */
function wpvulnerability_sanitize_version( $version ) {

	// Remove any leading/trailing whitespace.
	$version = trim( $version );

	// Strip out any non-alphanumeric characters except for hyphens, underscores, and dots.
	$version = preg_replace( '/[^a-zA-Z0-9_\-.]+/', '', $version );

	return $version;
}

/**
 * Sanitize a PHP version string to ensure it follows the standard format.
 *
 * This function checks the input version string against a regular expression
 * to match the standard PHP versioning format (major.minor[.patch]).
 * It returns the matched version if it conforms to the expected format,
 * otherwise, it returns the original input version.
 *
 * @since 3.1.0 Introduced.
 *
 * @param string $version The PHP version string to sanitize.
 * @return string The sanitized version string, if it matches the standard format; otherwise, the original version string.
 */
function wpvulnerability_sanitize_version_php( $version ) {

	// Sanitize the version string using the base sanitizer.
	$version = wpvulnerability_sanitize_version( $version );

	// Validate format (major.minor[.patch]) and sanitize.
	if ( preg_match( '/^(\d+\.\d+(\.\d+)?)/', $version, $match ) ) {
		if ( isset( $match[0] ) ) {
			return trim( $match[0] );
		}
	}

	return $version;
}

/**
 * Sanitize an Apache HTTPD version string to ensure it follows the standard format.
 *
 * This function checks the input version string against a regular expression
 * to match the standard Apache versioning format (major.minor[.patch]).
 * It returns the matched version if it conforms to the expected format,
 * otherwise, it returns the original input version.
 *
 * @since 3.2.0 Introduced.
 *
 * @param string $version The Apache version string to sanitize.
 * @return string The sanitized version string, if it matches the standard format; otherwise, the original version string.
 */
function wpvulnerability_sanitize_version_apache( $version ) {

	// Sanitize the version string using the base sanitizer.
	$version = wpvulnerability_sanitize_version( $version );

	// Validate format (major.minor[.patch]) and sanitize.
	if ( preg_match( '/^(\d+\.\d+(\.\d+)?)/', $version, $match ) ) {
		if ( isset( $match[0] ) ) {
			return trim( $match[0] );
		}
	}

	return $version;
}

/**
 * Sanitize an nginx version string to ensure it follows the standard format.
 *
 * This function checks the input version string against a regular expression
 * to match the standard nginx versioning format (major.minor[.patch]).
 * It returns the matched version if it conforms to the expected format,
 * otherwise, it returns the original input version.
 *
 * @since 3.2.0 Introduced.
 *
 * @param string $version The nginx version string to sanitize.
 * @return string The sanitized version string, if it matches the standard format; otherwise, the original version string.
 */
function wpvulnerability_sanitize_version_nginx( $version ) {

	// Sanitize the version string using the base sanitizer.
	$version = wpvulnerability_sanitize_version( $version );

	// Validate format (major.minor[.patch]) and sanitize.
	if ( preg_match( '/^(\d+\.\d+(\.\d+)?)/', $version, $match ) ) {
		if ( isset( $match[0] ) ) {
			return trim( $match[0] );
		}
	}

	return $version;
}

/**
 * Detects the web server software and version from the SERVER_SOFTWARE server variable.
 *
 * This function attempts to identify the web server software (e.g., Apache, nginx) and its version
 * based on the 'SERVER_SOFTWARE' environment variable provided by the server. It uses regular expressions
 * to parse the web server name and version. The function also sanitizes the detected version number
 * to a standard format (major.minor.patch).
 *
 * @since 3.2.0 Introduced.
 *
 * @return array Returns an associative array with three keys:
 *               'id' => A short, lowercase identifier for the web server (e.g., 'apache', 'nginx'),
 *               'name' => A more readable name for the web server (e.g., 'Apache HTTPD', 'nginx'),
 *               'version' => The detected version of the web server, sanitized to a standard format.
 */
function wpvulnerability_detect_webserver() {
	// Initialize an array to hold the web server information.
	$webserver = array(
		'id'      => null,
		'name'    => null,
		'version' => null,
	);

	// Check if the SERVER_SOFTWARE variable is set.
	if ( isset( $_SERVER['SERVER_SOFTWARE'] ) ) {
		// Trim any leading or trailing whitespace from the server software string.
		$webserver_software = trim( wp_kses( wp_unslash( $_SERVER['SERVER_SOFTWARE'] ), 'strip' ) );

		// Use regular expressions to extract the web server name and version.
		if ( preg_match( '/^(\w+)\/?([^\s]*)/', $webserver_software, $matches ) ) {
			// Check and assign the web server name if available.
			if ( isset( $matches[1] ) && trim( (string) $matches[1] ) ) {
				$webserver['name'] = trim( $matches[1] );
			}

			// Check and assign the web server version if available.
			if ( isset( $matches[2] ) && trim( (string) $matches[2] ) ) {
				$webserver['version'] = trim( $matches[2] );
			}
		}
	}

	// Normalize and set the web server ID based on the detected name.
	if ( isset( $webserver['name'] ) && $webserver['name'] ) {
		switch ( trim( strtolower( $webserver['name'] ) ) ) {
			case 'httpd':
			case 'apache':
				$webserver['id']   = 'apache';
				$webserver['name'] = 'Apache HTTPD'; // Provide a more readable name.
				break;
			case 'nginx':
				$webserver['id']   = 'nginx'; // 'nginx' is both the ID and the readable name.
				$webserver['name'] = 'nginx'; // Provide a more readable name.
				break;
			// Additional web servers can be added here.
		}
	}

	// If the version is not detected, try to get it from the OS.
	if ( empty( $webserver['version'] ) && wpvulnerability_can_shell_exec() ) {
		if ( 'apache' === $webserver['id'] ) {
			$apache_version = shell_exec( 'apache2 -v 2>&1' ); // phpcs:ignore
			if ( empty( $apache_version ) ) {
				$apache_version = shell_exec( 'httpd -v 2>&1' ); // phpcs:ignore
			}
			if ( preg_match( '/Apache\/([\d.]+)/', $apache_version, $version_matches ) ) {
				$webserver['version'] = $version_matches[1];
			}
		} elseif ( 'nginx' === $webserver['id'] ) {
			// Try to get Nginx version from the OS.
			$nginx_version = shell_exec( 'nginx -v 2>&1' ); // phpcs:ignore
			if ( preg_match( '/nginx\/([\d.]+)/', $nginx_version, $version_matches ) ) {
				$webserver['version'] = $version_matches[1];
			}
			if ( empty( $nginx_version ) ) {
				$angie_version = shell_exec( 'angie -v 2>&1' ); // phpcs:ignore
				if ( preg_match( '/angie\/([\d.]+)/', $angie_version, $version_matches ) ) {
						$webserver['version'] = $version_matches[1];
				}
			}
		}
	}

	// Sanitize and validate the web server version format.
	if ( isset( $webserver['version'] ) && $webserver['version'] ) {
		// Sanitize the version number to ensure it's in a 'major.minor.patch' format.
		$version = wpvulnerability_sanitize_version( $webserver['version'] );

		// Validate and format the version number.
		if ( preg_match( '/^(\d+\.\d+(\.\d+)?)/', $version, $match ) ) {
			if ( isset( $match[0] ) ) {
				$webserver['version'] = trim( $match[0] );
			}
		}
	}

	// Return the detected web server information.
	return $webserver;
}

/**
 * Sanitize a MariaDB version string to ensure it follows the standard format.
 *
 * This function checks the input version string against a regular expression
 * to match the standard MariaDB versioning format (major.minor[.patch]).
 * It returns the matched version if it conforms to the expected format;
 * otherwise, it returns the original input version.
 *
 * @since 3.4.0
 *
 * @param string $version The MariaDB version string to sanitize.
 * @return string The sanitized version string if it matches the standard format; otherwise, the original version string.
 */
function wpvulnerability_sanitize_version_mariadb( $version ) {

	// Sanitize the version string using the base sanitizer.
	$version = wpvulnerability_sanitize_version( $version );

	// Validate and extract the version format (major.minor[.patch]).
	if ( preg_match( '/^\d+\.\d+(\.\d+)?/', $version, $match ) ) {
		if ( isset( $match[0] ) ) {
			return trim( $match[0] );
		}
	}

	return $version;
}

/**
 * Sanitize a MySQL version string to ensure it follows the standard format.
 *
 * This function checks the input version string against a regular expression
 * to match the standard MySQL versioning format (major.minor[.patch]).
 * It returns the matched version if it conforms to the expected format;
 * otherwise, it returns the original input version.
 *
 * @since 3.4.0
 *
 * @param string $version The MySQL version string to sanitize.
 * @return string The sanitized version string if it matches the standard format; otherwise, the original version string.
 */
function wpvulnerability_sanitize_version_mysql( $version ) {

	// Sanitize the version string using the base sanitizer.
	$version = wpvulnerability_sanitize_version( $version );

	// Validate and extract the version format (major.minor[.patch]).
	if ( preg_match( '/^\d+\.\d+(\.\d+)?/', $version, $match ) ) {
		if ( isset( $match[0] ) ) {
			return trim( $match[0] );
		}
	}

	return $version;
}

/**
 * Detect the SQL server software and version from the database server.
 *
 * This function identifies the SQL server software (e.g., MariaDB, MySQL) and its version
 * by querying the database using the 'SHOW VARIABLES' command. It parses the server name
 * and version using the results and sanitizes the detected version number to a standard format (major.minor.patch).
 *
 * @since 3.4.0
 *
 * @return array Returns an associative array with three keys:
 *               'id' => A short, lowercase identifier for the SQL server (e.g., 'mariadb', 'mysql'),
 *               'name' => A more readable name for the SQL server (e.g., 'MariaDB', 'MySQL'),
 *               'version' => The detected version of the SQL server, sanitized to a standard format.
 */
function wpvulnerability_detect_sqlserver() {
	// Initialize an array to hold the SQL server information.
	$sqlserver = array(
		'id'      => null,
		'name'    => null,
		'version' => null,
	);

	$possible_version  = null;
	$possible_database = null;

	global $wpdb;

	// Query to get the database server type (version_comment).
	// IGNORE REASON: this is not a usual query to WordPress but to the system.
	$results = $wpdb->get_results( $wpdb->prepare( 'SHOW VARIABLES LIKE %s', 'version_comment' ) ); // phpcs:ignore

	if ( $wpdb->last_error ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			do_action( 'wpdb_last_error', $wpdb->last_error );
		}
	}

	// Process the results to determine the database type.
	if ( ! empty( $results ) && isset( $results[0]->Value ) ) {
		$possible_database = trim( $results[0]->Value );
	}

	// Query to get the database server version.
	// IGNORE REASON: this is not a usual query to WordPress but to the system.
	$results = $wpdb->get_results( $wpdb->prepare( 'SHOW VARIABLES LIKE %s', 'version' ) ); // phpcs:ignore

	if ( $wpdb->last_error ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			do_action( 'wpdb_last_error', $wpdb->last_error );
		}
	}

	// Process the results to determine the database version.
	if ( ! empty( $results ) && isset( $results[0]->Value ) ) {
		$possible_version = trim( $results[0]->Value );
	}

	// Determine the database type and set the appropriate values in the array.
	if ( isset( $possible_database ) && $possible_database ) {

		if ( false !== stripos( $possible_database, 'mariadb' ) ) {

			$sqlserver['id']   = 'mariadb';
			$sqlserver['name'] = 'MariaDB';

		} elseif ( false !== stripos( $possible_database, 'mysql' ) ) {

			$sqlserver['id']   = 'mysql';
			$sqlserver['name'] = 'MySQL';

		}
	}

	// Sanitize and set the version if it was detected.
	if ( isset( $possible_version ) && $possible_version && $sqlserver['id'] ) {

		if ( 'mariadb' === $sqlserver['id'] ) {

			$sqlserver['version'] = wpvulnerability_sanitize_version_mariadb( $possible_version );

		} elseif ( 'mysql' === $sqlserver['id'] ) {

			$sqlserver['version'] = wpvulnerability_sanitize_version_mysql( $possible_version );

		}
	}

	// Return the detected SQL server information.
	return $sqlserver;
}

/**
 * Returns a human-readable HTML entity for the given comparison operator.
 *
 * This function takes a comparison operator in string format and returns
 * its corresponding HTML entity for better readability in web contexts.
 *
 * @version 2.0.0
 *
 * @param string $op The operator string to prettify.
 *
 * @return string The pretty operator HTML string.
 */
function wpvulnerability_pretty_operator( $op ) {

	switch ( trim( strtolower( $op ) ) ) {
		// Less than.
		case 'lt':
			return '&lt;&nbsp;';
		// Less than or equal to.
		case 'le':
			return '&le;&nbsp;';
		// Greater than.
		case 'gt':
			return '&gt;&nbsp;';
		// Greater than or equal to.
		case 'ge':
			return '&ge;&nbsp;';
		// Equal to.
		case 'eq':
			return '&equals;&nbsp;';
		// Not equal to.
		case 'ne':
			return '&ne;&nbsp;';
		// Return the original operator if it's not recognized.
		default:
			return $op;
	}
}

/**
 * Returns a human-readable severity level.
 *
 * This function takes a severity string and returns a human-readable
 * severity level, localized for translation.
 *
 * @version 2.0.0
 *
 * @param string $severity The severity string to prettify.
 *
 * @return string The human-readable severity string.
 */
function wpvulnerability_severity( $severity ) {

	switch ( trim( strtolower( $severity ) ) ) {
		// No severity.
		case 'n':
			/* translators: Severity: None */
			return __( 'None', 'wpvulnerability' );
		// Low severity.
		case 'l':
			/* translators: Severity: Low */
			return __( 'Low', 'wpvulnerability' );
		// Medium severity.
		case 'm':
			/* translators: Severity: Medium */
			return __( 'Medium', 'wpvulnerability' );
		// High severity.
		case 'h':
			/* translators: Severity: High */
			return __( 'High', 'wpvulnerability' );
		// Critical severity.
		case 'c':
			/* translators: Severity: Critical */
			return __( 'Critical', 'wpvulnerability' );
		// Return the original severity if it's not recognized.
		default:
			return $severity;
	}
}

/**
 * Retrieves vulnerabilities information from the API.
 *
 * This function fetches vulnerability information based on the provided type and slug.
 * It supports caching to minimize API requests and improve performance.
 *
 * @version 2.0.0
 *
 * @param string $type  The type of vulnerability. Can be 'core', 'plugin' or 'theme'.
 * @param string $slug  The slug of the plugin or theme. For core vulnerabilities, it is the version string.
 * @param int    $cache Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|bool An array with the vulnerability information or false if there's an error.
 */
function wpvulnerability_get( $type, $slug = '', $cache = 1 ) {

	$vulnerability_data = null;

	// Validate vulnerability type.
	switch ( trim( strtolower( $type ) ) ) {
		case 'core':
			$type = 'core';
			break;
		case 'plugin':
			$type = 'plugin';
			break;
		case 'theme':
			$type = 'theme';
			break;
		default:
			wp_die( 'Unknown vulnerability type sent.' );
			break;
	}

	// Validate slug for plugin or theme.
	if ( 'plugin' === $type || 'theme' === $type ) {
		if ( empty( sanitize_title( $slug ) ) ) {
			return false;
		}
		// Validate slug for core.
	} elseif ( 'core' === $type ) {
		if ( ! wpvulnerability_sanitize_version( $slug ) ) {
			return false;
		}
	}

	// Cache key.
	$key = 'wpvulnerability_' . $type . '_' . $slug;
	if ( $cache ) {
		if ( is_multisite() ) {
			$vulnerability_data = get_site_transient( $key );
		} else {
			$vulnerability_data = get_transient( $key );
		}
	}

	// If not cached, get the updated data.
	if ( empty( $vulnerability_data ) ) {
		$url      = WPVULNERABILITY_API_HOST . $type . '/' . $slug . '/';
		$response = wp_remote_get( $url, array( 'timeout' => 2500 ) );

		if ( ! is_wp_error( $response ) ) {
			$body = wp_remote_retrieve_body( $response );
			if ( is_multisite() ) {
				set_site_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_site_transient( $key );
			} else {
				set_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_transient( $key );
			}
		}
	}

	return json_decode( $vulnerability_data, true );
}

/**
 * Retrieve vulnerabilities for a specific version of WordPress Core.
 *
 * This function fetches vulnerability information for a given version of WordPress Core.
 * If no version is provided, it retrieves vulnerabilities for the currently installed version.
 * It supports caching to minimize API requests and improve performance.
 *
 * @since 2.0.0
 *
 * @param string|null $version The version number of WordPress Core. If null, retrieves for the installed version.
 * @param int         $cache   Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false Array of vulnerabilities, or false on error.
 */
function wpvulnerability_get_core( $version = null, $cache = 1 ) {

	// Sanitize the version number.
	if ( ! wpvulnerability_sanitize_version( $version ) ) {
		$version = null;
	}

	// If version number is null, retrieve for the installed version.
	if ( is_null( $version ) ) {
		$version = get_bloginfo( 'version' );
	}

	// Get vulnerabilities from API.
	$response = wpvulnerability_get( 'core', $version, $cache );

	// Check for errors.
	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Process vulnerabilities and return as an array.
	$vulnerability = array();
	foreach ( $response['data']['vulnerability'] as $v ) {

		// Set vulnerability name if available.
		$name = null;
		if ( isset( $v['name'] ) ) {
			$name = $v['name'];
		}

		// Set vulnerability link if available.
		$link = null;
		if ( isset( $v['link'] ) ) {
			$link = $v['link'];
		}

		$vulnerability[] = array(
			'name'   => wp_kses( (string) $name, 'strip' ),
			'link'   => esc_url_raw( (string) $link ),
			'source' => $v['source'],
			'impact' => $v['impact'],
		);
	}

	return $vulnerability;
}

/**
 * Retrieves vulnerabilities for a specified plugin, optionally returning general plugin data.
 *
 * This function sanitizes the plugin slug and verifies the version number before querying the vulnerability API.
 * If `$data` is set to 1, it returns general information about the plugin instead of vulnerabilities.
 * The function returns an array of vulnerabilities or plugin data based on the `$data` parameter, or `false`
 * if no vulnerabilities are found or the version number is invalid and `$data` is not set.
 *
 * @since 2.0.0 Introduced.
 *
 * @param string $slug    The slug of the plugin to check for vulnerabilities.
 * @param string $version The version of the plugin to check. The function may return `false` if this is invalid and `$data` is not set.
 * @param int    $data    Optional. Set to 1 to return general plugin data instead of vulnerabilities. Default 0 (return vulnerabilities).
 * @param int    $cache   Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false An array of vulnerabilities or plugin data if `$data` is set to 1, or `false` if no vulnerabilities are found or the version number is invalid and `$data` is not set.
 */
function wpvulnerability_get_plugin( $slug, $version, $data = 0, $cache = 1 ) {

	// Sanitize the plugin slug.
	$slug = sanitize_title( $slug );

	// If the version number is invalid, return false.
	if ( ! wpvulnerability_sanitize_version( $version ) && ! $data ) {
		return false;
	}

	// Get the response from the vulnerability API.
	$response = wpvulnerability_get( 'plugin', $slug, $cache );

	// Create an empty array to store the vulnerabilities.
	$vulnerability = array();

	if ( 1 === $data ) {

		if ( isset( $response['data'] ) ) {
			$vulnerability = array(
				'name'   => wp_kses( (string) $response['data']['name'], 'strip' ),
				'link'   => esc_url( (string) $response['data']['link'] ),
				'latest' => number_format( (int) $response['data']['latest'], 0, '.', '' ),
				'closed' => number_format( (int) $response['data']['closed'], 0, '.', '' ),
			);
		}
	} else {

		// If there are no vulnerabilities, return false.
		if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
			return false;
		}

		// Loop through each vulnerability and check if it affects the specified version of the plugin.
		foreach ( $response['data']['vulnerability'] as $v ) {

			// If the vulnerability has minimum and maximum versions, check if the specified version falls within that range.
			if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

				if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

					// Add the vulnerability to the array.
					$vulnerability[] = array(
						'name'        => wp_kses( (string) $v['name'], 'strip' ),
						'description' => wp_kses_post( (string) $v['description'] ),
						'versions'    => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
						'version'     => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
						'unfixed'     => (int) $v['operator']['unfixed'],
						'closed'      => (int) $v['operator']['closed'],
						'source'      => $v['source'],
						'impact'      => $v['impact'],
					);

				}

				// If the vulnerability has only a maximum version, check if the specified version is below that version.
			} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

				if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

					// Add the vulnerability to the list.
					$vulnerability[] = array(
						'name'        => wp_kses( (string) $v['name'], 'strip' ),
						'description' => wp_kses_post( (string) $v['description'] ),
						'versions'    => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
						'version'     => wp_kses( (string) $v['operator']['max_version'], 'strip' ),
						'unfixed'     => (int) $v['operator']['unfixed'],
						'closed'      => (int) $v['operator']['closed'],
						'source'      => $v['source'],
						'impact'      => $v['impact'],
					);

				}

				// If the vulnerability has only a minimum version, check if the specified version is above that version.
			} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

				if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

					// Add the vulnerability to the list.
					$vulnerability[] = array(
						'name'        => wp_kses( (string) $v['name'], 'strip' ),
						'description' => wp_kses_post( (string) $v['description'] ),
						'versions'    => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'], 'strip' ),
						'version'     => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
						'unfixed'     => (int) $v['operator']['unfixed'],
						'closed'      => (int) $v['operator']['closed'],
						'source'      => $v['source'],
						'impact'      => $v['impact'],
					);

				}
			}
		}
	}

	return $vulnerability;
}

/**
 * Get vulnerabilities for a specific theme.
 *
 * This function retrieves and sanitizes the theme slug and version before querying the vulnerability API.
 * It returns an array of vulnerabilities if any are found, or false if there are none.
 *
 * @since 2.0.0
 *
 * @param string $slug    Slug of the theme.
 * @param string $version Version of the theme.
 * @param int    $cache   Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false Returns an array of vulnerabilities, or false if there are none.
 */
function wpvulnerability_get_theme( $slug, $version, $cache = 1 ) {

	// Sanitize the theme slug.
	$slug = sanitize_title( $slug );

	// Validate the version number.
	if ( ! wpvulnerability_sanitize_version( $version ) ) {
		return false;
	}

	// Get the response from the vulnerability API.
	$response = wpvulnerability_get( 'theme', $slug, $cache );

	// Create an empty array to store the vulnerabilities.
	$vulnerability = array();

	// If there are no vulnerabilities, return false.
	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Process each vulnerability.
	foreach ( $response['data']['vulnerability'] as $v ) {

		// Check if the version falls within the min and max operator range.
		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'        => wp_kses( (string) $v['name'], 'strip' ),
					'description' => wp_kses_post( (string) $v['description'] ),
					'versions'    => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'     => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'     => (int) $v['operator']['unfixed'],
					'closed'      => (int) $v['operator']['closed'],
					'source'      => $v['source'],
					'impact'      => $v['impact'],
				);

			}

			// Check if the version is below the max operator.
		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'        => wp_kses( (string) $v['name'], 'strip' ),
					'description' => wp_kses_post( (string) $v['description'] ),
					'versions'    => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'     => wp_kses( (string) $v['operator']['max_version'], 'strip' ),
					'unfixed'     => (int) $v['operator']['unfixed'],
					'closed'      => (int) $v['operator']['closed'],
					'source'      => $v['source'],
					'impact'      => $v['impact'],
				);

			}

			// Check if the version is above the min operator.
		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'        => wp_kses( (string) $v['name'], 'strip' ),
					'description' => wp_kses_post( (string) $v['description'] ),
					'versions'    => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'], 'strip' ),
					'version'     => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'     => (int) $v['operator']['unfixed'],
					'closed'      => (int) $v['operator']['closed'],
					'source'      => $v['source'],
					'impact'      => $v['impact'],
				);

			}
		}
	}

	return $vulnerability;
}

/**
 * Get statistics
 *
 * Returns an array with statistical information about vulnerabilities and their respective products.
 *
 * @since 2.0.0
 *
 * @param int $cache Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false Returns an array with the statistical information if successful, false otherwise.
 */
function wpvulnerability_get_statistics( $cache = 1 ) {

	$vulnerability = null;
	$key           = 'wpvulnerability_stats';

	// Get cached statistics if available.
	if ( $cache ) {
		if ( is_multisite() ) {
			$vulnerability = get_site_transient( $key );
		} else {
			$vulnerability = get_transient( $key );
		}
	}

	// If cached statistics are not available, retrieve them from the API and store them in cache.
	if ( empty( $vulnerability ) ) {

		$url      = WPVULNERABILITY_API_HOST;
		$response = wp_remote_get( $url, array( 'timeout' => 2500 ) );

		if ( ! is_wp_error( $response ) ) {

			$body = wp_remote_retrieve_body( $response );
			if ( is_multisite() ) {
				set_site_transient( $key, $body, HOUR_IN_SECONDS * 24 );
				$vulnerability = get_site_transient( $key );
			} else {
				set_transient( $key, $body, HOUR_IN_SECONDS * 24 );
				$vulnerability = get_transient( $key );
			}
		}
	}

	// If the response does not contain statistics, return false.
	$response = json_decode( $vulnerability, true );

	if ( ! isset( $response['stats'] ) ) {
		return false;
	}

	$sponsors = array();
	if ( isset( $response['behindtheproject']['sponsors'] ) && count( $response['behindtheproject']['sponsors'] ) ) {

		foreach ( $response['behindtheproject']['sponsors'] as $s ) {

			$sponsors[] = $s;

			unset( $s );
		}
	}

	$contributors = array();
	if ( isset( $response['behindtheproject']['contributors'] ) && count( $response['behindtheproject']['contributors'] ) ) {

		foreach ( $response['behindtheproject']['contributors'] as $s ) {

			$contributors[] = $s;

			unset( $s );
		}
	}

	// Return an array with statistical information.
	return array(
		'core'         => array(
			'versions' => (int) $response['stats']['products']['core'],
		),
		'plugins'      => array(
			'products'        => (int) $response['stats']['products']['plugins'],
			'vulnerabilities' => (int) $response['stats']['plugins'],
		),
		'themes'       => array(
			'products'        => (int) $response['stats']['products']['themes'],
			'vulnerabilities' => (int) $response['stats']['themes'],
		),
		'php'          => array(
			'vulnerabilities' => (int) $response['stats']['php'],
		),
		'apache'       => array(
			'vulnerabilities' => (int) $response['stats']['apache'],
		),
		'nginx'        => array(
			'vulnerabilities' => (int) $response['stats']['nginx'],
		),
		'mariadb'      => array(
			'vulnerabilities' => (int) $response['stats']['mariadb'],
		),
		'mysql'        => array(
			'vulnerabilities' => (int) $response['stats']['mysql'],
		),
		'sponsors'     => $sponsors,
		'contributors' => $contributors,
		'updated'      => array(
			'unixepoch' => (int) $response['updated'],
			'datetime'  => gmdate( 'Y-m-d H:i:s', (int) $response['updated'] ),
			'iso8601'   => gmdate( 'c', (int) $response['updated'] ),
			'rfc2822'   => gmdate( 'r', (int) $response['updated'] ),
		),
	);
}

/**
 * Retrieves the latest vulnerability statistics.
 *
 * This function calls the wpvulnerability API to get fresh statistics related to vulnerabilities
 * and returns the updated information.
 *
 * @since 3.4.0
 *
 * @return array The updated vulnerability statistics.
 */
function wpvulnerability_get_fresh_statistics() {

	$statistics_api_response = wpvulnerability_get_statistics();

	return $statistics_api_response;
}

/**
 * Retrieves and caches the latest vulnerability statistics.
 *
 * This function retrieves the most recent vulnerability statistics, caches the data,
 * and returns the information as a JSON-encoded array. The cache expiration timestamp is also updated.
 *
 * @since 3.4.0
 *
 * @return string JSON-encoded array containing the vulnerability statistics.
 */
function wpvulnerability_statistics_get() {

	// Retrieve fresh statistics.
	$statistics = wpvulnerability_get_fresh_statistics();

	// Cache the statistics data and the timestamp for cache expiration.
	if ( is_multisite() ) {
		update_site_option( 'wpvulnerability-statistics', wp_json_encode( $statistics ) );
		update_site_option( 'wpvulnerability-statistics-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );
	} else {
		update_option( 'wpvulnerability-statistics', wp_json_encode( $statistics ) );
		update_option( 'wpvulnerability-statistics-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );
	}

	// Return the JSON-encoded array of statistics data.
	return wp_json_encode( $statistics );
}

/**
 * Get vulnerabilities for a specific PHP version.
 *
 * This function retrieves vulnerability data for a specified PHP version.
 * It supports caching to minimize API requests and improve performance.
 *
 * @since 3.0.0
 *
 * @param string $version PHP Version.
 * @param int    $cache   Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false Returns an array of vulnerabilities, or false if there are none.
 */
function wpvulnerability_get_php( $version, $cache = 1 ) {

	$key                = 'wpvulnerability_php';
	$vulnerability_data = null;
	$vulnerability      = array();

	// Get cached statistics if available.
	if ( $cache ) {
		if ( is_multisite() ) {
			$vulnerability_data = get_site_transient( $key );
		} else {
			$vulnerability_data = get_transient( $key );
		}
	}

	// If cached statistics are not available, retrieve them from the API and store them in cache.
	if ( empty( $vulnerability_data ) ) {

		$url      = WPVULNERABILITY_API_HOST . 'php/' . $version . '/';
		$response = wp_remote_get( $url, array( 'timeout' => 2500 ) );

		if ( ! is_wp_error( $response ) ) {

			$body = wp_remote_retrieve_body( $response );
			if ( is_multisite() ) {
				set_site_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_site_transient( $key );
			} else {
				set_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_transient( $key );
			}
		}
	}

	// If the response does not contain vulnerabilities, return false.
	$response = json_decode( $vulnerability_data, true );

	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Process each vulnerability.
	foreach ( $response['data']['vulnerability'] as $v ) {

		// Check if the version falls within the min and max operator range.
		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}

			// Check if the version is below the max operator.
		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['max_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}

			// Check if the version is above the min operator.
		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}
		}
	}

	return $vulnerability;
}

/**
 * Get vulnerabilities for a specific Apache HTTPD version.
 *
 * This function retrieves vulnerability data for a specified Apache HTTPD version.
 * It supports caching to minimize API requests and improve performance.
 *
 * @since 3.2.0
 *
 * @param string $version Apache Version.
 * @param int    $cache   Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false Returns an array of vulnerabilities, or false if there are none.
 */
function wpvulnerability_get_apache( $version, $cache = 1 ) {

	$key                = 'wpvulnerability_apache';
	$vulnerability_data = null;
	$vulnerability      = array();

	// Get cached statistics if available.
	if ( $cache ) {
		if ( is_multisite() ) {
			$vulnerability_data = get_site_transient( $key );
		} else {
			$vulnerability_data = get_transient( $key );
		}
	}

	// If cached statistics are not available, retrieve them from the API and store them in cache.
	if ( empty( $vulnerability_data ) ) {

		$url      = WPVULNERABILITY_API_HOST . 'apache/' . $version . '/';
		$response = wp_remote_get( $url, array( 'timeout' => 2500 ) );

		if ( ! is_wp_error( $response ) ) {

			$body = wp_remote_retrieve_body( $response );
			if ( is_multisite() ) {
				set_site_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_site_transient( $key );
			} else {
				set_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_transient( $key );
			}
		}
	}

	// If the response does not contain vulnerabilities, return false.
	$response = json_decode( $vulnerability_data, true );

	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Process each vulnerability.
	foreach ( $response['data']['vulnerability'] as $v ) {

		// Check if the version falls within the min and max operator range.
		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_version'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}

			// Check if the version is below the max operator.
		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['max_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}

			// Check if the version is above the min operator.
		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_version'] ) . $v['operator']['min_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}
		}
	}

	return $vulnerability;
}

/**
 * Get vulnerabilities for a specific nginx version.
 *
 * This function retrieves vulnerability data for a specified nginx version.
 * It supports caching to minimize API requests and improve performance.
 *
 * @since 3.2.0
 *
 * @param string $version nginx Version.
 * @param int    $cache   Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false Returns an array of vulnerabilities, or false if there are none.
 */
function wpvulnerability_get_nginx( $version, $cache = 1 ) {

	$key                = 'wpvulnerability_nginx';
	$vulnerability_data = null;
	$vulnerability      = array();

	// Get cached statistics if available.
	if ( $cache ) {
		if ( is_multisite() ) {
			$vulnerability_data = get_site_transient( $key );
		} else {
			$vulnerability_data = get_transient( $key );
		}
	}

	// If cached statistics are not available, retrieve them from the API and store them in cache.
	if ( empty( $vulnerability_data ) ) {

		$url      = WPVULNERABILITY_API_HOST . 'nginx/' . $version . '/';
		$response = wp_remote_get( $url, array( 'timeout' => 2500 ) );

		if ( ! is_wp_error( $response ) ) {

			$body = wp_remote_retrieve_body( $response );
			if ( is_multisite() ) {
				set_site_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_site_transient( $key );
			} else {
				set_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_transient( $key );
			}
		}
	}

	// If the response does not contain vulnerabilities, return false.
	$response = json_decode( $vulnerability_data, true );

	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Process each vulnerability.
	foreach ( $response['data']['vulnerability'] as $v ) {

		// Check if the version falls within the min and max operator range.
		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}

			// Check if the version is below the max operator.
		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['max_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}

			// Check if the version is above the min operator.
		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}
		}
	}

	return $vulnerability;
}

/**
 * Retrieve vulnerabilities for a specific MariaDB version.
 *
 * This function fetches vulnerability data for a specified MariaDB version.
 * It supports caching to minimize API requests and improve performance.
 *
 * @since 3.4.0
 *
 * @param string $version The MariaDB version to check for vulnerabilities.
 * @param int    $cache   Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false Returns an array of vulnerabilities, or false if none are found.
 */
function wpvulnerability_get_mariadb( $version, $cache = 1 ) {

	$key                = 'wpvulnerability_mariadb';
	$vulnerability_data = null;
	$vulnerability      = array();

	// Retrieve cached vulnerability data if available.
	if ( $cache ) {
		if ( is_multisite() ) {
			$vulnerability_data = get_site_transient( $key );
		} else {
			$vulnerability_data = get_transient( $key );
		}
	}

	// If cached data is not available, fetch it from the API and cache it.
	if ( empty( $vulnerability_data ) ) {

		$url      = WPVULNERABILITY_API_HOST . 'mariadb/' . $version . '/';
		$response = wp_remote_get( $url, array( 'timeout' => 2500 ) );

		if ( ! is_wp_error( $response ) ) {

			$body = wp_remote_retrieve_body( $response );
			if ( is_multisite() ) {
				set_site_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_site_transient( $key );
			} else {
				set_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_transient( $key );
			}
		}
	}

	// If the API response does not contain vulnerabilities, return false.
	$response = json_decode( $vulnerability_data, true );

	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Process and compile the list of vulnerabilities.
	foreach ( $response['data']['vulnerability'] as $v ) {

		// Check if the version falls within the specified min and max operator range.
		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}

			// Check if the version is below the max operator.
		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['max_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}

			// Check if the version is above the min operator.
		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}
		}
	}

	return $vulnerability;
}

/**
 * Retrieve vulnerabilities for a specific MySQL version.
 *
 * This function fetches vulnerability data for a specified MySQL version.
 * It supports caching to minimize API requests and improve performance.
 *
 * @since 3.4.0
 *
 * @param string $version The MySQL version to check for vulnerabilities.
 * @param int    $cache   Optional. Whether to use cache. Default is 1 (true).
 *
 * @return array|false Returns an array of vulnerabilities, or false if none are found.
 */
function wpvulnerability_get_mysql( $version, $cache = 1 ) {

	$key                = 'wpvulnerability_mysql';
	$vulnerability_data = null;
	$vulnerability      = array();

	// Retrieve cached vulnerability data if available.
	if ( $cache ) {
		if ( is_multisite() ) {
			$vulnerability_data = get_site_transient( $key );
		} else {
			$vulnerability_data = get_transient( $key );
		}
	}

	// If cached data is not available, fetch it from the API and cache it.
	if ( empty( $vulnerability_data ) ) {

		$url      = WPVULNERABILITY_API_HOST . 'mysql/' . $version . '/';
		$response = wp_remote_get( $url, array( 'timeout' => 2500 ) );

		if ( ! is_wp_error( $response ) ) {

			$body = wp_remote_retrieve_body( $response );
			if ( is_multisite() ) {
				set_site_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_site_transient( $key );
			} else {
				set_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_transient( $key );
			}
		}
	}

	// If the API response does not contain vulnerabilities, return false.
	$response = json_decode( $vulnerability_data, true );

	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Process and compile the list of vulnerabilities.
	foreach ( $response['data']['vulnerability'] as $v ) {

		// Check if the version falls within the specified min and max operator range.
		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}

			// Check if the version is below the max operator.
		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['max_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}

			// Check if the version is above the min operator.
		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( (string) $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'], 'strip' ),
					'version'  => wp_kses( (string) $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}
		}
	}

	return $vulnerability;
}
