<?php
/**
 * CLI functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Registers a WP-CLI command for WPVulnerability plugin.
 *
 * @since 2.0.0
 *
 * @return void
 */
if ( defined( 'WP_CLI' ) && WP_CLI ) {

	/**
	 * Core section in WP-CLI command
	 *
	 * @since 2.0.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output, either 'table' or 'json'.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_core( $args, $format ) {

		// Validate the format.
		switch ( $format ) {
			case 'table':
				$format = 'table';
				break;
			case 'json':
				$format = 'json';
				break;
			default:
				WP_CLI::error( "'$format' is not a valid format.\nAvailable formats: table, json" );
				break;
		}

		$core_vulnerabilities = array();
		if ( wpvulnerability_analyze_filter( 'core' ) ) {

			// Get core vulnerabilities.
			$core_vulnerabilities = wpvulnerability_core_get_vulnerabilities();

		}

		$core_complete   = array();
		$vulnerabilities = array();

		if ( $core_vulnerabilities && is_array( $core_vulnerabilities ) ) {

			// Loop through each core vulnerability.
			foreach ( $core_vulnerabilities as $vulnerability ) {

				$core_complete_temp = array();

				// Process theme name and slug.
				$core_complete_temp['version'] = trim( html_entity_decode( wp_kses( (string) $vulnerability['name'], 'strip' ) ) );

				// Process vulnerability severity.
				$core_complete_temp['severity'] = null;
				if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
					$core_complete_temp['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
				}

				// Process CWE details.
				$core_complete_temp['cwe'] = array();
				if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {
					foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
						$core_complete_temp['cwe'][] = array(
							'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) ) ),
							'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['description'], 'strip' ) ) ),
						);
					}
				}

				// Process CVSS score.
				$core_complete_temp['score'] = null;
				if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
					$core_complete_temp['score'] = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
				}

				// Process vulnerability sources.
				$core_complete_temp['source'] = array();
				if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {
					foreach ( $vulnerability['source'] as $vulnerability_source ) {
						$core_complete_temp['source'][] = array(
							'name' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['name'], 'strip' ) ) ),
							'link' => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
						);
					}
				}

				$core_complete[] = $core_complete_temp;
				unset( $core_complete_temp );
			}
		}

		// Format output based on the selected format.
		if ( 'table' === $format ) {

			foreach ( $core_complete as $c_vuln ) {

				$v_version = $c_vuln['version'];

				$v_score    = $c_vuln['score'];
				$v_severity = $c_vuln['severity'];

				// Compile CWE descriptions.
				$v_description_array = array();
				foreach ( $c_vuln['cwe'] as $c_cwe ) {
					if ( isset( $c_cwe['name'] ) ) {
						$v_description_array[] = $c_cwe['name'];
					}
				}
				$v_description = trim( implode( ' + ', $v_description_array ) );

				// Add to vulnerabilities array for table output.
				array_push(
					$vulnerabilities,
					array(
						'version'     => $v_version,
						'score'       => $v_score,
						'severity'    => $v_severity,
						'description' => $v_description,
					)
				);
			}

			// Format and output the vulnerabilities in a table.
			WP_CLI\Utils\format_items(
				'table',
				$vulnerabilities,
				array( 'version', 'score', 'severity', 'description' )
			);

		} elseif ( 'json' === $format ) {
			// Format and output the vulnerabilities in JSON.
			echo wp_json_encode( $core_complete );
		}
	}

	/**
	 * Plugin section in WP-CLI command
	 *
	 * This function retrieves and displays plugin vulnerabilities
	 * based on the specified format (table or JSON).
	 *
	 * @since 2.0.0
	 *
	 * @param array  $args   The arguments passed from the command line.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_plugins( $args, $format ) {

		// Validate the format.
		switch ( $format ) {
			case 'table':
				$format = 'table';
				break;
			case 'json':
				$format = 'json';
				break;
			default:
				WP_CLI::error( "'$format' is not a valid format.\nAvailable formats: table, json" );
				break;
		}

		$plugin_vulnerabilities = array();
		if ( wpvulnerability_analyze_filter( 'plugins' ) ) {

			// Get plugin vulnerabilities.
			$plugin_vulnerabilities = wpvulnerability_plugin_get_vulnerabilities();

		}

		$plugins_complete = array();
		$vulnerabilities  = array();

		// Loop through each plugin vulnerability.
		foreach ( $plugin_vulnerabilities as $plugin ) {

			if ( 1 === $plugin['vulnerable'] ) {

				$plugins_complete_temp                 = array();
				$plugins_complete_temp_vulnerabilities = array();

				// Process theme name and slug.
				$plugins_complete_temp['name'] = trim( html_entity_decode( wp_kses( (string) $plugin['Name'], 'strip' ) ) );
				$plugins_complete_temp['slug'] = trim( html_entity_decode( wp_kses( (string) $plugin['slug'], 'strip' ) ) );

				// Prepare the vulnerabilities array for table format output.
				foreach ( $plugin['vulnerabilities'] as $vulnerability ) {

					// Process vulnerability severity.
					$plugins_complete_temp_vulnerabilities['severity'] = null;
					if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
						$plugins_complete_temp_vulnerabilities['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
					}

					// Process vulnerability details.
					$plugins_complete_temp_vulnerabilities['version'] = trim( html_entity_decode( wp_kses( (string) $vulnerability['versions'], 'strip' ) ) );
					$plugins_complete_temp_vulnerabilities['name']    = trim( html_entity_decode( wp_kses( (string) $vulnerability['name'], 'strip' ) ) );
					$plugins_complete_temp_vulnerabilities['closed']  = (int) $vulnerability['closed'];
					$plugins_complete_temp_vulnerabilities['unfixed'] = (int) $vulnerability['unfixed'];

					// Process CWE details.
					$plugins_complete_temp_vulnerabilities['cwe'] = array();
					if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {
						foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
							$plugins_complete_temp_vulnerabilities['cwe'][] = array(
								'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) ) ),
								'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['description'], 'strip' ) ) ),
							);
						}
					}

					// Process CVSS score.
					$plugins_complete_temp_vulnerabilities['score'] = null;
					if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
						$plugins_complete_temp_vulnerabilities['score'] = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
					}
					$plugins_complete_temp_vulnerabilities['severity'] = null;
					if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
						$plugins_complete_temp_vulnerabilities['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
					}

					// Process vulnerability sources.
					$plugins_complete_temp_vulnerabilities['source'] = array();
					if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {
						foreach ( $vulnerability['source'] as $vulnerability_source ) {
							$plugins_complete_temp_vulnerabilities['source'][] = array(
								'name' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['name'], 'strip' ) ) ),
								'link' => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
							);
						}
					}

					$plugins_complete_temp['vulnerabilities'][] = $plugins_complete_temp_vulnerabilities;
					unset( $plugins_complete_temp_vulnerabilities, $vulnerability );
				}

				$plugins_complete[] = $plugins_complete_temp;
				unset( $plugins_complete_temp );
			}
			unset( $plugin );
		}

		// Format output based on the selected format.
		if ( 'table' === $format ) {

			foreach ( $plugins_complete as $p_vuln ) {
				$v_name = $p_vuln['slug'];

				foreach ( $p_vuln['vulnerabilities'] as $p_vul ) {

					$v_version = $p_vul['version'];

					// Determine if vulnerability is fixed.
					switch ( $p_vul['unfixed'] ) {
						default:
						case 0:
							$v_fixed = 'yes';
							break;
						case 1:
							$v_fixed = 'no';
							break;
					}

					// Determine if theme is closed.
					switch ( $p_vul['closed'] ) {
						default:
						case 0:
							$v_closed = 'no';
							break;
						case 1:
							$v_closed = 'yes';
							break;
					}
					$v_score    = $p_vul['score'];
					$v_severity = $p_vul['severity'];

					// Compile CWE descriptions.
					$v_description_array = array();
					foreach ( $p_vul['cwe'] as $p_cwe ) {
						if ( isset( $p_cwe['name'] ) ) {
							$v_description_array[] = $p_cwe['name'];
						}
					}
					$v_description = trim( implode( ' + ', $v_description_array ) );

					// Add to vulnerabilities array for table output.
					array_push(
						$vulnerabilities,
						array(
							'name'        => $v_name,
							'version'     => $v_version,
							'fixed'       => $v_fixed,
							'closed'      => $v_closed,
							'score'       => $v_score,
							'severity'    => $v_severity,
							'description' => $v_description,
						)
					);
				}
			}

			// Format and output the vulnerabilities in a table.
			WP_CLI\Utils\format_items(
				'table',
				$vulnerabilities,
				array( 'name', 'version', 'fixed', 'closed', 'score', 'severity', 'description' )
			);

		} elseif ( 'json' === $format ) {
			// Format and output the vulnerabilities in a JSON.
			echo wp_json_encode( $plugins_complete );
		}
	}


	/**
	 * Theme section in WP-CLI command
	 *
	 * This function retrieves and displays theme vulnerabilities
	 * based on the specified format (table or JSON).
	 *
	 * @since 2.0.0
	 *
	 * @param array  $args   The arguments passed from the command line.
	 * @param string $format The format for the output.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_themes( $args, $format ) {

		// Validate the format.
		switch ( $format ) {
			case 'table':
				$format = 'table';
				break;
			case 'json':
				$format = 'json';
				break;
			default:
				WP_CLI::error( "'$format' is not a valid format.\nAvailable formats: table, json" );
				break;
		}

		$theme_vulnerabilities = array();
		if ( wpvulnerability_analyze_filter( 'themes' ) ) {
			// Get theme vulnerabilities.
			$theme_vulnerabilities = wpvulnerability_theme_get_vulnerabilities();
		}

		$themes_complete = array();
		$vulnerabilities = array();

		// Loop through each theme vulnerability.
		foreach ( $theme_vulnerabilities as $theme ) {

			if ( 1 === $theme['wpvulnerability']['vulnerable'] ) {

				$themes_complete_temp                 = array();
				$themes_complete_temp_vulnerabilities = array();

				// Process theme name and slug.
				$themes_complete_temp['name'] = trim( html_entity_decode( wp_kses( (string) $theme['wpvulnerability']['name'], 'strip' ) ) );
				$themes_complete_temp['slug'] = trim( html_entity_decode( wp_kses( (string) $theme['wpvulnerability']['slug'], 'strip' ) ) );

				// Prepare the vulnerabilities array for table format output.
				foreach ( $theme['wpvulnerability']['vulnerabilities'] as $vulnerability ) {

					// Process vulnerability severity.
					$themes_complete_temp_vulnerabilities['severity'] = null;
					if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
						$themes_complete_temp_vulnerabilities['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
					}

					// Process vulnerability details.
					$themes_complete_temp_vulnerabilities['version'] = trim( html_entity_decode( wp_kses( (string) $vulnerability['versions'], 'strip' ) ) );
					$themes_complete_temp_vulnerabilities['name']    = trim( html_entity_decode( wp_kses( (string) $vulnerability['name'], 'strip' ) ) );
					$themes_complete_temp_vulnerabilities['closed']  = (int) $vulnerability['closed'];
					$themes_complete_temp_vulnerabilities['unfixed'] = (int) $vulnerability['unfixed'];

					// Process CWE details.
					$themes_complete_temp_vulnerabilities['cwe'] = array();
					if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {
						foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
							$themes_complete_temp_vulnerabilities['cwe'][] = array(
								'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) ) ),
								'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['description'], 'strip' ) ) ),
							);
						}
					}

					// Process CVSS score.
					$themes_complete_temp_vulnerabilities['score'] = null;
					if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
						$themes_complete_temp_vulnerabilities['score'] = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
					}
					$themes_complete_temp_vulnerabilities['severity'] = null;
					if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
						$themes_complete_temp_vulnerabilities['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
					}

					// Process vulnerability sources.
					$themes_complete_temp_vulnerabilities['source'] = array();
					if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {
						foreach ( $vulnerability['source'] as $vulnerability_source ) {
							$themes_complete_temp_vulnerabilities['source'][] = array(
								'name' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['name'], 'strip' ) ) ),
								'link' => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
							);
						}
					}

					$themes_complete_temp['vulnerabilities'][] = $themes_complete_temp_vulnerabilities;
					unset( $themes_complete_temp_vulnerabilities, $vulnerability );
				}

				$themes_complete[] = $themes_complete_temp;
				unset( $themes_complete_temp );
			}
			unset( $theme );
		}

		// Format output based on the selected format.
		if ( 'table' === $format ) {

			foreach ( $themes_complete as $t_vuln ) {
				$v_name = $t_vuln['slug'];

				foreach ( $t_vuln['vulnerabilities'] as $t_vul ) {

					$v_version = $t_vul['version'];

					// Determine if vulnerability is fixed.
					switch ( $t_vul['unfixed'] ) {
						default:
						case 0:
							$v_fixed = 'yes';
							break;
						case 1:
							$v_fixed = 'no';
							break;
					}

					// Determine if theme is closed.
					switch ( $t_vul['closed'] ) {
						default:
						case 0:
							$v_closed = 'no';
							break;
						case 1:
							$v_closed = 'yes';
							break;
					}
					$v_score    = $t_vul['score'];
					$v_severity = $t_vul['severity'];

					// Compile CWE descriptions.
					$v_description_array = array();
					foreach ( $t_vul['cwe'] as $t_cwe ) {
						if ( isset( $t_cwe['name'] ) ) {
							$v_description_array[] = $t_cwe['name'];
						}
					}
					$v_description = trim( implode( ' + ', $v_description_array ) );

					// Add to vulnerabilities array for table output.
					array_push(
						$vulnerabilities,
						array(
							'name'        => $v_name,
							'version'     => $v_version,
							'fixed'       => $v_fixed,
							'closed'      => $v_closed,
							'score'       => $v_score,
							'severity'    => $v_severity,
							'description' => $v_description,
						)
					);
				}
			}

			// Format and output the vulnerabilities in a table.
			WP_CLI\Utils\format_items(
				'table',
				$vulnerabilities,
				array( 'name', 'version', 'fixed', 'closed', 'score', 'severity', 'description' )
			);

		} elseif ( 'json' === $format ) {
			// Format and output the vulnerabilities in a JSON.
			echo wp_json_encode( $themes_complete );
		}
	}

	/**
	 * PHP section in WP-CLI command.
	 *
	 * This function handles the PHP vulnerabilities section in the WP-CLI command.
	 * It validates the format (table or json), retrieves PHP vulnerabilities, and
	 * outputs the information in the specified format.
	 *
	 * @since 3.0.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output, either 'table' or 'json'.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_php( $args, $format ) {

		// Validate the format.
		switch ( $format ) {
			case 'table':
				$format = 'table';
				break;
			case 'json':
				$format = 'json';
				break;
			default:
				WP_CLI::error( "'$format' is not a valid format.\nAvailable formats: table, json" );
				break;
		}

		$php_vulnerabilities = array();
		if ( wpvulnerability_analyze_filter( 'php' ) ) {
			// Get PHP vulnerabilities.
			$php_vulnerabilities = wpvulnerability_php_get_vulnerabilities();
		}

		$php_complete    = array();
		$vulnerabilities = array();

		if ( isset( $php_vulnerabilities['vulnerabilities'] ) ) {

			// Loop through each PHP vulnerability.
			foreach ( $php_vulnerabilities['vulnerabilities'] as $php ) {

				$php_complete_temp = array();

				$php_complete_temp['version']  = trim( html_entity_decode( wp_kses( (string) $php['version'], 'strip' ) ) );
				$php_complete_temp['affected'] = trim( html_entity_decode( wp_kses( (string) $php['versions'], 'strip' ) ) );
				$php_complete_temp['unfixed']  = (int) $php['unfixed'];

				// Process vulnerability sources.
				$php_complete_temp['source'] = array();
				if ( isset( $php['source'] ) && count( $php['source'] ) ) {
					foreach ( $php['source'] as $vulnerability_source ) {
						$php_complete_temp['source'][] = array(
							'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['id'], 'strip' ) ) ),
							'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['description'], 'strip' ) ) ),
							'link'        => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
						);
					}
				}

				$php_complete[] = $php_complete_temp;
				unset( $php_complete_temp, $php );

			}
		}

		// Format output based on the selected format.
		if ( 'table' === $format ) {

			foreach ( $php_complete as $p_vuln ) {
				$v_version  = $p_vuln['version'];
				$v_affected = $p_vuln['affected'];

				// Determine if vulnerability is fixed.
				switch ( $p_vuln['unfixed'] ) {
					default:
					case 0:
						$v_fixed = 'yes';
						break;
					case 1:
						$v_fixed = 'no';
						break;
				}

				// Compile CWE descriptions.
				$v_description_array = array();
				foreach ( $p_vuln['source'] as $p_source ) {
					$v_description_array[] = $p_source['name'] . ': ' . $p_source['description'];
				}
				$v_description = trim( implode( ' + ', $v_description_array ) );

				// Add to vulnerabilities array for table output.
				array_push(
					$vulnerabilities,
					array(
						'version'     => $v_version,
						'affected'    => $v_affected,
						'fixed'       => $v_fixed,
						'description' => $v_description,
					)
				);
			}

			// Format and output the vulnerabilities in a table.
			WP_CLI\Utils\format_items(
				'table',
				$vulnerabilities,
				array( 'version', 'affected', 'fixed', 'description' )
			);

		} elseif ( 'json' === $format ) {
			// Format and output the vulnerabilities in JSON.
			echo wp_json_encode( $php_complete );
		}
	}

	/**
	 * Apache HTTPD section in WP-CLI command.
	 *
	 * This function handles the Apache HTTPD vulnerabilities section in the WP-CLI command.
	 * It validates the format (table or json), retrieves Apache HTTPD vulnerabilities, and
	 * outputs the information in the specified format.
	 *
	 * @since 3.2.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output, either 'table' or 'json'.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_apache( $args, $format ) {

		// Validate the format.
		switch ( $format ) {
			case 'table':
				$format = 'table';
				break;
			case 'json':
				$format = 'json';
				break;
			default:
				WP_CLI::error( "'$format' is not a valid format.\nAvailable formats: table, json" );
				break;
		}

		$apache_vulnerabilities = array();
		if ( wpvulnerability_analyze_filter( 'apache' ) ) {
			// Get Apache HTTPD vulnerabilities.
			$apache_vulnerabilities = wpvulnerability_apache_get_vulnerabilities();
		}

		$apache_complete = array();
		$vulnerabilities = array();

		if ( isset( $apache_vulnerabilities['vulnerabilities'] ) ) {

			$webserver = wpvulnerability_detect_webserver();

			if ( isset( $webserver['id'] ) && 'apache' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {

				// Loop through each Apache vulnerability.
				foreach ( $apache_vulnerabilities['vulnerabilities'] as $apache ) {

					$apache_complete_temp = array();

					$apache_complete_temp['version']  = trim( html_entity_decode( wp_kses( (string) $apache['version'], 'strip' ) ) );
					$apache_complete_temp['affected'] = trim( html_entity_decode( wp_kses( (string) $apache['versions'], 'strip' ) ) );
					$apache_complete_temp['unfixed']  = (int) $apache['unfixed'];

					// Process vulnerability sources.
					$apache_complete_temp['source'] = array();
					if ( isset( $apache['source'] ) && count( $apache['source'] ) ) {
						foreach ( $apache['source'] as $vulnerability_source ) {
							$apache_complete_temp['source'][] = array(
								'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['id'], 'strip' ) ) ),
								'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['description'], 'strip' ) ) ),
								'link'        => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
							);
						}
					}

					$apache_complete[] = $apache_complete_temp;
					unset( $apache_complete_temp, $apache );

				}
			}
		}

		// Format output based on the selected format.
		if ( 'table' === $format ) {

			foreach ( $apache_complete as $a_vuln ) {
				$v_version  = $a_vuln['version'];
				$v_affected = $a_vuln['affected'];

				// Determine if vulnerability is fixed.
				switch ( $a_vuln['unfixed'] ) {
					default:
					case 0:
						$v_fixed = 'yes';
						break;
					case 1:
						$v_fixed = 'no';
						break;
				}

				// Compile source descriptions.
				$v_description_array = array();
				foreach ( $a_vuln['source'] as $a_source ) {
					$v_description_array[] = $a_source['name'] . ': ' . $a_source['description'];
				}
				$v_description = trim( implode( ' + ', $v_description_array ) );

				// Add to vulnerabilities array for table output.
				array_push(
					$vulnerabilities,
					array(
						'version'     => $v_version,
						'affected'    => $v_affected,
						'fixed'       => $v_fixed,
						'description' => $v_description,
					)
				);
			}

			// Format and output the vulnerabilities in a table.
			WP_CLI\Utils\format_items(
				'table',
				$vulnerabilities,
				array( 'version', 'affected', 'fixed', 'description' )
			);

		} elseif ( 'json' === $format ) {
			// Format and output the vulnerabilities in JSON.
			echo wp_json_encode( $apache_complete );
		}
	}

	/**
	 * Nginx section in WP-CLI command.
	 *
	 * This function handles the Nginx vulnerabilities section in the WP-CLI command.
	 * It validates the format (table or json), retrieves Nginx vulnerabilities, and
	 * outputs the information in the specified format.
	 *
	 * @since 3.2.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output, either 'table' or 'json'.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_nginx( $args, $format ) {

		// Validate the format.
		switch ( $format ) {
			case 'table':
				$format = 'table';
				break;
			case 'json':
				$format = 'json';
				break;
			default:
				WP_CLI::error( "'$format' is not a valid format.\nAvailable formats: table, json" );
				break;
		}

		$nginx_vulnerabilities = array();
		if ( wpvulnerability_analyze_filter( 'nginx' ) ) {
			// Get nginx vulnerabilities.
			$nginx_vulnerabilities = wpvulnerability_nginx_get_vulnerabilities();
		}

		$nginx_complete  = array();
		$vulnerabilities = array();

		if ( isset( $nginx_vulnerabilities['vulnerabilities'] ) ) {

			$webserver = wpvulnerability_detect_webserver();

			if ( isset( $webserver['id'] ) && 'nginx' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {

				// Loop through each Nginx vulnerability.
				foreach ( $nginx_vulnerabilities['vulnerabilities'] as $nginx ) {

					$nginx_complete_temp = array();

					$nginx_complete_temp['version']  = trim( html_entity_decode( wp_kses( (string) $nginx['version'], 'strip' ) ) );
					$nginx_complete_temp['affected'] = trim( html_entity_decode( wp_kses( (string) $nginx['versions'], 'strip' ) ) );
					$nginx_complete_temp['unfixed']  = (int) $nginx['unfixed'];

					// Process vulnerability sources.
					$nginx_complete_temp['source'] = array();
					if ( isset( $nginx['source'] ) && count( $nginx['source'] ) ) {
						foreach ( $nginx['source'] as $vulnerability_source ) {
							$nginx_complete_temp['source'][] = array(
								'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['id'], 'strip' ) ) ),
								'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['description'], 'strip' ) ) ),
								'link'        => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
							);
						}
					}

					$nginx_complete[] = $nginx_complete_temp;
					unset( $nginx_complete_temp, $nginx );

				}
			}
		}

		// Format output based on the selected format.
		if ( 'table' === $format ) {

			foreach ( $nginx_complete as $n_vuln ) {
				$v_version  = $n_vuln['version'];
				$v_affected = $n_vuln['affected'];

				// Determine if vulnerability is fixed.
				switch ( $n_vuln['unfixed'] ) {
					default:
					case 0:
						$v_fixed = 'yes';
						break;
					case 1:
						$v_fixed = 'no';
						break;
				}

				// Compile source descriptions.
				$v_description_array = array();
				foreach ( $n_vuln['source'] as $n_source ) {
					$v_description_array[] = $n_source['name'] . ': ' . $n_source['description'];
				}
				$v_description = trim( implode( ' + ', $v_description_array ) );

				// Add to vulnerabilities array for table output.
				array_push(
					$vulnerabilities,
					array(
						'version'     => $v_version,
						'affected'    => $v_affected,
						'fixed'       => $v_fixed,
						'description' => $v_description,
					)
				);
			}

			// Format and output the vulnerabilities in a table.
			WP_CLI\Utils\format_items(
				'table',
				$vulnerabilities,
				array( 'version', 'affected', 'fixed', 'description' )
			);

		} elseif ( 'json' === $format ) {
			// Format and output the vulnerabilities in JSON.
			echo wp_json_encode( $nginx_complete );
		}
	}

	/**
	 * Handles the MariaDB section in the WP-CLI command.
	 *
	 * This function manages the output of MariaDB vulnerabilities in the WP-CLI command.
	 * It validates the output format (either 'table' or 'json'), retrieves MariaDB vulnerabilities,
	 * and displays the information in the specified format.
	 *
	 * @since 3.4.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output, either 'table' or 'json'.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_mariadb( $args, $format ) {

		// Validate the format.
		switch ( $format ) {
			case 'table':
			case 'json':
				break;
			default:
				WP_CLI::error( "'$format' is not a valid format.\nAvailable formats: table, json" );
				break;
		}

		$mariadb_vulnerabilities = array();
		if ( wpvulnerability_analyze_filter( 'mariadb' ) ) {
			// Get MariaDB vulnerabilities.
			$mariadb_vulnerabilities = wpvulnerability_mariadb_get_vulnerabilities();
		}

		$mariadb_complete = array();
		$vulnerabilities  = array();

		if ( isset( $mariadb_vulnerabilities['vulnerabilities'] ) ) {

			$webserver = wpvulnerability_detect_webserver();

			if ( isset( $webserver['id'] ) && 'mariadb' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {

				// Loop through each MariaDB vulnerability.
				foreach ( $mariadb_vulnerabilities['vulnerabilities'] as $mariadb ) {

					$mariadb_complete_temp = array();

					$mariadb_complete_temp['version']  = trim( html_entity_decode( wp_kses( (string) $mariadb['version'], 'strip' ) ) );
					$mariadb_complete_temp['affected'] = trim( html_entity_decode( wp_kses( (string) $mariadb['versions'], 'strip' ) ) );
					$mariadb_complete_temp['unfixed']  = (int) $mariadb['unfixed'];

					// Process vulnerability sources.
					$mariadb_complete_temp['source'] = array();
					if ( isset( $mariadb['source'] ) && count( $mariadb['source'] ) ) {
						foreach ( $mariadb['source'] as $vulnerability_source ) {
							$mariadb_complete_temp['source'][] = array(
								'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['id'], 'strip' ) ) ),
								'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['description'], 'strip' ) ) ),
								'link'        => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
							);
						}
					}

					$mariadb_complete[] = $mariadb_complete_temp;
					unset( $mariadb_complete_temp, $mariadb );

				}
			}
		}

		// Format output based on the selected format.
		if ( 'table' === $format ) {

			foreach ( $mariadb_complete as $n_vuln ) {
				$v_version  = $n_vuln['version'];
				$v_affected = $n_vuln['affected'];

				// Determine if the vulnerability is fixed.
				$v_fixed = $n_vuln['unfixed'] ? 'no' : 'yes';

				// Compile source descriptions.
				$v_description_array = array();
				foreach ( $n_vuln['source'] as $n_source ) {
					$v_description_array[] = $n_source['name'] . ': ' . $n_source['description'];
				}
				$v_description = trim( implode( ' + ', $v_description_array ) );

				// Add to vulnerabilities array for table output.
				$vulnerabilities[] = array(
					'version'     => $v_version,
					'affected'    => $v_affected,
					'fixed'       => $v_fixed,
					'description' => $v_description,
				);
			}

			// Format and output the vulnerabilities in a table.
			WP_CLI\Utils\format_items(
				'table',
				$vulnerabilities,
				array( 'version', 'affected', 'fixed', 'description' )
			);

		} elseif ( 'json' === $format ) {
			// Format and output the vulnerabilities in JSON.
			echo wp_json_encode( $mariadb_complete );
		}
	}

	/**
	 * Handles the MySQL section in the WP-CLI command.
	 *
	 * This function manages the output of MySQL vulnerabilities in the WP-CLI command.
	 * It validates the output format (either 'table' or 'json'), retrieves MySQL vulnerabilities,
	 * and displays the information in the specified format.
	 *
	 * @since 3.4.0
	 *
	 * @param array  $args   Arguments passed to the command.
	 * @param string $format The format for the output, either 'table' or 'json'.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_mysql( $args, $format ) {

		// Validate the format.
		switch ( $format ) {
			case 'table':
			case 'json':
				break;
			default:
				WP_CLI::error( "'$format' is not a valid format.\nAvailable formats: table, json" );
				break;
		}

		$mysql_vulnerabilities = array();
		if ( wpvulnerability_analyze_filter( 'mysql' ) ) {
			// Get MySQL vulnerabilities.
			$mysql_vulnerabilities = wpvulnerability_mysql_get_vulnerabilities();
		}

		$mysql_complete  = array();
		$vulnerabilities = array();

		if ( isset( $mysql_vulnerabilities['vulnerabilities'] ) ) {

			$webserver = wpvulnerability_detect_webserver();

			if ( isset( $webserver['id'] ) && 'mysql' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {

				// Loop through each MySQL vulnerability.
				foreach ( $mysql_vulnerabilities['vulnerabilities'] as $mysql ) {

					$mysql_complete_temp = array();

					$mysql_complete_temp['version']  = trim( html_entity_decode( wp_kses( (string) $mysql['version'], 'strip' ) ) );
					$mysql_complete_temp['affected'] = trim( html_entity_decode( wp_kses( (string) $mysql['versions'], 'strip' ) ) );
					$mysql_complete_temp['unfixed']  = (int) $mysql['unfixed'];

					// Process vulnerability sources.
					$mysql_complete_temp['source'] = array();
					if ( isset( $mysql['source'] ) && count( $mysql['source'] ) ) {
						foreach ( $mysql['source'] as $vulnerability_source ) {
							$mysql_complete_temp['source'][] = array(
								'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['id'], 'strip' ) ) ),
								'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['description'], 'strip' ) ) ),
								'link'        => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
							);
						}
					}

					$mysql_complete[] = $mysql_complete_temp;
					unset( $mysql_complete_temp, $mysql );

				}
			}
		}

		// Format output based on the selected format.
		if ( 'table' === $format ) {

			foreach ( $mysql_complete as $n_vuln ) {
				$v_version  = $n_vuln['version'];
				$v_affected = $n_vuln['affected'];

				// Determine if the vulnerability is fixed.
				$v_fixed = $n_vuln['unfixed'] ? 'no' : 'yes';

				// Compile source descriptions.
				$v_description_array = array();
				foreach ( $n_vuln['source'] as $n_source ) {
					$v_description_array[] = $n_source['name'] . ': ' . $n_source['description'];
				}
				$v_description = trim( implode( ' + ', $v_description_array ) );

				// Add to vulnerabilities array for table output.
				$vulnerabilities[] = array(
					'version'     => $v_version,
					'affected'    => $v_affected,
					'fixed'       => $v_fixed,
					'description' => $v_description,
				);
			}

			// Format and output the vulnerabilities in a table.
			WP_CLI\Utils\format_items(
				'table',
				$vulnerabilities,
				array( 'version', 'affected', 'fixed', 'description' )
			);

		} elseif ( 'json' === $format ) {
			// Format and output the vulnerabilities in JSON.
			echo wp_json_encode( $mysql_complete );
		}
	}

	/**
	 * Dispatches the WPVulnerability CLI command to show the list of detected vulnerabilities on the site.
	 *
	 * This function selects and executes the appropriate function based on the provided subcommand.
	 * It supports different output formats, such as 'table' and 'json', to display the vulnerabilities.
	 *
	 * @since 2.0.0
	 *
	 * @param array $args       The subcommand to execute.
	 *                          Accepted values: 'core', 'plugins', 'themes', 'php', 'apache', 'nginx', 'mariadb', 'mysql'.
	 * @param array $assoc_args Associative arguments passed from the command line.
	 *                          'format' (optional) - The format for the output. Defaults to 'table'.
	 *                          Accepted values: 'table', 'json'.
	 *
	 * @return void
	 */
	function wpvulnerability_cli_command( $args, $assoc_args ) {

		$subcommand = $args[0];
		$format     = isset( $assoc_args['format'] ) ? $assoc_args['format'] : 'table';

		// Select the correct function to execute based on the subcommand.
		switch ( $subcommand ) {
			case 'core':
				wpvulnerability_cli_core( $args, $format );
				break;
			case 'plugins':
				wpvulnerability_cli_plugins( $args, $format );
				break;
			case 'themes':
				wpvulnerability_cli_themes( $args, $format );
				break;
			case 'php':
				wpvulnerability_cli_php( $args, $format );
				break;
			case 'apache':
				wpvulnerability_cli_apache( $args, $format );
				break;
			case 'nginx':
				wpvulnerability_cli_nginx( $args, $format );
				break;
			case 'mariadb':
				wpvulnerability_cli_mariadb( $args, $format );
				break;
			case 'mysql':
				wpvulnerability_cli_mysql( $args, $format );
				break;
			default:
				// Display an error message for an invalid subcommand.
				WP_CLI::error( "'$subcommand' is not a valid subcommand of 'wpvulnerability'.\nAvailable subcommands: core, plugins, themes, php, apache, nginx, mariadb, mysql" );
				break;
		}
	}

	/**
	 * Registers a WP-CLI command to show the list of vulnerabilities detected on your site.
	 *
	 * EXAMPLES:
	 *
	 *  - wp wpvulnerability core
	 *  - wp wpvulnerability plugins
	 *  - wp wpvulnerability themes
	 *  - wp wpvulnerability php
	 *  - wp wpvulnerability apache
	 *  - wp wpvulnerability nginx
	 *  - wp wpvulnerability mariadb
	 *  - wp wpvulnerability mysql
	 *
	 * @param object $args Arguments passed from the command line.
	 *
	 * @return void
	 */
	WP_CLI::add_command(
		'wpvulnerability',
		'wpvulnerability_cli_command',
		array(
			'shortdesc' => 'Show the list of vulnerabilities detected on your site.',
			'synopsis'  => array(
				array(
					'type'        => 'positional',
					'name'        => 'subcommand',
					'description' => 'Type of vulnerability [ core | plugins | themes | php | apache | nginx | mariadb | mysql ].',
					'optional'    => false,
				),
				array(
					'type'        => 'assoc',
					'name'        => 'format',
					'description' => 'Format for the output [ table | json ].',
					'optional'    => true,
					'default'     => 'table',
				),
			),
			'when'      => 'after_wp_load',
			'longdesc'  => "EXAMPLES:\n\n - wp wpvulnerability core\n - wp wpvulnerability plugins\n - wp wpvulnerability themes\n - wp wpvulnerability php\n - wp wpvulnerability apache\n - wp wpvulnerability nginx\n - wp wpvulnerability mariadb\n - wp wpvulnerability mysql",
		)
	);

}
