<?php
/**
 * WPVulnerability REST API Endpoints
 *
 * @package WPVulnerability
 *
 * @since 3.3.0
 */

/**
 * Handle the core vulnerabilities REST API request.
 *
 * This function handles the request for retrieving core vulnerabilities.
 * It includes the necessary file and fetches the vulnerabilities data.
 *
 * @since 3.3.0
 *
 * @return WP_REST_Response Core vulnerabilities data or a message if none found.
 */
function wpvulnerability_rest_core_vulnerabilities() {

	// Include the files containing the functions to get core vulnerabilities.
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-general.php';
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-core.php';

	// Get the core vulnerabilities.
	$core_vulnerabilities = wpvulnerability_core_get_vulnerabilities();

	$core_complete   = array();
	$vulnerabilities = array();

	// Check if vulnerabilities are found and is an array.
	if ( $core_vulnerabilities && is_array( $core_vulnerabilities ) ) {

		// Loop through each core vulnerability.
		foreach ( $core_vulnerabilities as $vulnerability ) {

			$core_complete_temp = array();

			// Process vulnerability version.
			$core_complete_temp['version'] = trim( html_entity_decode( wp_kses( (string) $vulnerability['name'], 'strip' ) ) );

			// Process vulnerability severity.
			$core_complete_temp['severity'] = null;
			if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
				$core_complete_temp['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
			}

			// Process CWE details.
			$core_complete_temp['cwe'] = array();
			if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {
				foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
					$core_complete_temp['cwe'][] = array(
						'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) ) ),
						'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['description'], 'strip' ) ) ),
					);
				}
			}

			// Process CVSS score.
			$core_complete_temp['score'] = null;
			if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
				$core_complete_temp['score'] = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
			}

			// Process vulnerability sources.
			$core_complete_temp['source'] = array();
			if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {
				foreach ( $vulnerability['source'] as $vulnerability_source ) {
					$core_complete_temp['source'][] = array(
						'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['name'], 'strip' ) ) ),
						'link'        => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
						'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['description'], 'strip' ) ) ),
					);
				}
			}

			$core_complete[] = $core_complete_temp;
			unset( $core_complete_temp );
		}
	}

	// Return the vulnerabilities in the response.
	return new WP_REST_Response( $core_complete, 200 );
}

/**
 * Handle the plugins vulnerabilities REST API request.
 *
 * This function handles the request for retrieving plugins vulnerabilities.
 * It includes the necessary file and fetches the vulnerabilities data.
 *
 * @since 3.3.0
 *
 * @return WP_REST_Response Plugins vulnerabilities data or a message if none found.
 */
function wpvulnerability_rest_plugins_vulnerabilities() {

	// Include the files containing the functions to get plugins vulnerabilities.
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-general.php';
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-plugins.php';

	// Get the plugins vulnerabilities.
	$plugins_vulnerabilities = wpvulnerability_plugin_get_vulnerabilities();

	$plugins_complete = array();

	// Loop through each plugin vulnerability.
	foreach ( $plugins_vulnerabilities as $plugin ) {

		// Check if the plugin is vulnerable.
		if ( 1 === $plugin['vulnerable'] ) {

			$plugins_complete_temp                 = array();
			$plugins_complete_temp_vulnerabilities = array();

			// Process plugin name and slug.
			$plugins_complete_temp['name'] = trim( html_entity_decode( wp_kses( (string) $plugin['Name'], 'strip' ) ) );
			$plugins_complete_temp['slug'] = trim( html_entity_decode( wp_kses( (string) $plugin['slug'], 'strip' ) ) );

			// Prepare the vulnerabilities array for table format output.
			foreach ( $plugin['vulnerabilities'] as $vulnerability ) {

				// Process vulnerability severity.
				$plugins_complete_temp_vulnerabilities['severity'] = null;
				if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
					$plugins_complete_temp_vulnerabilities['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
				}

				// Process vulnerability details.
				$plugins_complete_temp_vulnerabilities['version']  = trim( html_entity_decode( wp_kses( (string) $vulnerability['version'], 'strip' ) ) );
				$plugins_complete_temp_vulnerabilities['affected'] = trim( html_entity_decode( wp_kses( (string) $vulnerability['versions'], 'strip' ) ) );
				$plugins_complete_temp_vulnerabilities['name']     = trim( html_entity_decode( wp_kses( (string) $vulnerability['name'], 'strip' ) ) );
				$plugins_complete_temp_vulnerabilities['closed']   = (int) $vulnerability['closed'];
				$plugins_complete_temp_vulnerabilities['unfixed']  = (int) $vulnerability['unfixed'];

				// Process CWE details.
				$plugins_complete_temp_vulnerabilities['cwe'] = array();
				if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {
					foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
						$plugins_complete_temp_vulnerabilities['cwe'][] = array(
							'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) ) ),
							'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['description'], 'strip' ) ) ),
						);
					}
				}

				// Process CVSS score.
				$plugins_complete_temp_vulnerabilities['score'] = null;
				if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
					$plugins_complete_temp_vulnerabilities['score'] = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
				}
				$plugins_complete_temp_vulnerabilities['severity'] = null;
				if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
					$plugins_complete_temp_vulnerabilities['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
				}

				// Process vulnerability sources.
				$plugins_complete_temp_vulnerabilities['source'] = array();
				if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {
					foreach ( $vulnerability['source'] as $vulnerability_source ) {
						$plugins_complete_temp_vulnerabilities['source'][] = array(
							'name' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['name'], 'strip' ) ) ),
							'link' => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
						);
					}
				}

				// Add processed vulnerability to the temporary array.
				$plugins_complete_temp['vulnerabilities'][] = $plugins_complete_temp_vulnerabilities;
				unset( $plugins_complete_temp_vulnerabilities, $vulnerability );
			}

			// Add processed plugin data to the complete array.
			$plugins_complete[] = $plugins_complete_temp;
			unset( $plugins_complete_temp );
		}
		unset( $plugin );
	}

	// Return the vulnerabilities in the response.
	return new WP_REST_Response( $plugins_complete, 200 );
}

/**
 * Handle the themes vulnerabilities REST API request.
 *
 * This function handles the request for retrieving themes vulnerabilities.
 * It includes the necessary file and fetches the vulnerabilities data.
 *
 * @since 3.3.0
 *
 * @return WP_REST_Response Themes vulnerabilities data or a message if none found.
 */
function wpvulnerability_rest_themes_vulnerabilities() {

	// Include the file containing the function to get themes vulnerabilities.
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-general.php';
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-themes.php';

	// Get the themes vulnerabilities.
	$themes_vulnerabilities = wpvulnerability_theme_get_vulnerabilities();

	$themes_complete = array();

	// Loop through each theme vulnerability.
	foreach ( $themes_vulnerabilities as $theme ) {

		// Check if the theme is vulnerable.
		if ( 1 === $theme['wpvulnerability']['vulnerable'] ) {

			$themes_complete_temp                 = array();
			$themes_complete_temp_vulnerabilities = array();

			// Process theme name and slug.
			$themes_complete_temp['name'] = trim( html_entity_decode( wp_kses( (string) $theme['wpvulnerability']['name'], 'strip' ) ) );
			$themes_complete_temp['slug'] = trim( html_entity_decode( wp_kses( (string) $theme['wpvulnerability']['slug'], 'strip' ) ) );

			// Prepare the vulnerabilities array for table format output.
			foreach ( $theme['wpvulnerability']['vulnerabilities'] as $vulnerability ) {

				// Process vulnerability severity.
				$themes_complete_temp_vulnerabilities['severity'] = null;
				if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
					$themes_complete_temp_vulnerabilities['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
				}

				// Process vulnerability details.
				$themes_complete_temp_vulnerabilities['version']  = trim( html_entity_decode( wp_kses( (string) $vulnerability['version'], 'strip' ) ) );
				$themes_complete_temp_vulnerabilities['affected'] = trim( html_entity_decode( wp_kses( (string) $vulnerability['versions'], 'strip' ) ) );
				$themes_complete_temp_vulnerabilities['name']     = trim( html_entity_decode( wp_kses( (string) $vulnerability['name'], 'strip' ) ) );
				$themes_complete_temp_vulnerabilities['closed']   = (int) $vulnerability['closed'];
				$themes_complete_temp_vulnerabilities['unfixed']  = (int) $vulnerability['unfixed'];

				// Process CWE details.
				$themes_complete_temp_vulnerabilities['cwe'] = array();
				if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {
					foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
						$themes_complete_temp_vulnerabilities['cwe'][] = array(
							'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) ) ),
							'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_cwe['description'], 'strip' ) ) ),
						);
					}
				}

				// Process CVSS score.
				$themes_complete_temp_vulnerabilities['score'] = null;
				if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
					$themes_complete_temp_vulnerabilities['score'] = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
				}
				$themes_complete_temp_vulnerabilities['severity'] = null;
				if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
					$themes_complete_temp_vulnerabilities['severity'] = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
				}

				// Process vulnerability sources.
				$themes_complete_temp_vulnerabilities['source'] = array();
				if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {
					foreach ( $vulnerability['source'] as $vulnerability_source ) {
						$themes_complete_temp_vulnerabilities['source'][] = array(
							'name' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['name'], 'strip' ) ) ),
							'link' => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
						);
					}
				}

				// Add processed vulnerability to the temporary array.
				$themes_complete_temp['vulnerabilities'][] = $themes_complete_temp_vulnerabilities;
				unset( $themes_complete_temp_vulnerabilities, $vulnerability );
			}

			// Add processed theme data to the complete array.
			$themes_complete[] = $themes_complete_temp;
			unset( $themes_complete_temp );
		}
		unset( $theme );
	}

	// Return the vulnerabilities in the response.
	return new WP_REST_Response( $themes_complete, 200 );
}

/**
 * Handle the PHP vulnerabilities REST API request.
 *
 * This function handles the request for retrieving PHP vulnerabilities.
 * It includes the necessary file and fetches the vulnerabilities data.
 *
 * @since 3.3.0
 *
 * @return WP_REST_Response PHP vulnerabilities data or a message if none found.
 */
function wpvulnerability_rest_php_vulnerabilities() {

	// Include the file containing the function to get PHP vulnerabilities.
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-general.php';
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-php.php';

	// Get the PHP vulnerabilities.
	$php_vulnerabilities = wpvulnerability_php_get_vulnerabilities();

	$php_complete = array();

	if ( isset( $php_vulnerabilities['vulnerabilities'] ) ) {

		// Loop through each PHP vulnerability.
		foreach ( $php_vulnerabilities['vulnerabilities'] as $php ) {

			$php_complete_temp = array();

			// Process PHP version and affected versions.
			$php_complete_temp['version']  = trim( html_entity_decode( wp_kses( (string) $php['version'], 'strip' ) ) );
			$php_complete_temp['affected'] = trim( html_entity_decode( wp_kses( (string) $php['versions'], 'strip' ) ) );
			$php_complete_temp['unfixed']  = (int) $php['unfixed'];

			// Process vulnerability sources.
			$php_complete_temp['source'] = array();
			if ( isset( $php['source'] ) && count( $php['source'] ) ) {
				foreach ( $php['source'] as $vulnerability_source ) {
					$php_complete_temp['source'][] = array(
						'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['id'], 'strip' ) ) ),
						'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['description'], 'strip' ) ) ),
						'link'        => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
					);
				}
			}

			// Add processed vulnerability to the complete array.
			$php_complete[] = $php_complete_temp;
			unset( $php_complete_temp, $php );

		}
	}

	// Return the vulnerabilities in the response.
	return new WP_REST_Response( $php_complete, 200 );
}

/**
 * Handle the Apache vulnerabilities REST API request.
 *
 * This function handles the request for retrieving Apache vulnerabilities.
 * It includes the necessary file and fetches the vulnerabilities data.
 *
 * @since 3.3.0
 *
 * @return WP_REST_Response Apache vulnerabilities data or a message if none found.
 */
function wpvulnerability_rest_apache_vulnerabilities() {

	// Include the file containing the function to get Apache vulnerabilities.
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-general.php';
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-apache.php';

	// Get the Apache vulnerabilities.
	$apache_vulnerabilities = wpvulnerability_apache_get_vulnerabilities();

	$apache_complete = array();

	if ( isset( $apache_vulnerabilities['vulnerabilities'] ) ) {

		$webserver = wpvulnerability_detect_webserver();

		if ( isset( $webserver['id'] ) && 'apache' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {

			// Loop through each Apache vulnerability.
			foreach ( $apache_vulnerabilities['vulnerabilities'] as $apache ) {

				$apache_complete_temp = array();

				// Process Apache version and affected versions.
				$apache_complete_temp['version']  = trim( html_entity_decode( wp_kses( (string) $apache['version'], 'strip' ) ) );
				$apache_complete_temp['affected'] = trim( html_entity_decode( wp_kses( (string) $apache['versions'], 'strip' ) ) );
				$apache_complete_temp['unfixed']  = (int) $apache['unfixed'];

				// Process vulnerability sources.
				$apache_complete_temp['source'] = array();
				if ( isset( $apache['source'] ) && count( $apache['source'] ) ) {
					foreach ( $apache['source'] as $vulnerability_source ) {
						$apache_complete_temp['source'][] = array(
							'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['id'], 'strip' ) ) ),
							'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['description'], 'strip' ) ) ),
							'link'        => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
						);
					}
				}

				// Add processed vulnerability to the complete array.
				$apache_complete[] = $apache_complete_temp;
				unset( $apache_complete_temp, $apache );

			}
		}
	}

	// Return the vulnerabilities in the response.
	return new WP_REST_Response( $apache_complete, 200 );
}

/**
 * Handle the nginx vulnerabilities REST API request.
 *
 * This function handles the request for retrieving nginx vulnerabilities.
 * It includes the necessary file and fetches the vulnerabilities data.
 *
 * @since 3.3.0
 *
 * @return WP_REST_Response Nginx vulnerabilities data or a message if none found.
 */
function wpvulnerability_rest_nginx_vulnerabilities() {

	// Include the file containing the function to get nginx vulnerabilities.
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-general.php';
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-nginx.php';

	// Get the nginx vulnerabilities.
	$nginx_vulnerabilities = wpvulnerability_nginx_get_vulnerabilities();

	$nginx_complete = array();

	if ( isset( $nginx_vulnerabilities['vulnerabilities'] ) ) {

		$webserver = wpvulnerability_detect_webserver();

		// Check if the webserver is nginx and has a version.
		if ( isset( $webserver['id'] ) && 'nginx' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {

			// Loop through each Nginx vulnerability.
			foreach ( $nginx_vulnerabilities['vulnerabilities'] as $nginx ) {

				$nginx_complete_temp = array();

				// Process nginx version and affected versions.
				$nginx_complete_temp['version']  = trim( html_entity_decode( wp_kses( (string) $nginx['version'], 'strip' ) ) );
				$nginx_complete_temp['affected'] = trim( html_entity_decode( wp_kses( (string) $nginx['versions'], 'strip' ) ) );
				$nginx_complete_temp['unfixed']  = (int) $nginx['unfixed'];

				// Process vulnerability sources.
				$nginx_complete_temp['source'] = array();
				if ( isset( $nginx['source'] ) && count( $nginx['source'] ) ) {
					foreach ( $nginx['source'] as $vulnerability_source ) {
						$nginx_complete_temp['source'][] = array(
							'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['id'], 'strip' ) ) ),
							'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['description'], 'strip' ) ) ),
							'link'        => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
						);
					}
				}

				// Add processed vulnerability to the complete array.
				$nginx_complete[] = $nginx_complete_temp;
				unset( $nginx_complete_temp, $nginx );

			}
		}
	}

	// Return the vulnerabilities in the response.
	return new WP_REST_Response( $nginx_complete, 200 );
}

/**
 * Handles the MariaDB vulnerabilities REST API request.
 *
 * This function processes the request to retrieve MariaDB vulnerabilities.
 * It loads the necessary files and fetches the vulnerability data for MariaDB,
 * then returns the data in a structured format.
 *
 * @since 3.4.0
 *
 * @return WP_REST_Response MariaDB vulnerabilities data or an empty array if none found.
 */
function wpvulnerability_rest_mariadb_vulnerabilities() {

	// Include the files necessary for retrieving MariaDB vulnerabilities.
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-general.php';
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-mariadb.php';

	// Retrieve the MariaDB vulnerabilities.
	$mariadb_vulnerabilities = wpvulnerability_mariadb_get_vulnerabilities();

	$mariadb_complete = array();

	if ( isset( $mariadb_vulnerabilities['vulnerabilities'] ) ) {

		$webserver = wpvulnerability_detect_webserver();

		// Check if the web server is MariaDB and has a version.
		if ( isset( $webserver['id'] ) && 'mariadb' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {

			// Loop through each MariaDB vulnerability.
			foreach ( $mariadb_vulnerabilities['vulnerabilities'] as $mariadb ) {

				$mariadb_complete_temp = array();

				// Process MariaDB version and affected versions.
				$mariadb_complete_temp['version']  = trim( html_entity_decode( wp_kses( (string) $mariadb['version'], 'strip' ) ) );
				$mariadb_complete_temp['affected'] = trim( html_entity_decode( wp_kses( (string) $mariadb['versions'], 'strip' ) ) );
				$mariadb_complete_temp['unfixed']  = (int) $mariadb['unfixed'];

				// Process vulnerability sources.
				$mariadb_complete_temp['source'] = array();
				if ( isset( $mariadb['source'] ) && count( $mariadb['source'] ) ) {
					foreach ( $mariadb['source'] as $vulnerability_source ) {
						$mariadb_complete_temp['source'][] = array(
							'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['id'], 'strip' ) ) ),
							'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['description'], 'strip' ) ) ),
							'link'        => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
						);
					}
				}

				// Add processed vulnerability to the complete array.
				$mariadb_complete[] = $mariadb_complete_temp;
				unset( $mariadb_complete_temp, $mariadb );

			}
		}
	}

	// Return the vulnerabilities in the response.
	return new WP_REST_Response( $mariadb_complete, 200 );
}

/**
 * Handles the MySQL vulnerabilities REST API request.
 *
 * This function processes the request to retrieve MySQL vulnerabilities.
 * It loads the necessary files and fetches the vulnerability data for MySQL,
 * then returns the data in a structured format.
 *
 * @since 3.4.0
 *
 * @return WP_REST_Response MySQL vulnerabilities data or an empty array if none found.
 */
function wpvulnerability_rest_mysql_vulnerabilities() {

	// Include the files necessary for retrieving MySQL vulnerabilities.
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-general.php';
	require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-mysql.php';

	// Retrieve the MySQL vulnerabilities.
	$mysql_vulnerabilities = wpvulnerability_mysql_get_vulnerabilities();

	$mysql_complete = array();

	if ( isset( $mysql_vulnerabilities['vulnerabilities'] ) ) {

		$webserver = wpvulnerability_detect_webserver();

		// Check if the web server is MySQL and has a version.
		if ( isset( $webserver['id'] ) && 'mysql' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {

			// Loop through each MySQL vulnerability.
			foreach ( $mysql_vulnerabilities['vulnerabilities'] as $mysql ) {

				$mysql_complete_temp = array();

				// Process MySQL version and affected versions.
				$mysql_complete_temp['version']  = trim( html_entity_decode( wp_kses( (string) $mysql['version'], 'strip' ) ) );
				$mysql_complete_temp['affected'] = trim( html_entity_decode( wp_kses( (string) $mysql['versions'], 'strip' ) ) );
				$mysql_complete_temp['unfixed']  = (int) $mysql['unfixed'];

				// Process vulnerability sources.
				$mysql_complete_temp['source'] = array();
				if ( isset( $mysql['source'] ) && count( $mysql['source'] ) ) {
					foreach ( $mysql['source'] as $vulnerability_source ) {
						$mysql_complete_temp['source'][] = array(
							'name'        => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['id'], 'strip' ) ) ),
							'description' => trim( html_entity_decode( wp_kses( (string) $vulnerability_source['description'], 'strip' ) ) ),
							'link'        => esc_url_raw( (string) $vulnerability_source['link'], 'strip' ),
						);
					}
				}

				// Add processed vulnerability to the complete array.
				$mysql_complete[] = $mysql_complete_temp;
				unset( $mysql_complete_temp, $mysql );

			}
		}
	}

	// Return the vulnerabilities in the response.
	return new WP_REST_Response( $mysql_complete, 200 );
}

/**
 * Custom permission check for the WPVulnerability REST API.
 *
 * This function checks if the request is authenticated using an Application Password.
 *
 * @since 3.3.0
 *
 * @param WP_REST_Request $request The REST API request.
 *
 * @return bool True if the user has permission, false otherwise.
 */
function wpvulnerability_permission_check( WP_REST_Request $request ) {

	// Check if application passwords are available.
	if ( wp_is_application_passwords_available() ) {
		$authorization_header = $request->get_header( 'authorization' );

		// Check if the authorization header is present and properly formatted.
		if ( $authorization_header && preg_match( '/^Basic\s(.+)$/i', $authorization_header, $matches ) ) {
			$auth_string             = base64_decode( (string) $matches[1] ); // phpcs:ignore
			list( $user, $password ) = explode( ':', $auth_string );

			// Authenticate the user using the application password.
			if ( wp_authenticate_application_password( null, $user, $password ) instanceof WP_User ) {
				return true;
			}
		}
	}

	return false;
}

/**
 * Registers REST API routes for WPVulnerability.
 *
 * This function sets up the REST API routes for WPVulnerability to handle requests
 * related to vulnerabilities in various components like core, plugins, themes, PHP, and more.
 *
 * @since 3.3.0
 *
 * @return void
 */
function wpvulnerability_register_rest_routes() {

	// Define the endpoints to be registered.
	$endpoints = array(
		'core',
		'plugins',
		'themes',
		'php',
		'apache',
		'nginx',
		'mariadb',
		'mysql',
	);

	// Loop through each endpoint and register it.
	foreach ( $endpoints as $endpoint ) {
		register_rest_route(
			'wpvulnerability/v1',            // Namespace and version.
			'/' . $endpoint,                 // Endpoint URL.
			array(
				'methods'             => 'GET',  // HTTP method.
				'callback'            => 'wpvulnerability_rest_' . $endpoint . '_vulnerabilities',  // Callback function.
				'permission_callback' => 'wpvulnerability_permission_check',  // Permission check callback.
			)
		);
	}
}

// Hook to initialize REST API endpoints.
add_action( 'rest_api_init', 'wpvulnerability_register_rest_routes' );
