<?php
/**
 * Process functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Convert vulnerabilities into pretty HTML
 *
 * @version 2.0.0
 *
 * @param string $type Type: core, plugin, theme, php, apache, nginx.
 * @param array  $vulnerabilities Vulnerability data.
 *
 * @return string The HTML representation of vulnerabilities.
 */
function wpvulnerability_html( $type, $vulnerabilities ) {

	$html = '';

	if ( 'plugin' === $type || 'theme' === $type ) {

		foreach ( $vulnerabilities as $vulnerability ) {

			$what = array();
			if ( isset( $vulnerability['impact']['cwe'] ) ) {
				foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
					$what[] = '<div><b>' . wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) . '</b></div><div><i>' . wp_kses_post( (string) $vulnerability_cwe['description'] ) . '</i></div>';
				}
			}
			$sources = array();
			if ( isset( $vulnerability['source'] ) ) {
				foreach ( $vulnerability['source'] as $vulnerability_source ) {
					$sources[] = '<a href="' . esc_url_raw( (string) $vulnerability_source['link'], 'strip' ) . '" target="_blank">[+]</a>&nbsp;' . wp_kses( (string) $vulnerability_source['name'], 'strip' );
				}
			}
			if ( count( $sources ) ) {
				$source = '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>';
			}

			$score = null;
			if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
				$score = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
			}
			$severity = null;
			if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
				$severity = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
			}

			$html .= '<h4>' . wp_kses( (string) $vulnerability['name'], 'strip' ) . '</h4>';
			if ( (int) $vulnerability['closed'] || (int) $vulnerability['unfixed'] ) {
				$html .= '<div style="padding-bottom: 5px;">';
				if ( (int) $vulnerability['closed'] ) {
					$html .= '<div class="text-red">' . __( 'This plugin is closed. Please replace it with another.', 'wpvulnerability' ) . '</div>';
				}
				if ( (int) $vulnerability['unfixed'] ) {
					$html .= '<div class="text-red">' . __( 'This vulnerability appears to be unpatched. Stay tuned for upcoming plugin updates.', 'wpvulnerability' ) . '</div>';
				}
				$html .= '</div>';
			}
			if ( count( $what ) ) {
				$html .= '<div style="padding-bottom: 5px;">';
				foreach ( $what as $w ) {
					$html .= $w;
				}
				$html .= '</div>';
			}
			if ( ! is_null( $score ) || ! is_null( $severity ) ) {
				$html .= '<div style="padding-bottom: 5px;">';
				if ( ! is_null( $score ) ) {
					$html .= '<div>' . __( 'Global score: ', 'wpvulnerability' ) . $score . ' / 10</div>';
				}
				if ( ! is_null( $severity ) ) {
					$html .= '<div>' . __( 'Severity: ', 'wpvulnerability' ) . $severity . '</div>';
				}
				$html .= '</div>';
			}
			$html .= wp_kses( (string) $source, 'post' );

		}
	} elseif ( 'core' === $type ) {

		foreach ( $vulnerabilities as $vulnerability ) {

			$what = array();
			foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
				$what[] = '<div><b>' . wp_kses( (string) $vulnerability_cwe['name'], 'strip' ) . '</b></div><div><i>' . wp_kses_post( (string) $vulnerability_cwe['description'] ) . '</i></div>';
			}

			$sources = array();
			foreach ( $vulnerability['source'] as $vulnerability_source ) {
				$sources[] = '<a href="' . esc_url_raw( (string) $vulnerability_source['link'], 'strip' ) . '" target="_blank">[+]</a>&nbsp;' . wp_kses( (string) $vulnerability_source['name'], 'strip' );
			}
			$source = '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>';

			$score = null;
			if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
				$score = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
			}
			$severity = null;
			if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
				$severity = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
			}

			$html .= '<h3> WordPress ' . wp_kses( (string) $vulnerability['name'], 'strip' ) . '</h3>';
			if ( count( $what ) ) {
				$html .= '<div style="padding-bottom: 5px;">';
				foreach ( $what as $w ) {
					$html .= $w;
				}
				$html .= '</div>';
			}
			if ( ! is_null( $score ) || ! is_null( $severity ) ) {
				$html .= '<div style="padding-bottom: 5px;">';
				if ( ! is_null( $score ) ) {
					$html .= '<div>' . __( 'Global score: ', 'wpvulnerability' ) . $score . ' / 10</div>';
				}
				if ( ! is_null( $severity ) ) {
					$html .= '<div>' . __( 'Severity: ', 'wpvulnerability' ) . $severity . '</div>';
				}
				$html .= '</div>';
			}
			$html .= wp_kses( (string) $source, 'post' );

		}
	} elseif ( 'php' === $type ) {

		foreach ( $vulnerabilities as $vulnerability ) {

			$sources = array();
			foreach ( $vulnerability['source'] as $vulnerability_source ) {
				$sources[] = '<a href="' . esc_url_raw( (string) $vulnerability_source['link'], 'strip' ) . '" target="_blank">[+]</a>&nbsp;' . wp_kses( (string) $vulnerability_source['id'], 'strip' ) . '<br>' . wp_kses( (string) $vulnerability_source['description'], 'strip' );
			}
			$source = '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>';

			$html .= '<h4> ' . wp_kses( (string) $vulnerability['name'], 'strip' ) . '</h4>';
			$html .= '<div style="padding-bottom: 5px;"></div>';
			$html .= wp_kses( (string) $source, 'post' );

		}
	} elseif ( 'apache' === $type ) {

		foreach ( $vulnerabilities as $vulnerability ) {

			$sources = array();
			foreach ( $vulnerability['source'] as $vulnerability_source ) {
				$sources[] = '<a href="' . esc_url_raw( (string) $vulnerability_source['link'], 'strip' ) . '" target="_blank">[+]</a>&nbsp;' . wp_kses( (string) $vulnerability_source['id'], 'strip' ) . '<br>' . wp_kses( (string) $vulnerability_source['description'], 'strip' );
			}
			$source = '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>';

			$html .= '<h4> ' . wp_kses( (string) $vulnerability['name'], 'strip' ) . '</h4>';
			$html .= '<div style="padding-bottom: 5px;"></div>';
			$html .= wp_kses( (string) $source, 'post' );

		}
	} elseif ( 'nginx' === $type ) {

		foreach ( $vulnerabilities as $vulnerability ) {

			$sources = array();
			foreach ( $vulnerability['source'] as $vulnerability_source ) {
				$sources[] = '<a href="' . esc_url_raw( (string) $vulnerability_source['link'], 'strip' ) . '" target="_blank">[+]</a>&nbsp;' . wp_kses( (string) $vulnerability_source['id'], 'strip' ) . '<br>' . wp_kses( (string) $vulnerability_source['description'], 'strip' );
			}
			$source = '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>';

			$html .= '<h4> ' . wp_kses( (string) $vulnerability['name'], 'strip' ) . '</h4>';
			$html .= '<div style="padding-bottom: 5px;"></div>';
			$html .= wp_kses( (string) $source, 'post' );

		}
	}

	return $html;
}

/**
 * Convert plugin vulnerabilities into HTML format.
 *
 * @version 2.0.0
 *
 * @return string|false The HTML output if plugin vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_plugins() {

	$html  = '';
	$found = false;

	$plugins = wpvulnerability_plugin_get_vulnerabilities();

	foreach ( $plugins as $file_path => $plugin_data ) {

		// Check if the plugin is marked as vulnerable.
		if ( isset( $plugin_data['vulnerable'] ) && 1 === $plugin_data['vulnerable'] ) {

			$found = true;

			// Generate HTML markup for the plugin vulnerability.
			$html .= '<h3>' . esc_html__( 'Plugin', 'wpvulnerability' ) . ': ' . wp_kses( (string) $plugin_data['Name'], 'strip' ) . '</h3>';
			$html .= wpvulnerability_html( 'plugin', $plugin_data['vulnerabilities'] );

		}
	}

	// Return the HTML if vulnerabilities were found.
	if ( $found ) {
		return $html;
	}

	// Return false if no vulnerabilities were found.
	return false;
}

/**
 * Convert PHP vulnerabilities into HTML format.
 *
 * @version 3.1.0
 *
 * @return string|false The HTML output if PHP vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_php() {

	$html  = '';
	$found = false;

	// Get the PHP vulnerabilities data.
	$php = wpvulnerability_php_get_vulnerabilities();

	// Check if PHP is marked as vulnerable.
	if ( isset( $php['vulnerable'] ) && 1 === $php['vulnerable'] ) {

		$found = true;

		// Generate HTML markup for the PHP vulnerability.
		$html .= '<h3>' . esc_html__( 'PHP running', 'wpvulnerability' ) . ': ' . wp_kses( wpvulnerability_sanitize_version_php( phpversion() ), 'strip' ) . '</h3>';
		$html .= wpvulnerability_html( 'php', $php['vulnerabilities'] );

	}

	// Return the HTML if vulnerabilities were found.
	if ( $found ) {
		return $html;
	}

	// Return false if no vulnerabilities were found.
	return false;
}

/**
 * Convert Apache HTTPD vulnerabilities into HTML format.
 *
 * @version 3.2.0
 *
 * @return string|false The HTML output if Apache HTTPD vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_apache() {

	$html  = '';
	$found = false;

	// Check if the function to get Apache vulnerabilities is available, if not include the necessary file.
	if ( ! function_exists( 'wpvulnerability_apache_get_vulnerabilities' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-apache.php';
	}

	// Get Apache vulnerabilities data.
	$apache = wpvulnerability_apache_get_vulnerabilities();

	// Check if Apache is marked as vulnerable.
	if ( isset( $apache['vulnerable'] ) && 1 === $apache['vulnerable'] ) {

		$found = true;

		// Generate HTML markup for the Apache vulnerability.
		$webserver = wpvulnerability_detect_webserver();
		if ( isset( $webserver['id'] ) && 'apache' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {
			// Get the Apache HTTPD version.
			$apache_version = wp_kses( (string) $webserver['version'], 'strip' );

			if ( $apache_version ) {
				$html .= '<h3>' . esc_html__( 'Apache running', 'wpvulnerability' ) . ': ' . wp_kses( wpvulnerability_sanitize_version_apache( (string) $apache_version ), 'strip' ) . '</h3>';
				$html .= wpvulnerability_html( 'apache', $apache['vulnerabilities'] );
			}
		}
	}

	// Return the HTML if vulnerabilities were found.
	if ( $found ) {
		return $html;
	}

	// Return false if no vulnerabilities were found.
	return false;
}

/**
 * Convert nginx vulnerabilities into HTML format.
 *
 * @version 3.2.0
 *
 * @return string|false The HTML output if nginx vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_nginx() {

	$html  = '';
	$found = false;

	// Check if the function to get nginx vulnerabilities is available, if not include the necessary file.
	if ( ! function_exists( 'wpvulnerability_nginx_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-nginx.php';
	}

	// Get nginx vulnerabilities data.
	$nginx = wpvulnerability_nginx_get_vulnerabilities();

	// Check if nginx is marked as vulnerable.
	if ( isset( $nginx['vulnerable'] ) && 1 === $nginx['vulnerable'] ) {

		$found = true;

		// Generate HTML markup for the nginx vulnerability.
		$webserver = wpvulnerability_detect_webserver();
		if ( isset( $webserver['id'] ) && 'nginx' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {
			// Get the nginx version.
			$nginx_version = wp_kses( (string) $webserver['version'], 'strip' );

			if ( $nginx_version ) {
				$html .= '<h3>' . esc_html__( 'nginx running', 'wpvulnerability' ) . ': ' . wp_kses( wpvulnerability_sanitize_version_nginx( (string) $nginx_version ), 'strip' ) . '</h3>';
				$html .= wpvulnerability_html( 'nginx', $nginx['vulnerabilities'] );
			}
		}
	}

	// Return the HTML if vulnerabilities were found.
	if ( $found ) {
		return $html;
	}

	// Return false if no vulnerabilities were found.
	return false;
}

/**
 * Convert plugin vulnerabilities into list format.
 *
 * @version 2.2.0
 *
 * @return string|false The HTML output if plugin vulnerabilities were found, false otherwise.
 */
function wpvulnerability_list_plugins() {

	$html  = '<ul class="inside">';
	$found = false;

	// Get vulnerabilities data for plugins.
	$plugins = wpvulnerability_plugin_get_vulnerabilities();

	// Iterate through each plugin's data.
	foreach ( $plugins as $file_path => $plugin_data ) {

		// Check if the plugin is marked as vulnerable.
		if ( isset( $plugin_data['vulnerable'] ) && 1 === $plugin_data['vulnerable'] ) {

			$found = true;

			// Generate HTML markup for the plugin vulnerability.
			$html .= '<li>' . wp_kses( (string) $plugin_data['Name'], 'strip' ) . '</li>';

		}
	}

	$html .= '</ul>';

	// Return the HTML if vulnerabilities were found.
	if ( $found ) {
		return $html;
	}

	// Return false if no vulnerabilities were found.
	return false;
}

/**
 * Convert theme vulnerabilities into HTML format.
 *
 * @version 2.0.0
 *
 * @return string|false The HTML output if theme vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_themes() {

	$html  = '';
	$found = false;

	// Get vulnerabilities data for themes.
	$themes = wpvulnerability_theme_get_vulnerabilities();

	// Iterate through each theme's data.
	foreach ( $themes as $theme_data ) {

		// Check if the theme is marked as vulnerable.
		if ( isset( $theme_data['wpvulnerability']['vulnerable'] ) && 1 === $theme_data['wpvulnerability']['vulnerable'] ) {

			$found = true;

			// Generate HTML markup for the theme vulnerability.
			$html .= '<h3>' . esc_html__( 'Theme', 'wpvulnerability' ) . ': ' . wp_kses( (string) $theme_data['wpvulnerability']['name'], 'strip' ) . '</h3>';
			$html .= wpvulnerability_html( 'theme', $theme_data['wpvulnerability']['vulnerabilities'] );

		}
	}

	// Return the HTML if vulnerabilities were found.
	if ( $found ) {
		return $html;
	}

	// Return false if no vulnerabilities were found.
	return false;
}

/**
 * Convert theme vulnerabilities into list format.
 *
 * @version 2.2.0
 *
 * @return string|false The HTML output if theme vulnerabilities were found, false otherwise.
 */
function wpvulnerability_list_themes() {

	$html  = '<ul class="inside">';
	$found = false;

	// Get vulnerabilities data for themes.
	$themes = wpvulnerability_theme_get_vulnerabilities();

	// Iterate through each theme's data.
	foreach ( $themes as $theme_data ) {

		// Check if the theme is marked as vulnerable.
		if ( isset( $theme_data['wpvulnerability']['vulnerable'] ) && 1 === $theme_data['wpvulnerability']['vulnerable'] ) {

			$found = true;

			// Generate HTML markup for the theme vulnerability.
			$html .= '<li>' . wp_kses( (string) $theme_data['wpvulnerability']['name'], 'strip' ) . '</li>';

		}
	}

	$html .= '</ul>';

	// Return the HTML if vulnerabilities were found.
	if ( $found ) {
		return $html;
	}

	// Return false if no vulnerabilities were found.
	return false;
}

/**
 * Convert core vulnerabilities into HTML format.
 *
 * @version 2.0.0
 *
 * @return string|false The HTML output if core vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_core() {

	$html  = '';
	$found = false;

	// Get vulnerabilities data for WordPress core.
	$core = wpvulnerability_core_get_vulnerabilities();

	// Check if there are any vulnerabilities.
	if ( is_array( $core ) && count( $core ) ) {

		$found = true;

		// Generate HTML markup for the core vulnerabilities.
		$html .= wpvulnerability_html( 'core', $core );

	}

	// Return the HTML if vulnerabilities were found.
	if ( $found ) {
		return $html;
	}

	// Return false if no vulnerabilities were found.
	return false;
}
