<?php
/**
 * PHP functions
 *
 * @package WPVulnerability
 *
 * @version 3.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Retrieves vulnerabilities for the current PHP version and updates its data.
 *
 * This function sanitizes and retrieves the current PHP version, checks for vulnerabilities
 * using an external API, and updates the PHP data array with the vulnerabilities found.
 *
 * @since 3.0.0
 *
 * @return array The updated PHP data array containing vulnerability information.
 */
function wpvulnerability_get_fresh_php_vulnerabilities() {

	// Get the PHP version and sanitize it.
	$php_version = wp_kses( wpvulnerability_sanitize_version_php( phpversion() ), 'strip' );

	// Initialize PHP data with default values.
	$php_data = array(
		'vulnerabilities' => null,
		'vulnerable'      => 0,
	);

	// Retrieve vulnerabilities for the PHP version.
	if ( $php_version ) {
		// Call the API to get vulnerabilities for the sanitized PHP version.
		$php_api_response = wpvulnerability_get_php( $php_version, 0 );

		// If vulnerabilities are found, update the PHP data accordingly.
		if ( ! empty( $php_api_response ) ) {
			$php_data['vulnerabilities'] = $php_api_response;
			$php_data['vulnerable']      = 1;
		}
	}

	// Return the updated PHP data array.
	return $php_data;
}

/**
 * Get Installed PHP
 *
 * Retrieves the list of installed PHP versions, checks for vulnerabilities,
 * caches the data, and sends an email notification if vulnerabilities are detected.
 *
 * @since 3.0.0
 *
 * @return string JSON-encoded array of PHP data with vulnerabilities and vulnerable status.
 */
function wpvulnerability_php_get_installed() {

	$wpvulnerability_php_vulnerable = 0;

	// Retrieve fresh vulnerabilities for the installed PHP version.
	$php = wpvulnerability_get_fresh_php_vulnerabilities();

	// Check if the PHP version is vulnerable and count the vulnerabilities.
	if ( isset( $php['vulnerable'] ) && (int) $php['vulnerable'] ) {
		$wpvulnerability_php_vulnerable = count( $php['vulnerabilities'] );
	}

	// Cache the vulnerability data and the timestamp for cache expiration.
	if ( is_multisite() ) {
		update_site_option( 'wpvulnerability-php', wp_json_encode( $php ) );
		update_site_option( 'wpvulnerability-php-vulnerable', wp_json_encode( number_format( $wpvulnerability_php_vulnerable, 0, '.', '' ) ) );
		update_site_option( 'wpvulnerability-php-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );
	} else {
		update_option( 'wpvulnerability-php', wp_json_encode( $php ) );
		update_option( 'wpvulnerability-php-vulnerable', wp_json_encode( number_format( $wpvulnerability_php_vulnerable, 0, '.', '' ) ) );
		update_option( 'wpvulnerability-php-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );
	}

	// Return the JSON-encoded array of PHP data.
	return wp_json_encode( $php );
}

/**
 * Get the cached PHP vulnerabilities or update the cache if it's stale or missing.
 *
 * @since 3.0.0
 *
 * @return array Array of PHP with their vulnerabilities.
 */
function wpvulnerability_php_get_vulnerabilities() {

	if ( is_multisite() ) {

		// Get the cached PHP data and decode it.
		$php_data_cache = json_decode( get_site_option( 'wpvulnerability-php-cache' ) );

		// Get the installed PHP data and decode it.
		$php_data = json_decode( get_site_option( 'wpvulnerability-php' ), true );

	} else {

		// Get the cached PHP data and decode it.
		$php_data_cache = json_decode( get_option( 'wpvulnerability-php-cache' ) );

		// Get the installed PHP data and decode it.
		$php_data = json_decode( get_option( 'wpvulnerability-php' ), true );

	}

	// If the cache is stale or the PHP data is empty, update the cache.
	if ( $php_data_cache < time() || empty( $php_data ) ) {

		// Get the installed PHP data and update the cache.
		$php_data = json_decode( wpvulnerability_php_get_installed(), true );

	}

	return $php_data;
}

/**
 * Update the PHP cache and remove any old cache data.
 *
 * @since 3.0.0
 *
 * @return void
 */
function wpvulnerability_php_get_vulnerabilities_clean() {

	// Update the installed PHP cache.
	wpvulnerability_php_get_installed();
}
