<?php
/**
 * Notifications functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Adds a custom schedule for a weekly cron job.
 *
 * This function adds a new schedule interval of one week (604800 seconds)
 * to the system's available cron schedules. It allows tasks to be scheduled
 * to run every week using the 'weekly' interval.
 *
 * @since 2.0.0
 *
 * @param array $schedules The existing system schedules.
 *
 * @return array The updated list of schedules with the added weekly interval.
 */
function wpvulnerability_add_every_week( $schedules ) {

	// Add a weekly schedule interval of 604800 seconds (1 week).
	$schedules['weekly'] = array(
		'interval' => 604800,
		'display'  => __( 'Every week', 'wpvulnerability' ),
	);

	// Return the modified list of schedules.
	return $schedules;
}

// Hook the function to the 'cron_schedules' filter to add the custom schedule.
add_filter( 'cron_schedules', 'wpvulnerability_add_every_week' );

/**
 * Adds a custom schedule for daily events.
 *
 * This function adds a new schedule interval of one day (86400 seconds)
 * to the system's available cron schedules. It allows tasks to be scheduled
 * to run every day using the 'daily' interval.
 *
 * @since 2.0.0
 *
 * @param array $schedules List of available schedules.
 *
 * @return array Modified list of available schedules with the added daily interval.
 */
function wpvulnerability_add_every_day( $schedules ) {

	// Define a new schedule with a 24 hour interval.
	$schedules['daily'] = array(
		'interval' => 86400,
		'display'  => __( 'Every day', 'wpvulnerability' ),
	);

	// Return the modified list of schedules.
	return $schedules;
}

// Hook the function to the 'cron_schedules' filter to add the custom schedule.
add_filter( 'cron_schedules', 'wpvulnerability_add_every_day' );

/**
 * Prepares the HTML email message
 *
 * This function generates an HTML email message with the given title and content.
 * It includes basic styling and structure to ensure compatibility with most email clients.
 *
 * @since 2.0.0
 *
 * @param string $title   The title of the email message.
 * @param string $content The content of the email message.
 *
 * @return string The prepared HTML email message.
 */
function wpvulnerability_email_prepare( $title, $content ) {

	$message  = '';
	$message .= '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">' . "\n";
	$message .= '<html xmlns="http://www.w3.org/1999/xhtml" style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '<head>' . "\n";
	$message .= '	<meta name="viewport" content="width=device-width">' . "\n";
	$message .= '	<meta http-equiv="Content-Type" content="text/html; charset=UTF-8">' . "\n";
	$message .= '	<title>WPVulnerability</title>' . "\n";
	$message .= '	<style type="text/css">' . "\n";
	$message .= '	img { max-width: 100%; }' . "\n";
	$message .= '  body { -webkit-font-smoothing: antialiased; -webkit-text-size-adjust: none; width: 100% !important; height: 100%; line-height: 1.2em; }' . "\n";
	$message .= '  body { background-color: #f6f6f6; }' . "\n";
	$message .= '  @media only screen and (max-width: 640px) {' . "\n";
	$message .= '		body { padding: 0 !important; }' . "\n";
	$message .= '		h1 { margin: 20px 0 5px 0 !important; }' . "\n";
	$message .= '		h2 { margin: 20px 0 5px 0 !important; }' . "\n";
	$message .= '		h3 { margin: 20px 0 5px 0 !important; }' . "\n";
	$message .= '		h4 { margin: 20px 0 5px 0 !important; }' . "\n";
	$message .= '		.container { padding: 0 !important; width: 100% !important; }' . "\n";
	$message .= '		.content { padding: 0 !important; }' . "\n";
	$message .= '		.content-wrap { padding: 10px !important; }' . "\n";
	$message .= '		.invoice { width: 100% !important; }' . "\n";
	$message .= '  }' . "\n";
	$message .= '  </style>' . "\n";
	$message .= '</head>' . "\n";
	$message .= '<body itemscope itemtype="http://schema.org/EmailMessage" style="box-sizing: border-box; -webkit-font-smoothing: antialiased; -webkit-text-size-adjust: none; width: 100% !important; height: 100%; line-height: 1.6em; background-color: #f6f6f6; margin: 0;" bgcolor="#f6f6f6">' . "\n";
	$message .= '	<table class="body-wrap" style="box-sizing: border-box; width: 100%; background-color: #f6f6f6; margin: 0;" bgcolor="#f6f6f6">' . "\n";
	$message .= '		<tr style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '			<td style="box-sizing: border-box; vertical-align: top; margin: 0;" valign="top"></td>' . "\n";
	$message .= '			<td class="container" width="600" style="box-sizing: border-box; vertical-align: top; display: block !important; max-width: 600px !important; clear: both !important; margin: 0 auto;" valign="top">' . "\n";
	$message .= '				<div class="content" style="box-sizing: border-box; max-width: 600px; display: block; margin: 0 auto; padding: 20px;">' . "\n";
	$message .= '					<table class="main" width="100%" cellpadding="0" cellspacing="0" style="box-sizing: border-box; border-radius: 3px; background-color: #fff; margin: 0; border: 1px solid #e9e9e9;" bgcolor="#fff">' . "\n";
	$message .= '						<tr style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '							<td class="content-wrap aligncenter" style="box-sizing: border-box; vertical-align: top; text-align: center; margin: 0; padding: 20px;" align="center" valign="top">' . "\n";
	$message .= '								<table width="100%" cellpadding="0" cellspacing="0" style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '									<tr style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '										<td class="content-block" style="box-sizing: border-box; vertical-align: top; margin: 0; padding: 0 0 20px; text-align: center;" valign="top">' . "\n";
	$message .= '											<img class="aligncenter" src="' . WPVULNERABILITY_PLUGIN_URL . 'assets/logo64.png" width="64" height="64" alt="WPVulnerability">' . "\n";
	$message .= '											<h1 class="aligncenter" style="box-sizing: border-box; color: #000; line-height: 1.2em; text-align: center; margin: 40px 0 0;" align="center">' . $title . '</h1>' . "\n";

	// Add the site URL based on the multisite configuration.
	if ( is_multisite() ) {
		$message .= '											<p class="aligncenter" style="box-sizing: border-box; color: #000; line-height: 1.2em; text-align: center; margin: 5px 0 0;" align="center"><a href="' . network_site_url() . '" target="_blank" rel="noopener noreferrer">' . network_site_url() . '</a></p>' . "\n";
	} else {
		$message .= '											<p class="aligncenter" style="box-sizing: border-box; color: #000; line-height: 1.2em; text-align: center; margin: 5px 0 0;" align="center"><a href="' . site_url() . '" target="_blank" rel="noopener noreferrer">' . site_url() . '</a></p>' . "\n";
	}

	$message .= '										</td>' . "\n";
	$message .= '									</tr>' . "\n";
	$message .= '									<tr style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '										<td class="content-block alignleft" style="box-sizing: border-box; vertical-align: top; text-align: left; margin: 0; padding: 0 0 20px;" valign="top">' . "\n";
	$message .= $content; // Add the main content of the email.
	$message .= '									</td>' . "\n";
	$message .= '									</tr>' . "\n";
	$message .= '								</table>' . "\n";
	$message .= '								<div class="footer" style="box-sizing: border-box; width: 100%; clear: both; color: #999; margin: 0; padding: 20px;">' . "\n";
	$message .= '									<table width="100%" style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '										<tr style="box-sizing: border-box; margin: 0;">' . "\n";
	$message .= '											<td class="aligncenter content-block" style="box-sizing: border-box; vertical-align: top; color: #999; text-align: center; margin: 0; padding: 0 0 20px;" align="center" valign="top">' . "\n";
	$message .= sprintf(
		// translators: %1$s the website of Database, %2$s database site name.
		__( 'Learn more about the WordPress Vulnerability Database API at <a href="%1$s">%2$s</a>', 'wpvulnerability' ),
		'https://www.wpvulnerability.com/',
		'WPVulnerability'
	);
	$message .= '											</td>' . "\n";
	$message .= '										</tr>' . "\n";

	// Add the site URL in the footer based on the multisite configuration.
	if ( is_multisite() ) {
		$message .= '											<td class="aligncenter content-block" style="box-sizing: border-box; vertical-align: top; color: #999; text-align: center; margin: 0; padding: 0 0 20px;" align="center" valign="top"><a href="' . network_site_url() . '" target="_blank" rel="noopener noreferrer">' . network_site_url() . '</a></td>' . "\n";
	} else {
		$message .= '											<td class="aligncenter content-block" style="box-sizing: border-box; vertical-align: top; color: #999; text-align: center; margin: 0; padding: 0 0 20px;" align="center" valign="top"><a href="' . site_url() . '" target="_blank" rel="noopener noreferrer">' . site_url() . '</a></td>' . "\n";
	}

	$message .= '										</tr>' . "\n";
	$message .= '									</table>' . "\n";
	$message .= '								</div>' . "\n";
	$message .= '							</td>' . "\n";
	$message .= '							<td style="box-sizing: border-box; vertical-align: top; margin: 0;" valign="top"></td>' . "\n";
	$message .= '						</tr>' . "\n";
	$message .= '					</table>' . "\n";
	$message .= '				</div>' . "\n";
	$message .= '			</td>' . "\n";
	$message .= '		</tr>' . "\n";
	$message .= '	</table>' . "\n";
	$message .= '</body>' . "\n";
	$message .= '</html>';

	// Return the prepared HTML email message.
	return $message;
}

/**
 * Executes the vulnerability notification process for a WordPress site.
 *
 * This function checks for vulnerabilities in the WordPress core, plugins, themes, and PHP environment.
 * It generates an HTML email report detailing any vulnerabilities found. If the function is called with
 * the $forced parameter set to true, it will send an email even if no vulnerabilities are found, which is useful for testing purposes.
 *
 * @since 2.0.0
 *
 * @param bool $forced Optional. If set to true, forces the sending of a notification email regardless of whether vulnerabilities are found. Default false.
 * @return string Returns the content if the email was successfully sent, false otherwise.
 */
function wpvulnerability_execute_notification( $forced = false ) {

	$email_content = '';
	if ( is_multisite() ) {
		$wpvulnerability_settings = get_site_option( 'wpvulnerability-config' );
	} else {
		$wpvulnerability_settings = get_option( 'wpvulnerability-config' );
	}

	// Required by the "forced email" button.
	if ( ! $forced ) {

		// If email or period settings are empty, return.
		if ( ! isset( $wpvulnerability_settings['emails'] ) || ! isset( $wpvulnerability_settings['period'] ) ) {
			return;
		}
		if ( ! $wpvulnerability_settings['emails'] || ! $wpvulnerability_settings['period'] ) {
			return;
		}
	}

	// Get HTML for core, plugins, and themes vulnerabilities.
	$html_core = null;
	if ( wpvulnerability_analyze_filter( 'core' ) && is_multisite() && json_decode( get_site_option( 'wpvulnerability-core-vulnerable' ) ) ) {
		$html_core = wpvulnerability_html_core();
	} elseif ( wpvulnerability_analyze_filter( 'core' ) && json_decode( get_option( 'wpvulnerability-core-vulnerable' ) ) ) {
		$html_core = wpvulnerability_html_core();
	}

	$html_plugins = null;
	if ( wpvulnerability_analyze_filter( 'plugins' ) && is_multisite() && json_decode( get_site_option( 'wpvulnerability-plugins-vulnerable' ) ) ) {
		$html_plugins = wpvulnerability_html_plugins();
	} elseif ( wpvulnerability_analyze_filter( 'plugins' ) && json_decode( get_option( 'wpvulnerability-plugins-vulnerable' ) ) ) {
		$html_plugins = wpvulnerability_html_plugins();
	}

	$html_themes = null;
	if ( wpvulnerability_analyze_filter( 'themes' ) && is_multisite() && json_decode( get_site_option( 'wpvulnerability-themes-vulnerable' ) ) ) {
		$html_themes = wpvulnerability_html_themes();
	} elseif ( wpvulnerability_analyze_filter( 'themes' ) && json_decode( get_option( 'wpvulnerability-themes-vulnerable' ) ) ) {
		$html_themes = wpvulnerability_html_themes();
	}

	$html_php = null;
	if ( wpvulnerability_analyze_filter( 'php' ) && is_multisite() && json_decode( get_site_option( 'wpvulnerability-php-vulnerable' ) ) ) {
		$html_php = wpvulnerability_html_php();
	} elseif ( wpvulnerability_analyze_filter( 'php' ) && json_decode( get_option( 'wpvulnerability-php-vulnerable' ) ) ) {
		$html_php = wpvulnerability_html_php();
	}

	$html_apache = null;
	if ( wpvulnerability_analyze_filter( 'apache' ) && is_multisite() && json_decode( get_site_option( 'wpvulnerability-apache-vulnerable' ) ) ) {
		$html_apache = wpvulnerability_html_apache();
	} elseif ( wpvulnerability_analyze_filter( 'apache' ) && json_decode( get_option( 'wpvulnerability-apache-vulnerable' ) ) ) {
		$html_apache = wpvulnerability_html_apache();
	}

	$html_nginx = null;
	if ( wpvulnerability_analyze_filter( 'nginx' ) && is_multisite() && json_decode( get_site_option( 'wpvulnerability-nginx-vulnerable' ) ) ) {
		$html_nginx = wpvulnerability_html_nginx();
	} elseif ( wpvulnerability_analyze_filter( 'nginx' ) && json_decode( get_option( 'wpvulnerability-nginx-vulnerable' ) ) ) {
		$html_nginx = wpvulnerability_html_nginx();
	}

	// Required by the "forced email" button.
	if ( ! $forced ) {

		// If no vulnerabilities were found, return.
		if ( empty( $html_core ) && empty( $html_plugins ) && empty( $html_themes ) && empty( $html_php ) && empty( $html_apache ) && empty( $html_nginx ) ) {
			return;
		}
	} elseif ( $forced ) {

		// If no vulnerabilities were found, return.
		if ( empty( $html_core ) && empty( $html_plugins ) && empty( $html_themes ) && empty( $html_php ) && empty( $html_apache ) && empty( $html_nginx ) ) {
			$email_content .= '<h2>' . esc_html__( 'There are no vulnerabilities', 'wpvulnerability' ) . '</h2>';
			$email_content .= '<p>' . esc_html__( 'This is probably a test. The site probably does not have vulnerabilities.', 'wpvulnerability' ) . '</p>';
		}
	}

	// Add core vulnerabilities HTML to email content.
	if ( ! empty( $html_core ) ) {
		$email_content .= '<h2>' . esc_html__( 'Core vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_core;
	}

	// Add plugins vulnerabilities HTML to email content.
	if ( $html_plugins ) {
		$email_content .= '<h2>' . esc_html__( 'Plugins vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_plugins;
	}

	// Add themes vulnerabilities HTML to email content.
	if ( $html_themes ) {
		$email_content .= '<h2>' . esc_html__( 'Themes vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_themes;
	}

	// Add PHP vulnerabilities HTML to email content.
	if ( $html_php ) {
		$email_content .= '<h2>' . esc_html__( 'PHP vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_php;
	}

	// Add Apache vulnerabilities HTML to email content.
	if ( $html_apache ) {
		$email_content .= '<h2>' . esc_html__( 'Apache HTTPD vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_apache;
	}

	// Add nginx vulnerabilities HTML to email content.
	if ( $html_nginx ) {
		$email_content .= '<h2>' . esc_html__( 'nginx vulnerabilities', 'wpvulnerability' ) . '</h2>';
		$email_content .= $html_nginx;
	}

	// Get the site name.
	if ( is_multisite() ) {
		$admin_site = get_site_option( 'network_name_option' );
	} else {
		$admin_site = get_bloginfo( 'name' );
	}

	// Get the admin email.
	if ( is_multisite() ) {
		$admin_email = get_site_option( 'admin_email' );
	} else {
		$admin_email = get_bloginfo( 'admin_email' );
	}
	$from_email = $admin_email;

	// Check if WPVULNERABILITY_MAIL is defined and valid.
	if ( defined( 'WPVULNERABILITY_MAIL' ) ) {
		$wpvulnerability_sender_email = sanitize_email( trim( WPVULNERABILITY_MAIL ) );
		if ( is_email( $wpvulnerability_sender_email ) ) {
			$from_email = $wpvulnerability_sender_email;
		}
		unset( $wpvulnerability_sender_email );
	}

	// Prepare email subject and content.
	$email_subject = sprintf(
		// translators: Site name.
		__( 'Vulnerability found: %s', 'wpvulnerability' ),
		$admin_site
	);

	$email_prepared = wpvulnerability_email_prepare( esc_html__( 'Vulnerability found', 'wpvulnerability' ), $email_content );

	// Prepare email headers.
	$email_headers   = array();
	$email_headers[] = 'From: WPVulnerability <' . $from_email . '>';
	$email_headers[] = 'Content-Type: text/html; charset=UTF-8';

	// Send email.
	$wpmail = wp_mail( $wpvulnerability_settings['emails'], $email_subject, $email_prepared, $email_headers );

	return $wpmail;
}
