<?php
/**
 * Nginx functions
 *
 * @package WPVulnerability
 *
 * @version 3.2.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Retrieves vulnerabilities for an nginx version and updates its data.
 *
 * @since 3.2.0
 *
 * @return array The updated nginx data array.
 */
function get_fresh_nginx_vulnerabilities() {

	$nginx_version = null;
	$webserver     = wpvulnerability_detect_webserver();
	if ( isset( $webserver['id'] ) && 'nginx' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {
		// Get the nginx version.
		$nginx_version = wp_kses( (string) $webserver['version'], 'strip' );
	}

	// Initialize vulnerability related fields.
	$nginx_data['vulnerabilities'] = null;
	$nginx_data['vulnerable']      = 0;

	// Retrieve vulnerabilities for the nginx using its version.
	if ( $nginx_version ) {

		$nginx_api_response = wpvulnerability_get_nginx( $nginx_version );

		// If vulnerabilities are found, update the nginx data accordingly.
		if ( ! empty( $nginx_api_response ) ) {

			$nginx_data['vulnerabilities'] = $nginx_api_response;
			$nginx_data['vulnerable']      = 1;

		}
	}

	return $nginx_data;
}

/**
 * Get Installed nginx
 * Retrieves the list of installed nginx, checks for vulnerabilities, caches the data, and sends an email notification if vulnerabilities are detected.
 *
 * @since 3.2.0
 *
 * @return string JSON-encoded array of nginx data with vulnerabilities and vulnerable status
 */
function wpvulnerability_nginx_get_installed() {

	$wpvulnerability_nginx_vulnerable = 0;

	$nginx = get_fresh_nginx_vulnerabilities();

	if ( isset( $nginx['vulnerable'] ) && (int) $nginx['vulnerable'] ) {

		$wpvulnerability_nginx_vulnerable = count( $nginx['vulnerabilities'] );

	}

	if ( is_multisite() ) {

		update_site_option( 'wpvulnerability-nginx', wp_json_encode( $nginx ) );
		update_site_option( 'wpvulnerability-nginx-vulnerable', wp_json_encode( number_format( $wpvulnerability_nginx_vulnerable, 0, '.', '' ) ) );

	} elseif ( ! is_multisite() ) {

		update_option( 'wpvulnerability-nginx', wp_json_encode( $nginx ) );
		update_option( 'wpvulnerability-nginx-vulnerable', wp_json_encode( number_format( $wpvulnerability_nginx_vulnerable, 0, '.', '' ) ) );

	}

	return wp_json_encode( $nginx );
}

/**
 * Get the cached nginx vulnerabilities or update the cache if it's stale or missing.
 *
 * @since 3.2.0
 *
 * @return array Array of nginx with their vulnerabilities.
 */
function wpvulnerability_nginx_get_vulnerabilities() {

	if ( is_multisite() ) {

		// Get the cached plugin data and decode it.
		$nginx_data_cache = json_decode( get_site_option( 'wpvulnerability-nginx-cache' ) );

		// Get the installed plugin data and decode it.
		$nginx_data = json_decode( get_site_option( 'wpvulnerability-nginx' ), true );

	} elseif ( ! is_multisite() ) {

		// Get the cached plugin data and decode it.
		$nginx_data_cache = json_decode( get_option( 'wpvulnerability-nginx-cache' ) );

		// Get the installed plugin data and decode it.
		$nginx_data = json_decode( get_option( 'wpvulnerability-nginx' ), true );

	}

	// If the cache is stale or the plugin data is empty, update the cache.
	if ( $nginx_data_cache < time() || empty( $nginx_data ) ) {

		// Get the installed plugin data and update the cache.
		$nginx_data = json_decode( wpvulnerability_nginx_get_installed(), true );

		if ( is_multisite() ) {

			update_site_option( 'wpvulnerability-nginx-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

		} elseif ( ! is_multisite() ) {

			update_option( 'wpvulnerability-nginx-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

		}
	}

	return $nginx_data;
}

/**
 * Update the nginx cache and remove any old cache data
 *
 * @since 3.2.0
 *
 * @return void
 */
function wpvulnerability_nginx_get_vulnerabilities_clean() {

	// Update the installed plugins cache.
	wpvulnerability_nginx_get_installed();

	if ( is_multisite() ) {

		update_site_option( 'wpvulnerability-nginx-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

	} elseif ( ! is_multisite() ) {

		update_option( 'wpvulnerability-nginx-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

	}
}
