<?php
/**
 * Apache HTTPD functions
 *
 * @package WPVulnerability
 *
 * @version 3.2.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Retrieves vulnerabilities for an Apache HTTPD version and updates its data.
 *
 * @since 3.2.0
 *
 * @return array The updated Apache HTTPD data array.
 */
function get_fresh_apache_vulnerabilities() {

	$apache_version = null;
	$webserver      = wpvulnerability_detect_webserver();
	if ( isset( $webserver['id'] ) && 'apache' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {
		// Get the Apache HTTPD version.
		$apache_version = wp_kses( (string) $webserver['version'], 'strip' );
	}

	// Initialize vulnerability related fields.
	$apache_data['vulnerabilities'] = null;
	$apache_data['vulnerable']      = 0;

	// Retrieve vulnerabilities for the Apache HTTPD using its version.
	if ( $apache_version ) {

		$apache_api_response = wpvulnerability_get_apache( $apache_version );

		// If vulnerabilities are found, update the Apache HTTPD data accordingly.
		if ( ! empty( $apache_api_response ) ) {

			$apache_data['vulnerabilities'] = $apache_api_response;
			$apache_data['vulnerable']      = 1;

		}
	}

	return $apache_data;
}

/**
 * Get Installed Apache HTTPD
 * Retrieves the list of installed Apache HTTPD, checks for vulnerabilities, caches the data, and sends an email notification if vulnerabilities are detected.
 *
 * @since 3.2.0
 *
 * @return string JSON-encoded array of Apache HTTPD data with vulnerabilities and vulnerable status
 */
function wpvulnerability_apache_get_installed() {

	$wpvulnerability_apache_vulnerable = 0;

	$apache = get_fresh_apache_vulnerabilities();

	if ( isset( $apache['vulnerable'] ) && (int) $apache['vulnerable'] ) {

		$wpvulnerability_apache_vulnerable = count( $apache['vulnerabilities'] );

	}

	if ( is_multisite() ) {

		update_site_option( 'wpvulnerability-apache', wp_json_encode( $apache ) );
		update_site_option( 'wpvulnerability-apache-vulnerable', wp_json_encode( number_format( $wpvulnerability_apache_vulnerable, 0, '.', '' ) ) );

	} elseif ( ! is_multisite() ) {

		update_option( 'wpvulnerability-apache', wp_json_encode( $apache ) );
		update_option( 'wpvulnerability-apache-vulnerable', wp_json_encode( number_format( $wpvulnerability_apache_vulnerable, 0, '.', '' ) ) );

	}

	return wp_json_encode( $apache );
}

/**
 * Get the cached Apache HTTPD vulnerabilities or update the cache if it's stale or missing.
 *
 * @since 3.2.0
 *
 * @return array Array of Apache HTTPD with their vulnerabilities.
 */
function wpvulnerability_apache_get_vulnerabilities() {

	if ( is_multisite() ) {

		// Get the cached plugin data and decode it.
		$apache_data_cache = json_decode( get_site_option( 'wpvulnerability-apache-cache' ) );

		// Get the installed plugin data and decode it.
		$apache_data = json_decode( get_site_option( 'wpvulnerability-apache' ), true );

	} elseif ( ! is_multisite() ) {

		// Get the cached plugin data and decode it.
		$apache_data_cache = json_decode( get_option( 'wpvulnerability-apache-cache' ) );

		// Get the installed plugin data and decode it.
		$apache_data = json_decode( get_option( 'wpvulnerability-apache' ), true );

	}

	// If the cache is stale or the plugin data is empty, update the cache.
	if ( $apache_data_cache < time() || empty( $apache_data ) ) {

		// Get the installed plugin data and update the cache.
		$apache_data = json_decode( wpvulnerability_apache_get_installed(), true );

		if ( is_multisite() ) {

			update_site_option( 'wpvulnerability-apache-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

		} elseif ( ! is_multisite() ) {

			update_option( 'wpvulnerability-apache-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

		}
	}

	return $apache_data;
}

/**
 * Update the Apache HTTPD cache and remove any old cache data
 *
 * @since 3.2.0
 *
 * @return void
 */
function wpvulnerability_apache_get_vulnerabilities_clean() {

	// Update the installed plugins cache.
	wpvulnerability_apache_get_installed();

	if ( is_multisite() ) {

		update_site_option( 'wpvulnerability-apache-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

	} elseif ( ! is_multisite() ) {

		update_option( 'wpvulnerability-apache-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

	}
}
