<?php
/**
 * PHP functions
 *
 * @package WPVulnerability
 *
 * @version 3.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Retrieves vulnerabilities for a PHP version and updates its data.
 *
 * @since 3.0.0
 *
 * @return array The updated PHP data array.
 */
function get_fresh_php_vulnerabilities() {

	// Get the PHP version.
	$php_version = wp_kses( wpvulnerability_sanitize_version_php( phpversion() ), 'strip' );

	// Initialize vulnerability related fields.
	$php_data['vulnerabilities'] = null;
	$php_data['vulnerable']      = 0;

	// Retrieve vulnerabilities for the PHP using its version.
	if ( $php_version ) {

		$php_api_response = wpvulnerability_get_php( $php_version );

		// If vulnerabilities are found, update the PHP data accordingly.
		if ( ! empty( $php_api_response ) ) {

			$php_data['vulnerabilities'] = $php_api_response;
			$php_data['vulnerable']      = 1;

		}
	}

	return $php_data;
}

/**
 * Get Installed PHP
 * Retrieves the list of installed PHP, checks for vulnerabilities, caches the data, and sends an email notification if vulnerabilities are detected.
 *
 * @since 3.0.0
 *
 * @return string JSON-encoded array of PHP data with vulnerabilities and vulnerable status
 */
function wpvulnerability_php_get_installed() {

	$wpvulnerability_php_vulnerable = 0;

	$php = get_fresh_php_vulnerabilities();

	if ( isset( $php['vulnerable'] ) && (int) $php['vulnerable'] ) {

		$wpvulnerability_php_vulnerable = count( $php['vulnerabilities'] );

	}

	if ( is_multisite() ) {

		update_site_option( 'wpvulnerability-php', wp_json_encode( $php ) );
		update_site_option( 'wpvulnerability-php-vulnerable', wp_json_encode( number_format( $wpvulnerability_php_vulnerable, 0, '.', '' ) ) );

	} elseif ( ! is_multisite() ) {

		update_option( 'wpvulnerability-php', wp_json_encode( $php ) );
		update_option( 'wpvulnerability-php-vulnerable', wp_json_encode( number_format( $wpvulnerability_php_vulnerable, 0, '.', '' ) ) );

	}

	return wp_json_encode( $php );
}

/**
 * Get the cached PHP vulnerabilities or update the cache if it's stale or missing.
 *
 * @since 3.0.0
 *
 * @return array Array of PHP with their vulnerabilities.
 */
function wpvulnerability_php_get_vulnerabilities() {

	if ( is_multisite() ) {

		// Get the cached plugin data and decode it.
		$php_data_cache = json_decode( get_site_option( 'wpvulnerability-php-cache' ) );

		// Get the installed plugin data and decode it.
		$php_data = json_decode( get_site_option( 'wpvulnerability-php' ), true );

	} elseif ( ! is_multisite() ) {

		// Get the cached plugin data and decode it.
		$php_data_cache = json_decode( get_option( 'wpvulnerability-php-cache' ) );

		// Get the installed plugin data and decode it.
		$php_data = json_decode( get_option( 'wpvulnerability-php' ), true );

	}

	// If the cache is stale or the plugin data is empty, update the cache.
	if ( $php_data_cache < time() || empty( $php_data ) ) {

		// Get the installed plugin data and update the cache.
		$php_data = json_decode( wpvulnerability_php_get_installed(), true );

		if ( is_multisite() ) {

			update_site_option( 'wpvulnerability-php-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

		} elseif ( ! is_multisite() ) {

			update_option( 'wpvulnerability-php-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

		}
	}

	return $php_data;
}

/**
 * Update the PHP cache and remove any old cache data
 *
 * @since 3.0.0
 *
 * @return void
 */
function wpvulnerability_php_get_vulnerabilities_clean() {

	// Update the installed plugins cache.
	wpvulnerability_php_get_installed();

	if ( is_multisite() ) {

		update_site_option( 'wpvulnerability-php-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

	} elseif ( ! is_multisite() ) {

		update_option( 'wpvulnerability-php-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

	}
}
