<?php
/**
 * General functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Checks and validates user capabilities for managing vulnerability settings in a WordPress environment.
 *
 * This function verifies if the current user has the appropriate permissions to manage network settings in a multisite installation
 * or manage options in a single site installation. It is designed to be used in the context of a plugin or theme that interacts with
 * WordPress vulnerability settings.
 *
 * @since 3.0.0
 *
 * @return bool Returns false if the current user does not have the required capabilities, void otherwise.
 */
function wpvulnerability_capabilities() {

	if ( ! ( is_multisite() && ( is_network_admin() || is_main_site() ) && current_user_can( 'manage_network' ) ) ) {
		return false;
	} elseif ( ! ( ! is_multisite() && is_admin() && current_user_can( 'manage_options' ) ) ) {
		return false;
	}
}

/**
 * Sanitize a version string.
 *
 * @version 2.0.0
 *
 * @param string $version The version string to sanitize.
 *
 * @return string The sanitized version string.
 */
function wpvulnerability_sanitize_version( $version ) {

	// Remove any leading/trailing whitespace.
	// Strip out any non-alphanumeric characters except for hyphens, underscores, and dots.
	$version = trim( preg_replace( '/[^a-zA-Z0-9_\-.]+/', '', $version ) );

	return $version;
}

/**
 * Sanitize a PHP version string to ensure it follows the standard format.
 *
 * This function checks the input version string against a regular expression
 * to match the standard PHP versioning format (major.minor[.patch]).
 * It returns the matched version if it conforms to the expected format,
 * otherwise, it returns the original input version.
 *
 * @since 3.1.0 Introduced.
 *
 * @param string $version The PHP version string to sanitize.
 * @return string The sanitized version string, if it matches the standard format; otherwise, the original version string.
 */
function wpvulnerability_sanitize_version_php( $version ) {

	$version = wpvulnerability_sanitize_version( $version );

	// Validate format (major.minor[.patch]) and sanitize.
	if ( preg_match( '/^(\d+\.\d+(\.\d+)?)/', $version, $match ) ) {
		if ( isset( $match[0] ) ) {
			return trim( $match[0] );
		}
	}

	return $version;
}

/**
 * Returns a human-readable HTML entity for the given comparison operator.
 *
 * @version 2.0.0
 *
 * @param string $op The operator string to prettify.
 *
 * @return string The pretty operator HTML string.
 */
function wpvulnerability_pretty_operator( $op ) {

	switch ( trim( strtolower( $op ) ) ) {
		// Less than.
		case 'lt':
			return '&lt;&nbsp;';
		// Less than or equal to.
		case 'le':
			return '&le;&nbsp;';
		// Greater than.
		case 'gt':
			return '&gt;&nbsp;';
		// Greater than or equal to.
		case 'ge':
			return '&ge;&nbsp;';
		// Equal to.
		case 'eq':
			return '&equals;&nbsp;';
		// Not equal to.
		case 'ne':
			return '&ne;&nbsp;';
		// Return the original operator if it's not recognized.
		default:
			return $op;
	}
}

/**
 * Returns a human-readable HTML entity for the given comparison operator.
 *
 * @version 2.0.0
 *
 * @param string $op The operator string to prettify.
 *
 * @return string The pretty operator HTML string.
 */
function wpvulnerability_pretty_operator_cli( $op ) {

	switch ( trim( strtolower( $op ) ) ) {
		// Less than.
		case 'lt':
			return '< ';
		// Less than or equal to.
		case 'le':
			return '<= ';
		// Greater than.
		case 'gt':
			return '> ';
		// Greater than or equal to.
		case 'ge':
			return '>= ';
		// Equal to.
		case 'eq':
			return '= ';
		// Not equal to.
		case 'ne':
			return '!= ';
		// Return the original operator if it's not recognized.
		default:
			return $op;
	}
}

/**
 * Returns a Severity
 *
 * @version 2.0.0
 *
 * @param string $severity The severity string to prettify.
 *
 * @return string The severity string.
 */
function wpvulnerability_severity( $severity ) {

	switch ( trim( strtolower( $severity ) ) ) {
		case 'n':
			/* translators: Severity: None */
			return __( 'None', 'wpvulnerability' );
		case 'l':
			/* translators: Severity: Low */
			return __( 'Low', 'wpvulnerability' );
		case 'm':
			/* translators: Severity: Medium */
			return __( 'Medium', 'wpvulnerability' );
		case 'm':
			/* translators: Severity: High */
			return __( 'High', 'wpvulnerability' );
		case 'c':
			/* translators: Severity: Critical */
			return __( 'Critical', 'wpvulnerability' );
		// Return the original severity if it's not recognized.
		default:
			return $severity;
	}
}

/**
 * Retrieves vulnerabilities information from the API.
 *
 * @version 2.0.0
 *
 * @param string $type The type of vulnerability. Can be 'core', 'plugin' or 'theme'.
 * @param string $slug The slug of the plugin or theme. For core vulnerabilities, it is the version string.
 *
 * @return array|bool An array with the vulnerability information or false if there's an error.
 */
function wpvulnerability_get( $type, $slug = '' ) {

	$vulnerability_data = null;

	// Validate vulnerability type.
	switch ( trim( strtolower( $type ) ) ) {
		case 'core':
			$type = 'core';
			break;
		case 'plugin':
			$type = 'plugin';
			break;
		case 'theme':
			$type = 'theme';
			break;
		default:
			wp_die( 'Unknown vulnerability type sent.' );
			break;
	}

	// Validate slug.
	if ( 'plugin' === $type || 'theme' === $type ) {

		if ( empty( sanitize_title( $slug ) ) ) {
			return false;
		}
	} elseif ( 'core' === $type ) {

		if ( ! wpvulnerability_sanitize_version( $slug ) ) {
			return false;
		}
	}

	// Cache key.
	$key = 'wpvulnerability_' . $type . '_' . $slug;
	if ( is_multisite() ) {
		$vulnerability_data = get_site_transient( $key );
	} else {
		$vulnerability_data = get_transient( $key );
	}

	// If not cached, get the updated data.
	if ( empty( $vulnerability_data ) ) {

		$url      = WPVULNERABILITY_API_HOST . $type . '/' . $slug . '/';
		$response = wp_remote_get( $url, array( 'timeout' => 2500 ) );

		if ( ! is_wp_error( $response ) ) {

			$body = wp_remote_retrieve_body( $response );
			if ( is_multisite() ) {
				set_site_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_site_transient( $key );
			} else {
				set_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_transient( $key );
			}
		}
	}

	return json_decode( $vulnerability_data, true );
}

/**
 * Retrieve vulnerabilities for a specific version of WordPress Core.
 *
 * @since 2.0.0
 *
 * @param string|null $version The version number of WordPress Core. If null, retrieves for the installed version.
 *
 * @return array|false Array of vulnerabilities, or false on error.
 */
function wpvulnerability_get_core( $version = null ) {

	// Sanitize the version number.
	if ( ! wpvulnerability_sanitize_version( $version ) ) {
		$version = null;
	}

	// If version number is null, retrieve for the installed version.
	if ( is_null( $version ) ) {
		$version = get_bloginfo( 'version' );
	}

	// Get vulnerabilities from API.
	$response = wpvulnerability_get( 'core', $version );

	// Check for errors.
	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Process vulnerabilities and return as an array.
	$vulnerability = array();
	foreach ( $response['data']['vulnerability'] as $v ) {

		$vulnerability[] = array(
			'name'   => wp_kses( $v['name'], 'strip' ),
			'link'   => esc_url_raw( $v['link'] ),
			'source' => $v['source'],
			'impact' => $v['impact'],
		);

	}
	return $vulnerability;
}

/**
 * Retrieves vulnerabilities for a specified plugin, optionally returning general plugin data.
 *
 * This function sanitizes the plugin slug and verifies the version number before querying the vulnerability API.
 * If `$data` is set to 1, it returns general information about the plugin instead of vulnerabilities.
 * The function returns an array of vulnerabilities or plugin data based on the `$data` parameter, or `false`
 * if no vulnerabilities are found or the version number is invalid and `$data` is not set.
 *
 * @since 2.0.0 Introduced.
 *
 * @param string $slug    The slug of the plugin to check for vulnerabilities.
 * @param string $version The version of the plugin to check. The function may return `false` if this is invalid and `$data` is not set.
 * @param int    $data    Optional. Set to 1 to return general plugin data instead of vulnerabilities. Default 0 (return vulnerabilities).
 *
 * @return array|false An array of vulnerabilities or plugin data if `$data` is set to 1, or `false` if no vulnerabilities are found or the version number is invalid and `$data` is not set.
 */
function wpvulnerability_get_plugin( $slug, $version, $data = 0 ) {

	// Sanitize the plugin slug.
	$slug = sanitize_title( $slug );

	// If the version number is invalid, return false.
	if ( ! wpvulnerability_sanitize_version( $version ) && ! $data ) {
		return false;
	}

	// Get the response from the vulnerability API.
	$response = wpvulnerability_get( 'plugin', $slug );

	// Create an empty array to store the vulnerabilities.
	$vulnerability = array();

	if ( 1 === $data ) {

		$vulnerability = array(
			'name'   => wp_kses( $response['data']['name'], 'strip' ),
			'link'   => esc_url( $response['data']['link'] ),
			'latest' => number_format( (int) $response['data']['latest'], 0, '.', '' ),
			'closed' => number_format( (int) $response['data']['closed'], 0, '.', '' ),
		);

	} else {

		// If there are no vulnerabilities, return false.
		if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
			return false;
		}

		// Loop through each vulnerability and check if it affects the specified version of the plugin.
		foreach ( $response['data']['vulnerability'] as $v ) {

			// If the vulnerability has minimum and maximum versions, check if the specified version falls within that range.
			if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

				if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

					// Add the vulnerability to the array.
					$vulnerability[] = array(
						'name'        => wp_kses( $v['name'], 'strip' ),
						'description' => wp_kses_post( $v['description'] ),
						'versions'    => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
						'version'     => wp_kses( $v['operator']['min_version'], 'strip' ),
						'unfixed'     => (int) $v['operator']['unfixed'],
						'closed'      => (int) $v['operator']['closed'],
						'source'      => $v['source'],
						'impact'      => $v['impact'],
					);

				}

				// If the vulnerability has only a maximum version, check if the specified version is below that version.
			} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

				if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

					// Add the vulnerability to the list.
					$vulnerability[] = array(
						'name'        => wp_kses( $v['name'], 'strip' ),
						'description' => wp_kses_post( $v['description'] ),
						'versions'    => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
						'version'     => wp_kses( $v['operator']['max_version'], 'strip' ),
						'unfixed'     => (int) $v['operator']['unfixed'],
						'closed'      => (int) $v['operator']['closed'],
						'source'      => $v['source'],
						'impact'      => $v['impact'],
					);

				}

				// If the vulnerability has a minimum version and maximum version, check if the specified version is within that range.
			} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

				if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

					// Add the vulnerability to the list.
					$vulnerability[] = array(
						'name'        => wp_kses( $v['name'], 'strip' ),
						'description' => wp_kses_post( $v['description'] ),
						'versions'    => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'], 'strip' ),
						'version'     => wp_kses( $v['operator']['min_version'], 'strip' ),
						'unfixed'     => (int) $v['operator']['unfixed'],
						'closed'      => (int) $v['operator']['closed'],
						'source'      => $v['source'],
						'impact'      => $v['impact'],
					);

				}
			}
		}
	}

	return $vulnerability;
}

/**
 * Get vulnerabilities for a specific theme.
 *
 * @since 2.0.0
 *
 * @param string $slug    Slug of the theme.
 * @param string $version Version of the theme.
 *
 * @return array|false Returns an array of vulnerabilities, or false if there are none.
 */
function wpvulnerability_get_theme( $slug, $version ) {

	// Sanitize the plugin slug.
	$slug = sanitize_title( $slug );

	// Get the response from the vulnerability API.
	if ( ! wpvulnerability_sanitize_version( $version ) ) {
		return false;
	}

	// Get the response from the vulnerability API.
	$response = wpvulnerability_get( 'theme', $slug );

	// Create an empty array to store the vulnerabilities.
	$vulnerability = array();

	// If there are no vulnerabilities, return false.
	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	foreach ( $response['data']['vulnerability'] as $v ) {

		// If the vulnerability has minimum and maximum versions, check if the specified version falls within that range.
		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'        => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions'    => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_version'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'     => wp_kses( $v['operator']['min_version'], 'strip' ),
					'unfixed'     => (int) $v['operator']['unfixed'],
					'closed'      => (int) $v['operator']['closed'],
					'source'      => $v['source'],
					'impact'      => $v['impact'],
				);

			}

			// If the vulnerability has only a maximum version, check if the specified version is below that version.
		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'        => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions'    => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'     => wp_kses( $v['operator']['max_version'], 'strip' ),
					'unfixed'     => (int) $v['operator']['unfixed'],
					'closed'      => (int) $v['operator']['closed'],
					'source'      => $v['source'],
					'impact'      => $v['impact'],
				);

			}

			// If the vulnerability has only a maximum version, check if the specified version is below that version.
		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'        => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions'    => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_version'] ) . $v['operator']['min_version'], 'strip' ),
					'version'     => wp_kses( $v['operator']['min_version'], 'strip' ),
					'unfixed'     => (int) $v['operator']['unfixed'],
					'closed'      => (int) $v['operator']['closed'],
					'source'      => $v['source'],
					'impact'      => $v['impact'],
				);

			}
		}
	}

	return $vulnerability;
}

/**
 * Get statistics
 *
 * Returns an array with statistical information about vulnerabilities and their respective products.
 *
 * @since 2.0.0
 *
 * @return array|false Returns an array with the statistical information if successful, false otherwise.
 */
function wpvulnerability_get_statistics() {

	$key = 'wpvulnerability_stats';

	// Get cached statistics if available.
	if ( is_multisite() ) {
		$vulnerability = get_site_transient( $key );
	} else {
		$vulnerability = get_transient( $key );
	}

	// If cached statistics are not available, retrieve them from the API and store them in cache.
	if ( empty( $vulnerability ) ) {

		$url = WPVULNERABILITY_API_HOST;
		// Parse the JSON response into an associative array.
		$response = wp_remote_get( $url, array( 'timeout' => 2500 ) );

		if ( ! is_wp_error( $response ) ) {

			$body = wp_remote_retrieve_body( $response );
			if ( is_multisite() ) {
				set_site_transient( $key, $body, HOUR_IN_SECONDS * 24 );
				$vulnerability = get_site_transient( $key );
			} else {
				set_transient( $key, $body, HOUR_IN_SECONDS * 24 );
				$vulnerability = get_transient( $key );
			}
		}
	}

	// If the response does not contain statistics, return false.
	$response = json_decode( $vulnerability, true );

	if ( ! isset( $response['stats'] ) ) {
		return false;
	}

	// Return an array with statistical information.
	return array(
		'core'    => array(
			'versions' => (int) $response['stats']['products']['core'],
		),
		'plugins' => array(
			'products'        => (int) $response['stats']['products']['plugins'],
			'vulnerabilities' => (int) $response['stats']['plugins'],
		),
		'themes'  => array(
			'products'        => (int) $response['stats']['products']['themes'],
			'vulnerabilities' => (int) $response['stats']['themes'],
		),
		'updated' => array(
			'unixepoch' => (int) $response['updated'],
			'datetime'  => gmdate( 'Y-m-d H:i:s', (int) $response['updated'] ),
			'iso8601'   => gmdate( 'c', (int) $response['updated'] ),
			'rfc2822'   => gmdate( 'r', (int) $response['updated'] ),
		),
	);
}

/**
 * Get vulnerabilities for a specific PHP version.
 *
 * @since 3.0.0
 *
 * @param string $version PHP Version.
 *
 * @return array|false Returns an array of vulnerabilities, or false if there are none.
 */
function wpvulnerability_get_php( $version ) {

	$key                = 'wpvulnerability_php';
	$vulnerability_data = null;
	$vulnerability      = array();

	// Get cached statistics if available.
	if ( is_multisite() ) {
		$vulnerability_data = get_site_transient( $key );
	} else {
		$vulnerability_data = get_transient( $key );
	}

	// If cached statistics are not available, retrieve them from the API and store them in cache.
	if ( empty( $vulnerability_data ) ) {

		$url = WPVULNERABILITY_API_HOST . 'php/' . $version . '/';
		// Parse the JSON response into an associative array.
		$response = wp_remote_get( $url, array( 'timeout' => 2500 ) );

		if ( ! is_wp_error( $response ) ) {

			$body = wp_remote_retrieve_body( $response );
			if ( is_multisite() ) {
				set_site_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_site_transient( $key );
			} else {
				set_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );
				$vulnerability_data = get_transient( $key );
			}
		}
	}

	// If the response does not contain statistics, return false.
	$response = json_decode( $vulnerability_data, true );

	if ( ( isset( $response['error'] ) && $response['error'] ) || empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	foreach ( $response['data']['vulnerability'] as $v ) {

		// If the vulnerability has minimum and maximum versions, check if the specified version falls within that range.
		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_version'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}

			// If the vulnerability has only a maximum version, check if the specified version is below that version.
		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version'  => wp_kses( $v['operator']['max_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}

			// If the vulnerability has only a maximum version, check if the specified version is below that version.
		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name'     => wp_kses( $v['name'], 'strip' ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_version'] ) . $v['operator']['min_version'], 'strip' ),
					'version'  => wp_kses( $v['operator']['min_version'], 'strip' ),
					'unfixed'  => (int) $v['operator']['unfixed'],
					'source'   => $v['source'],
				);

			}
		}
	}

	return $vulnerability;
}
