<?php
/**
 * Running functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Conditionally adds a settings link to the plugin row in the plugins list for a multisite network or single site installation.
 *
 * In a multisite environment, it adds a network admin settings link that appears in the network admin plugin row.
 * For a single site installation, it adds a standard settings link that appears in the plugin row.
 *
 * @param array $links The links that appear in the plugin row.
 *
 * @return array The modified array of links.
 *
 * @version 3.0.0
 */
if ( is_multisite() && is_network_admin() ) {

	/**
	 * Adds a network admin settings link to the plugin row in the network admin plugins list.
	 *
	 * @param array $links The links that appear in the network admin plugin row.
	 *
	 * @return array The modified array of links.
	 *
	 * @version 3.0.0
	 */
	function wpvulnerability_add_network_settings_link( $links ) {
		if ( current_user_can( 'manage_network' ) ) {
			$links[] = '<a href="' . network_admin_url( 'settings.php?page=wpvulnerability-options' ) . '">' . __( 'Network Settings', 'wpvulnerability' ) . '</a>';
		}
		return $links;
	}
	add_filter( 'network_admin_plugin_action_links_' . WPVULNERABILITY_PLUGIN_BASE, 'wpvulnerability_add_network_settings_link' );

} elseif ( ! is_multisite() && is_admin() ) {

	/**
	 * Adds a standard settings link to the plugin row in the plugins list for a single site installation.
	 *
	 * @param array $links The links that appear in the plugin row.
	 *
	 * @return array The modified array of links.
	 *
	 * @version 2.0.0
	 */
	function wpvulnerability_add_settings_link( $links ) {
		if ( current_user_can( 'manage_options' ) ) {
			$links[] = '<a href="' . get_admin_url( null, 'options-general.php?page=wpvulnerability-options' ) . '">' . __( 'Settings', 'wpvulnerability' ) . '</a>';
		}
		return $links;
	}
	add_filter( 'plugin_action_links_' . WPVULNERABILITY_PLUGIN_BASE, 'wpvulnerability_add_settings_link' );

}

/**
 * Updates the plugin's data
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_update_database_data() {

	if ( ! function_exists( 'wpvulnerability_core_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-core.php';
	}
	if ( ! function_exists( 'wpvulnerability_plugin_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-plugins.php';
	}
	if ( ! function_exists( 'wpvulnerability_theme_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-themes.php';
	}
	if ( ! function_exists( 'wpvulnerability_php_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-php.php';
	}
	// Update core vulnerabilities.
	wpvulnerability_core_get_vulnerabilities_clean();
	// Update plugin vulnerabilities.
	wpvulnerability_plugin_get_vulnerabilities_clean();
	// Update theme vulnerabilities.
	wpvulnerability_theme_get_vulnerabilities_clean();
	// Update PHP vulnerabilities.
	wpvulnerability_php_get_vulnerabilities_clean();
}

/**
 * Updates the plugin's data when expired
 *
 * @since 3.0.0
 *
 * @return void
 */
function wpvulnerability_expired_database_data() {

	if ( ! function_exists( 'wpvulnerability_core_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-core.php';
	}
	if ( ! function_exists( 'wpvulnerability_plugin_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-plugins.php';
	}
	if ( ! function_exists( 'wpvulnerability_theme_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-themes.php';
	}
	if ( ! function_exists( 'wpvulnerability_php_get_vulnerabilities_clean' ) ) {
		require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-php.php';
	}

	$cache_time = time();

	if ( is_multisite() ) {

		if ( json_decode( get_site_option( 'wpvulnerability-core-cache' ) ) < $cache_time ) {
			wpvulnerability_core_get_vulnerabilities_clean();
		}
		if ( json_decode( get_site_option( 'wpvulnerability-plugins-cache' ) ) < $cache_time ) {
			wpvulnerability_plugin_get_vulnerabilities_clean();
		}
		if ( json_decode( get_site_option( 'wpvulnerability-themes-cache' ) ) < $cache_time ) {
			wpvulnerability_theme_get_vulnerabilities_clean();
		}
		if ( json_decode( get_site_option( 'wpvulnerability-php-cache' ) ) < $cache_time ) {
			wpvulnerability_php_get_vulnerabilities_clean();
		}
	} elseif ( ! is_multisite() ) {

		if ( json_decode( get_option( 'wpvulnerability-core-cache' ) ) < $cache_time ) {
			wpvulnerability_core_get_vulnerabilities_clean();
		}
		if ( json_decode( get_option( 'wpvulnerability-plugins-cache' ) ) < $cache_time ) {
			wpvulnerability_plugin_get_vulnerabilities_clean();
		}
		if ( json_decode( get_option( 'wpvulnerability-themes-cache' ) ) < $cache_time ) {
			wpvulnerability_theme_get_vulnerabilities_clean();
		}
		if ( json_decode( get_option( 'wpvulnerability-php-cache' ) ) < $cache_time ) {
			wpvulnerability_php_get_vulnerabilities_clean();
		}
	}

	unset( $cache_time );
}

/**
 * Callback function for when the plugin is activated.
 * Adds plugin data options if they are not already created.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_activation() {

	if ( is_multisite() ) {

		// Add wpvulnerability-config option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-config' ) ) {

			$config = get_option( 'wpvulnerability-config' );

			if ( isset( $config['emails'] ) && $config['emails'] && isset( $config['period'] ) && $config['period'] ) {

				add_site_option(
					'wpvulnerability-config',
					array(
						'emails' => $config['emails'],
						'period' => $config['period'],
					)
				);
				unset( $config );

			} else {

				add_site_option(
					'wpvulnerability-config',
					array(
						'emails' => get_bloginfo( 'admin_email' ),
						'period' => 'weekly',
					)
				);
			}
		}

		// Add wpvulnerability-plugins option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-plugins' ) ) {
			add_site_option( 'wpvulnerability-plugins', '' );
		}

		// Add wpvulnerability-plugins-cache option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-plugins-cache' ) ) {
			add_site_option( 'wpvulnerability-plugins-cache', 0 );
		}

		// Add wpvulnerability-plugins-vulnerable option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-plugins-vulnerable' ) ) {
			add_site_option( 'wpvulnerability-plugins-vulnerable', 0 );
		}

		// Add wpvulnerability-themes option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-themes' ) ) {
			add_site_option( 'wpvulnerability-themes', '' );
		}

		// Add wpvulnerability-themes-cache option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-themes-cache' ) ) {
			add_site_option( 'wpvulnerability-themes-cache', 0 );
		}

		// Add wpvulnerability-themes-vulnerable option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-themes-vulnerable' ) ) {
			add_site_option( 'wpvulnerability-themes-vulnerable', 0 );
		}

		// Add wpvulnerability-core option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-core' ) ) {
			add_site_option( 'wpvulnerability-core', '' );
		}

		// Add wpvulnerability-core-cache option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-core-cache' ) ) {
			add_site_option( 'wpvulnerability-core-cache', 0 );
		}

		// Add wpvulnerability-core-vulnerable option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-core-vulnerable' ) ) {
			add_site_option( 'wpvulnerability-core-vulnerable', 0 );
		}

		// Add wpvulnerability-php option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-php' ) ) {
			add_site_option( 'wpvulnerability-php', '' );
		}

		// Add wpvulnerability-php-cache option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-php-cache' ) ) {
			add_site_option( 'wpvulnerability-php-cache', 0 );
		}

		// Add wpvulnerability-php-vulnerable option if it does not exist.
		if ( ! get_site_option( 'wpvulnerability-php-vulnerable' ) ) {
			add_site_option( 'wpvulnerability-php-vulnerable', 0 );
		}
	} elseif ( ! is_multisite() ) {

		// Add wpvulnerability-config option if it does not exist.
		if ( ! get_option( 'wpvulnerability-config' ) ) {
			add_option(
				'wpvulnerability-config',
				array(
					'emails' => get_bloginfo( 'admin_email' ),
					'period' => 'weekly',
				)
			);
		}

		// Add wpvulnerability-plugins option if it does not exist.
		if ( ! get_option( 'wpvulnerability-plugins' ) ) {
			add_option( 'wpvulnerability-plugins', '' );
		}

		// Add wpvulnerability-plugins-cache option if it does not exist.
		if ( ! get_option( 'wpvulnerability-plugins-cache' ) ) {
			add_option( 'wpvulnerability-plugins-cache', 0 );
		}

		// Add wpvulnerability-plugins-vulnerable option if it does not exist.
		if ( ! get_option( 'wpvulnerability-plugins-vulnerable' ) ) {
			add_option( 'wpvulnerability-plugins-vulnerable', 0 );
		}

		// Add wpvulnerability-themes option if it does not exist.
		if ( ! get_option( 'wpvulnerability-themes' ) ) {
			add_option( 'wpvulnerability-themes', '' );
		}

		// Add wpvulnerability-themes-cache option if it does not exist.
		if ( ! get_option( 'wpvulnerability-themes-cache' ) ) {
			add_option( 'wpvulnerability-themes-cache', 0 );
		}

		// Add wpvulnerability-themes-vulnerable option if it does not exist.
		if ( ! get_option( 'wpvulnerability-themes-vulnerable' ) ) {
			add_option( 'wpvulnerability-themes-vulnerable', 0 );
		}

		// Add wpvulnerability-core option if it does not exist.
		if ( ! get_option( 'wpvulnerability-core' ) ) {
			add_option( 'wpvulnerability-core', '' );
		}

		// Add wpvulnerability-core-cache option if it does not exist.
		if ( ! get_option( 'wpvulnerability-core-cache' ) ) {
			add_option( 'wpvulnerability-core-cache', 0 );
		}

		// Add wpvulnerability-core-vulnerable option if it does not exist.
		if ( ! get_option( 'wpvulnerability-core-vulnerable' ) ) {
			add_option( 'wpvulnerability-core-vulnerable', 0 );
		}

		// Add wpvulnerability-php option if it does not exist.
		if ( ! get_option( 'wpvulnerability-php' ) ) {
			add_option( 'wpvulnerability-php', '' );
		}

		// Add wpvulnerability-php-cache option if it does not exist.
		if ( ! get_option( 'wpvulnerability-php-cache' ) ) {
			add_option( 'wpvulnerability-php-cache', 0 );
		}

		// Add wpvulnerability-php-vulnerable option if it does not exist.
		if ( ! get_option( 'wpvulnerability-php-vulnerable' ) ) {
			add_option( 'wpvulnerability-php-vulnerable', 0 );
		}
	}
}

/**
 * On Deactivation
 * Callback function to run when the plugin is deactivated
 * Deletes options and removes scheduled wp-cron jobs.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_deactivation() {

	if ( is_multisite() ) {

		// Delete all plugin options.
		delete_site_option( 'wpvulnerability_settings' );
		delete_site_option( 'wpvulnerability-data' );

		delete_site_option( 'wpvulnerability-themes' );
		delete_site_option( 'wpvulnerability-themes-cache' );
		delete_site_option( 'wpvulnerability-themes-vulnerable' );
		delete_site_option( 'wpvulnerability-plugins' );
		delete_site_option( 'wpvulnerability-plugins-cache' );
		delete_site_option( 'wpvulnerability-plugins-vulnerable' );
		delete_site_option( 'wpvulnerability-core' );
		delete_site_option( 'wpvulnerability-core-cache' );
		delete_site_option( 'wpvulnerability-core-vulnerable' );
		delete_site_option( 'wpvulnerability-php' );
		delete_site_option( 'wpvulnerability-php-cache' );
		delete_site_option( 'wpvulnerability-php-vulnerable' );

	} elseif ( ! is_multisite() ) {

		// Delete all plugin options.
		delete_option( 'wpvulnerability_settings' );
		delete_option( 'wpvulnerability-data' );

		delete_option( 'wpvulnerability-themes' );
		delete_option( 'wpvulnerability-themes-cache' );
		delete_option( 'wpvulnerability-themes-vulnerable' );
		delete_option( 'wpvulnerability-plugins' );
		delete_option( 'wpvulnerability-plugins-cache' );
		delete_option( 'wpvulnerability-plugins-vulnerable' );
		delete_option( 'wpvulnerability-core' );
		delete_option( 'wpvulnerability-core-cache' );
		delete_option( 'wpvulnerability-core-vulnerable' );
		delete_option( 'wpvulnerability-php' );
		delete_option( 'wpvulnerability-php-cache' );
		delete_option( 'wpvulnerability-php-vulnerable' );

	}

	// Unschedule and remove scheduled wp-cron jobs.
	wp_unschedule_event( wp_next_scheduled( 'wpvulnerability_notification' ), 'wpvulnerability_notification' );
	wp_clear_scheduled_hook( 'wpvulnerability_notification' );

	wp_unschedule_event( wp_next_scheduled( 'wpvulnerability_update_database' ), 'wpvulnerability_update_database' );
	wp_clear_scheduled_hook( 'wpvulnerability_update_database' );
}

/**
 * On Uninstall
 * Callback function to run when the plugin is uninstalled
 * Deletes options and removes scheduled wp-cron jobs.
 *
 * @since 3.0.0
 *
 * @return void
 */
function wpvulnerability_uninstall() {

	/**
	 * Deprecated options.
	 */
	delete_option( 'wpvulnerability_settings' );
	delete_option( 'wpvulnerability-data' );

	/**
	 * All the data.
	 */
	delete_option( 'wpvulnerability-themes' );
	delete_option( 'wpvulnerability-themes-cache' );
	delete_option( 'wpvulnerability-themes-vulnerable' );
	delete_option( 'wpvulnerability-plugins' );
	delete_option( 'wpvulnerability-plugins-cache' );
	delete_option( 'wpvulnerability-plugins-vulnerable' );
	delete_option( 'wpvulnerability-core' );
	delete_option( 'wpvulnerability-core-cache' );
	delete_option( 'wpvulnerability-core-vulnerable' );
	delete_option( 'wpvulnerability-php' );
	delete_option( 'wpvulnerability-php-cache' );
	delete_option( 'wpvulnerability-php-vulnerable' );

	/**
	 * Config data, not deleted when deactivated.
	 */
	delete_option( 'wpvulnerability-config' );

	/**
	 * All the data (Multisite).
	 */
	delete_site_option( 'wpvulnerability-themes' );
	delete_site_option( 'wpvulnerability-themes-cache' );
	delete_site_option( 'wpvulnerability-themes-vulnerable' );
	delete_site_option( 'wpvulnerability-plugins' );
	delete_site_option( 'wpvulnerability-plugins-cache' );
	delete_site_option( 'wpvulnerability-plugins-vulnerable' );
	delete_site_option( 'wpvulnerability-core' );
	delete_site_option( 'wpvulnerability-core-cache' );
	delete_site_option( 'wpvulnerability-core-vulnerable' );
	delete_site_option( 'wpvulnerability-php' );
	delete_site_option( 'wpvulnerability-php-cache' );
	delete_site_option( 'wpvulnerability-php-vulnerable' );

	/**
	 * Config data (Multisite), not deleted when deactivated.
	 */
	delete_site_option( 'wpvulnerability-config' );

	// Unschedule and remove scheduled wp-cron jobs.
	wp_unschedule_event( wp_next_scheduled( 'wpvulnerability_notification' ), 'wpvulnerability_notification' );
	wp_clear_scheduled_hook( 'wpvulnerability_notification' );

	wp_unschedule_event( wp_next_scheduled( 'wpvulnerability_update_database' ), 'wpvulnerability_update_database' );
	wp_clear_scheduled_hook( 'wpvulnerability_update_database' );
}

/**
 * Schedule Automatic Vulnerability Database Update
 * If the 'wpvulnerability_update_database' event is not already scheduled, schedule it to run twice daily.
 *
 * @since 2.0.0
 *
 * @return void
 */
if ( ! wp_next_scheduled( 'wpvulnerability_update_database' ) ) {

	if ( ( ! is_multisite() && is_admin() ) || ( is_multisite() && is_main_site() ) ) {
		wp_schedule_event( time(), 'twicedaily', 'wpvulnerability_update_database' );
	}
}
add_action( 'wpvulnerability_update_database', 'wpvulnerability_update_database_data' );

/**
 * Clean the cache after an update
 *
 * @since 2.0.0
 *
 * @return void
 */
add_action( 'upgrader_process_complete', 'wpvulnerability_update_database_data' );
