<?php
/**
 * Process functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

wpvulnerability_capabilities();

/**
 * Load the settings to be available always.
 *
 * @version 2.0.0
 *
 * @return array|false An array containing the WPVulnerability settings if they exist, or false if they don't.
 */
$wpvulnerability_settings = get_site_option( 'wpvulnerability-config' );

/**
 * Enqueues the WPVulnerability admin CSS file on all admin pages.
 *
 * @version 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_enqueue_scripts() {

	wp_enqueue_style(
		'wpvulnerability-admin',
		WPVULNERABILITY_PLUGIN_URL . 'assets/admin.css',
		array(),
		WPVULNERABILITY_PLUGIN_VERSION
	);
}
add_action( 'admin_enqueue_scripts', 'wpvulnerability_admin_enqueue_scripts' );

/**
 * Create the WP-Admin options page
 * This function generates the HTML output for the WPVulnerability settings page in the WP-Admin.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_create_admin_page() {

	/**
	 * Processes the form submission for the WPVulnerability plugin settings in a multisite network admin context.
	 *
	 * Checks if the current request is a submission from the WPVulnerability settings page.
	 * Verifies the security nonce to prevent CSRF attacks. If the verification fails or if the request
	 * is not from a network admin or the admin dashboard, the function will halt execution and display an error.
	 * Otherwise, it sanitizes and updates the plugin settings, reschedules any relevant wp-cron events,
	 * and registers a settings error to notify the user that the information has been updated.
	 *
	 * @since 3.0.0
	*/
	if ( isset( $_GET['page'] ) && 'wpvulnerability-options' === $_GET['page'] && ! empty( $_POST ) ) {

		if ( check_admin_referer( 'wpvulnerability_nonce', 'wpauto_nonce' ) ) {

			if ( isset( $_POST['wpvulnerability_submit'] ) && isset( $_POST['wpvulnerability-config'] ) ) {

				$wpvulnerability_sanitized_values['emails'] = null;
				$wpvulnerability_sanitized_values['period'] = 'weekly';

				$wpvulnerability_input['emails'] = array();
				if ( isset( $_POST['wpvulnerability-config']['emails'] ) ) {
					$wpvulnerability_config_emails    = wp_kses( wp_unslash( $_POST['wpvulnerability-config']['emails'] ), 'strip' );
					$wpvulnerability_input_email_text = explode( ',', $wpvulnerability_config_emails );
					foreach ( $wpvulnerability_input_email_text as $wpvulnerability_input_email ) {
						$wpvulnerability_input_email = sanitize_email( trim( $wpvulnerability_input_email ) );
						if ( is_email( $wpvulnerability_input_email ) ) {
							$wpvulnerability_input['emails'][] = $wpvulnerability_input_email;
						}
						unset( $wpvulnerability_input_email );
					}
					unset( $wpvulnerability_input_email_text, $wpvulnerability_config_emails );
				}
				if ( count( $wpvulnerability_input['emails'] ) ) {
					$wpvulnerability_sanitized_values['emails'] = implode( ',', $wpvulnerability_input['emails'] );
				} else {
					$wpvulnerability_sanitized_values['emails'] = null;
				}

				$wpvulnerability_input['period'] = null;
				if ( isset( $_POST['wpvulnerability-config']['period'] ) ) {
					$wpvulnerability_input['period'] = wp_kses( wp_unslash( $_POST['wpvulnerability-config']['period'] ), 'strip' );
				}
				switch ( $wpvulnerability_input['period'] ) {
					case 'daily':
						$wpvulnerability_sanitized_values['period'] = 'daily';
						break;
					case 'weekly':
					default:
						$wpvulnerability_sanitized_values['period'] = 'weekly';
						break;
				}
				unset( $wpvulnerability_input );

				update_site_option(
					'wpvulnerability-config',
					array(
						'emails' => $wpvulnerability_sanitized_values['emails'],
						'period' => $wpvulnerability_sanitized_values['period'],
					)
				);

				$current_schedule = wp_get_schedule( 'wpvulnerability_notification' );
				if ( $current_schedule !== $wpvulnerability_sanitized_values['period'] ) {
					wp_clear_scheduled_hook( 'wpvulnerability_notification' );
					wp_schedule_event( time(), $wpvulnerability_sanitized_values['period'], 'wpvulnerability_notification' );
				}
				unset( $current_schedule );

				unset( $wpvulnerability_sanitized_values );

				add_settings_error(
					'wpvulnerability-messages',
					'wpvulnerability-updated',
					__( 'Settings saved.', 'wpvulnerability' ),
					'success'
				);

			}
		}
	}

	/**
	 * Reset the data
	 *
	 * @since 3.0.0
	 */
	if ( isset( $_POST['wpvulnerability_reset'] ) && check_admin_referer( 'wpvulnerability_nonce', 'wpauto_nonce' ) ) {

		// Calls the reset function.
		wpvulnerability_update_database_data();

		set_transient( 'wpvulnerability_message_manual_success', __( 'Data from source has been reloaded.', 'wpvulnerability' ), 10 );

	}

	/**
	 * Send an test email
	 *
	 * @since 3.0.0
	 */
	if ( isset( $_POST['wpvulnerability_email'] ) && check_admin_referer( 'wpvulnerability_nonce', 'wpauto_nonce' ) ) {

		if ( ! function_exists( 'wpvulnerability_execute_notification' ) ) {
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-core.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-plugins.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-themes.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-php.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-process.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-notifications.php';
		}

		// Calls the notifications function, forced.
		$wpmail = wpvulnerability_execute_notification( true );

		if ( $wpmail ) {

			set_transient( 'wpvulnerability_message_manual_success', __( 'Test email has been sent.', 'wpvulnerability' ), 10 );

		} else {

			set_transient( 'wpvulnerability_message_manual_error', __( 'Test email has failed. Please, check your email settings.', 'wpvulnerability' ), 10 );

		}
	}

	$wpvulnerability_settings = get_site_option( 'wpvulnerability-config' );
	?>
	<div class="header-wrap">
		<div class="wrapper">
			<div class="header wpvulnerability-header">
				<div class="logo">
					<img src="<?php echo esc_url( WPVULNERABILITY_PLUGIN_URL ); ?>assets/logo64.png" style="height: 64px; vertical-align: text-top; width: 64px;" alt="" title="WPVulnerability">
					<h2><?php esc_html_e( 'WPVulnerability settings', 'wpvulnerability' ); ?></h2>
				</div>
			</div>
		</div>
	</div>
	<div class="wrap">
		<?php
		$wpvulnerability_message_manual_success = get_transient( 'wpvulnerability_message_manual_success' );
		if ( $wpvulnerability_message_manual_success ) {
			echo '<div class="notice notice-success"><p>' . esc_html( $wpvulnerability_message_manual_success ) . '</p></div>';
			delete_transient( 'wpvulnerability_message_manual_success' );
			unset( $wpvulnerability_message_manual_success );
		}
		$wpvulnerability_message_manual_error = get_transient( 'wpvulnerability_message_manual_error' );
		if ( $wpvulnerability_message_manual_error ) {
			echo '<div class="notice notice-error"><p>' . esc_html( $wpvulnerability_message_manual_error ) . '</p></div>';
			delete_transient( 'wpvulnerability_message_manual_error' );
			unset( $wpvulnerability_message_manual_error );
		}
		?>
		<?php settings_errors( 'wpvulnerability-messages', true ); ?>
		<form method="post">
			<?php
				settings_fields( 'admin_wpvulnerability_settings' );
				do_settings_sections( 'wpvulnerability-config' );
				wp_nonce_field( 'wpvulnerability_nonce', 'wpauto_nonce' );
				submit_button(
					__( 'Save settings', 'wpvulnerability' ),
					'primary',
					'wpvulnerability_submit'
				);
			?>
		</form>
		<section class="section">
			<header class="section-header">
				<hr>
				<h3><?php esc_html_e( 'Reload the data from source', 'wpvulnerability' ); ?></h3>
			</header>
			<div class="section-content">
				<p><?php esc_html_e( 'Reload all Core, Plugins, Themes and other components information directly from the API to have updated data.', 'wpvulnerability' ); ?></p>
				<form method="post" action="settings.php?page=wpvulnerability-options">
					<?php wp_nonce_field( 'wpvulnerability_nonce', 'wpauto_nonce' ); ?>
					<input type="submit" name="wpvulnerability_reset" value="<?php esc_attr_e( 'Reload Data', 'wpvulnerability' ); ?>" class="button button-secondary">
				</form>
			</div>
		</section>
		<section class="section">
			<header class="section-header">
				<hr>
				<h3><?php esc_html_e( 'Email test', 'wpvulnerability' ); ?></h3>
			</header>
			<div class="section-content">
				<p><?php esc_html_e( 'Send an email with the vulnerabilities (or empty).', 'wpvulnerability' ); ?></p>
				<form method="post" action="settings.php?page=wpvulnerability-options">
					<?php wp_nonce_field( 'wpvulnerability_nonce', 'wpauto_nonce' ); ?>
					<input type="submit" name="wpvulnerability_email" value="<?php esc_attr_e( 'Send email', 'wpvulnerability' ); ?>" class="button button-secondary">
				</form>
			</div>
		</section>
	<?php
}

/**
 * Adds a WP-Admin menu option for the WPVulnerability plugin
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_menu() {

	// Adds a submenu page under the Settings menu.
	add_submenu_page(
		'settings.php',
		__( 'WPVulnerability', 'wpvulnerability' ),
		__( 'WPVulnerability', 'wpvulnerability' ),
		'manage_network_options',
		'wpvulnerability-options',
		'wpvulnerability_create_admin_page'
	);
}
add_action( 'network_admin_menu', 'wpvulnerability_admin_menu' );

/**
 * Print the settings header information for the notifications section.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_section_notifications() {

	// Output the header information for the notifications section.
	esc_html_e( 'Configure and save these settings to receive email notifications.', 'wpvulnerability' );
}

/**
 * Callback function to display the email input field in the admin settings page.
 * This function retrieves the current WPVulnerability plugin settings and displays the email input field
 * for users to enter their email addresses. If no email is saved in the settings, the admin email is displayed.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_emails_callback() {

	// Retrieve the WPVulnerability plugin settings.
	$wpvulnerability_settings = get_site_option( 'wpvulnerability-config' );

	// Set a default value for the email input field if no email is saved in the settings.
	if ( ! isset( $wpvulnerability_settings['emails'] ) ) {
		$wpvulnerability_settings['emails'] = '';
	}

	// Output the email input field. Use the network admin email as a placeholder in a multisite environment.
	$admin_email = get_site_option( 'admin_email' );

	// Output the email input field and display the admin email as a hint.
	?>
	<input class="regular-text" type="text" name="wpvulnerability-config[emails]" id="wpvulnerability_emails" placeholder="<?php echo esc_attr( $admin_email ); ?>" value="<?php echo esc_attr( $wpvulnerability_settings['emails'] ); ?>">
	<br><small><?php esc_html_e( 'Default administrator email', 'wpvulnerability' ); ?>: <?php echo esc_attr( $admin_email ); ?></small>
	<?php

	unset( $admin_email );
}

/**
 * Print when to send the vulnerability scan emails.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_period_callback() {

	// Get the saved plugin settings.
	$wpvulnerability_settings = get_site_option( 'wpvulnerability-config' );

	// Set the default period to 'weekly' if not set or empty.
	if ( ! isset( $wpvulnerability_settings['period'] ) || empty( $wpvulnerability_settings['period'] ) ) {
		$wpvulnerability_settings['period'] = 'weekly';
	}

	// Output the period select box.
	?>
	<select name="wpvulnerability-config[period]" id="wpvulnerability_period">
		<option value="weekly" <?php selected( $wpvulnerability_settings['period'], 'weekly' ); ?>><?php esc_html_e( 'Weekly', 'wpvulnerability' ); ?></option>
		<option value="daily" <?php selected( $wpvulnerability_settings['period'], 'daily' ); ?>><?php esc_html_e( 'Daily', 'wpvulnerability' ); ?></option>
	</select>
	<?php
}

/**
 * Content for the Dashboard Widget
 *
 * @since 2.2.0
 *
 * @return void
 */
function wpvulnerability_admin_dashboard_content() {
	// Get the number of core vulnerabilites from cache.
	$wpvulnerability_test_core_counter = json_decode( get_site_option( 'wpvulnerability-core-vulnerable' ) );
	if ( ! is_numeric( $wpvulnerability_test_core_counter ) ) {
		$wpvulnerability_test_core_counter = 0;
	}
	/* translators: Show the number of vulnerabilities in a WP-Admin dashboard */
	$msg_core = sprintf( _n( 'Core: %d vulnerability', 'Core: %d vulnerabilities', $wpvulnerability_test_core_counter, 'wpvulnerability' ), $wpvulnerability_test_core_counter );

	// Get the number of themes vulnerabilites from cache.
	$wpvulnerability_test_themes_counter = json_decode( get_site_option( 'wpvulnerability-themes-vulnerable' ) );
	if ( ! is_numeric( $wpvulnerability_test_themes_counter ) ) {
		$wpvulnerability_test_themes_counter = 0;
	}
	/* translators: Show the number of vulnerabilities in a WP-Admin dashboard */
	$msg_themes = sprintf( _n( 'Themes: %d vulnerability', 'Themes: %d vulnerabilities', $wpvulnerability_test_themes_counter, 'wpvulnerability' ), $wpvulnerability_test_themes_counter );

	// Get the number of plugins vulnerabilites from cache.
	$wpvulnerability_test_plugins_counter = json_decode( get_site_option( 'wpvulnerability-plugins-vulnerable' ) );
	if ( ! is_numeric( $wpvulnerability_test_plugins_counter ) ) {
		$wpvulnerability_test_plugins_counter = 0;
	}
	/* translators: Show the number of vulnerabilities in a WP-Admin dashboard */
	$msg_plugins = sprintf( _n( 'Plugins: %d vulnerability', 'Plugins: %d vulnerabilities', $wpvulnerability_test_plugins_counter, 'wpvulnerability' ), $wpvulnerability_test_plugins_counter );

	// Get the number of PHP vulnerabilites from cache.
	$wpvulnerability_test_php_counter = json_decode( get_option( 'wpvulnerability-php-vulnerable' ) );
	if ( ! is_numeric( $wpvulnerability_test_php_counter ) ) {
		$wpvulnerability_test_php_counter = 0;
	}
	$wpvulnerability_test_php_version = wpvulnerability_sanitize_version( phpversion() );
	/* translators: Show the number of vulnerabilities in a WP-Admin dashboard */
	$msg_php = sprintf( __( 'PHP %s: ', 'wpvulnerability' ), $wpvulnerability_test_php_version ) . sprintf( _n( '%d vulnerability', '%d vulnerabilities', $wpvulnerability_test_php_counter, 'wpvulnerability' ), $wpvulnerability_test_php_counter );

	// Show the widget.
	echo esc_html( __( 'Vulnerability analysis of your WordPress installation:', 'wpvulnerability' ) );
	echo '<ul>';

	if ( ! $wpvulnerability_test_core_counter ) {
		echo '<li>✔️ <span class="dashicons dashicons-wordpress"></span> ' . esc_html( $msg_core ) . '</li>';
	} else {
		echo '<li>❌ <span class="dashicons dashicons-wordpress"></span> ' . esc_html( $msg_core ) . '</li>';
	}

	if ( ! $wpvulnerability_test_plugins_counter ) {
		echo '<li>✔️ <span class="dashicons dashicons-admin-plugins"></span> ' . esc_html( $msg_plugins ) . '</li>';
	} else {
		echo '<li>❌ <span class="dashicons dashicons-admin-plugins"></span> ' . esc_html( $msg_plugins );
		echo wpvulnerability_list_plugins(); // phpcs:ignore
		echo '</li>';
	}

	if ( ! $wpvulnerability_test_themes_counter ) {
		echo '<li>✔️ <span class="dashicons dashicons-admin-appearance"></span> ' . esc_html( $msg_themes ) . '</li>';
	} else {
		echo '<li>❌ <span class="dashicons dashicons-admin-appearance"></span> ' . esc_html( $msg_themes );
		echo wpvulnerability_list_themes(); // phpcs:ignore
		echo '</li>';
	}

	if ( ! $wpvulnerability_test_php_counter ) {
		echo '<li>✔️ <span class="dashicons dashicons-editor-code"></span> ' . esc_html( $msg_php ) . '</li>';
	} else {
		echo '<li>❌ <span class="dashicons dashicons-editor-code"></span> ' . esc_html( $msg_php ) . '</li>';
	}

	echo '</ul>';
	echo esc_html( __( 'More information? Visit', 'wpvulnerability' ) ) . ' <a href="' . esc_url( get_admin_url( null, 'site-health.php' ) ) . '">' . esc_html( __( 'Site Health' ) ) . '</a>';

	unset( $wpvulnerability_test_core_counter, $wpvulnerability_test_themes_counter, $wpvulnerability_test_plugins_counter );
}

/**
 * Created a widget in the WordPress dashboard with vulnerability info.
 *
 * @since 2.2.0
 *
 * @return void
 */
function wpvulnerability_admin_dashboard() {
	wp_add_dashboard_widget( 'wpvulnerability', __( 'WPVulnerability Status', 'wpvulnerability' ), 'wpvulnerability_admin_dashboard_content', null, null, 'side', 'high' );
}
add_action( 'wp_network_dashboard_setup', 'wpvulnerability_admin_dashboard' );

/**
 * Initializes the WP-Admin settings page for the WP Vulnerability plugin
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_init() {

	// Register the plugin settings to be saved in the database.
	register_setting(
		'admin_wpvulnerability_settings',
		'wpvulnerability-config'
	);

	// Register messages.
	register_setting(
		'admin_wpvulnerability_settings',
		'wpvulnerability-messages'
	);

	// Add a section to the settings page.
	add_settings_section(
		'admin_wpvulnerability_settings',
		__( 'Receive notifications in your email', 'wpvulnerability' ),
		'wpvulnerability_admin_section_notifications',
		'wpvulnerability-config'
	);

	// Add a field to the settings page for the email addresses.
	add_settings_field(
		'wpvulnerability_emails',
		__( 'eMail addresses to notify (separated by commas)', 'wpvulnerability' ),
		'wpvulnerability_admin_emails_callback',
		'wpvulnerability-config',
		'admin_wpvulnerability_settings'
	);

	// Add a field to the settings page for the notification period.
	add_settings_field(
		'wpvulnerability_period',
		__( 'How often you want to receive notifications', 'wpvulnerability' ),
		'wpvulnerability_admin_period_callback',
		'wpvulnerability-config',
		'admin_wpvulnerability_settings'
	);
}
add_action( 'admin_init', 'wpvulnerability_admin_init' );
