<?php
/**
 * Themes functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Adds a vulnerability notice under vulnerable themes.
 *
 * @since 2.0.0
 *
 * @param  string $theme_file Main themes folder/file name.
 * @param  array  $theme_data Plugin data.
 *
 * @return void
 */
function wpvulnerability_theme_info_after( $theme_file, $theme_data ) {

	// Retrieve the vulnerabilities for all themes from the options table and decode the JSON.
	if( is_multisite() ) {
		$theme_vulnerabilities = json_decode( get_site_option( 'wpvulnerability-themes' ), true );
	} elseif ( ! is_multisite() ) {
		$theme_vulnerabilities = json_decode( get_option( 'wpvulnerability-themes' ), true );
	}

	// Determine whether the theme is active and add an appropriate CSS class to the table row.
	$current_theme = wp_get_theme();
	$tr_class = '';
	if ( $theme_file === $current_theme->get_stylesheet() ) {
		$tr_class .= 'active';
	}

	// Generate the vulnerability notice message with the theme name.
	$message = sprintf(
		/* translators: 1: theme name */
		__( '%1$s has a known vulnerability that may be affecting this version.', 'wpvulnerability' ),
		wp_kses( $theme_data->get( 'Name' ), 'strip' )
	);

	// Begin generating the table row HTML markup with appropriate CSS classes and the vulnerability notice message.
	$information  = '<tr class="wpvulnerability ' . $tr_class . '">';
	$information .= '<td colspan="4">';
	$information .= '<p class="text-red"><img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/logo16.png" style="height: 16px; vertical-align: text-top; width: 16px;" alt="" title="WPVulnerability"> <strong>' . $message . '</strong>';
	$information .= '</p>';
	$information .= '<table>';

	// Loop through all vulnerabilities for the current theme and add their details to the table row HTML markup.
	$vulnerabilities = $theme_vulnerabilities[$theme_file]['wpvulnerability']['vulnerabilities'];

	foreach ( $vulnerabilities as $vulnerability ) {

		$what = array();
		if ( isset( $vulnerability['impact']['cwe'] ) ) {
			foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
				$what[] = '<div><b>' . wp_kses( $vulnerability_cwe['name'], 'strip' ) . '</b></div><div><i>' . wp_kses_post( $vulnerability_cwe['description'] ) . '</i></div>';
			}
		}

		$sources = array();
		if ( isset( $vulnerability['source'] ) ) {
			foreach ( $vulnerability['source'] as $vulnerability_source ) {
				$sources[] = '<a href="' . esc_url_raw( $vulnerability_source['link'], 'strip' ) . '" target="_blank" rel="external nofollow noopener noreferrer">[+]</a>&nbsp;' . wp_kses( $vulnerability_source['name'], 'strip' );
			}
		}
		if ( count( $sources ) ) {
			$source = '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>';
		}

		$score = null;
		if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
			$score = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
		}
		$severity = null;
		if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
			$severity = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
		}
		$exploitable = null;
		if ( isset( $vulnerability['impact']['cvss']['exploitable'] ) ) {
			$exploitable = number_format( (float) $vulnerability['impact']['cvss']['exploitable'], 1, '.', '' );
		}

		$information .= '<tr>';
		$information .= '<td style="max-width: 256px; min-width: 96px;"><b>' . wp_kses( $vulnerability['versions'], 'strip' ) . '</b></td>';
		$information .= '<td>';
		if ( (int) $vulnerability['closed'] || (int) $vulnerability['unfixed'] ) {
			$information .= '<div style="padding-bottom: 5px;">';
			if ( (int) $vulnerability['closed'] ) {
				$information .= '<div class="text-red">' . __( 'This theme is closed. Please replace it with another.', 'wpvulnerability' ) . '</div>';
			}
			if ( (int) $vulnerability['unfixed'] ) {
				$information .= '<div class="text-red">' . __( 'This vulnerability appears to be unpatched. Stay tuned for upcoming theme updates.', 'wpvulnerability' ) . '</div>';
			}
			$information .= '</div>';
		}
		if ( count( $what ) ) {
			$information .= '<div style="padding-bottom: 5px;">';
			foreach ( $what as $w ) {
				$information .= $w;
			}
			$information .= '</div>';
		}
		if ( ! is_null( $score ) || ! is_null( $severity ) || ! is_null( $exploitable ) ) {
			$information .= '<div style="padding-bottom: 5px;">';
			if ( ! is_null( $score ) ) {
				$information .= '<div>' . __( 'Global score: ', 'wpvulnerability' ) . $score . ' / 10</div>';
			}
			if ( ! is_null( $severity ) ) {
				$information .= '<div>' . __( 'Severity: ', 'wpvulnerability' ) . $severity . '</div>';
			}
			if ( ! is_null( $exploitable ) ) {
				$information .= '<div>' . __( 'Exploitability: ', 'wpvulnerability' ) . $exploitable . ' / 10</div>';
			}
			$information .= '</div>';
		}
		$information .= wp_kses( $source, 'post' );
		$information .= '</td>';
		$information .= '</tr>';

	}

	$information .= '</table>';
	$information .= '</td>';
	$information .= '</tr>';

	echo $information; // phpcs:ignore
}

/**
 * Retrieves vulnerabilities for a given theme and updates its data.
 *
 * @since 2.0.0
 *
 * @param array  $theme_data The theme data array.
 * @param string $theme_slug The slug to the theme.
 *
 * @return array The updated theme data array.
 */
function get_fresh_theme_vulnerabilities( $theme_data, $theme_slug ) {

	// Get the theme version and slug from the theme data.
	$theme_version        = wp_kses( $theme_data['data']->get( 'Version' ), 'strip' );
	$theme_data_v['slug'] = $theme_slug;

	// Initialize vulnerability related fields.
	$theme_data_v['name']            = $theme_data['data']->get( 'Name' );
	$theme_data_v['vulnerabilities'] = null;
	$theme_data_v['vulnerable']      = 0;

	// Retrieve vulnerabilities for the theme using its slug and version.
	if ( $theme_slug ) {

		$theme_api_response = wpvulnerability_get_theme( $theme_slug, $theme_version );

		// If vulnerabilities are found, update the theme data accordingly.
		if ( ! empty( $theme_api_response ) ) {

			$theme_data_v['vulnerabilities'] = $theme_api_response;
			$theme_data_v['vulnerable']      = 1;

		}
	}

	return $theme_data_v;
}

/**
 * Get Installed Themes
 * Retrieves the list of installed themes, checks for vulnerabilities in each of them, caches the data, and sends an email notification if vulnerabilities are detected.
 *
 * @since 2.0.0
 *
 * @return string JSON-encoded array of plugin data with vulnerabilities and vulnerable status
 */
function wpvulnerability_theme_get_installed() {

	$wpvulnerability_themes_vulnerable = 0;

	$themes_v = array();
	$themes   = wp_get_themes();

	foreach ( $themes as $slug => $theme_data ) {

		$themes_v[ $slug ]['data']            = $theme_data;
		$themes_v[ $slug ]['wpvulnerability'] = get_fresh_theme_vulnerabilities( $themes_v[ $slug ], $slug );

		if ( isset( $themes_v[ $slug ]['wpvulnerability']['vulnerable'] ) && (int) $themes_v[ $slug ]['wpvulnerability']['vulnerable'] ) {

			++$wpvulnerability_themes_vulnerable;

		}
	}

	if( is_multisite() ) {

		update_site_option( 'wpvulnerability-themes', wp_json_encode( $themes_v ) );
		update_site_option( 'wpvulnerability-themes-vulnerable', wp_json_encode( number_format( $wpvulnerability_themes_vulnerable, 0, '.', '' ) ) );

	} elseif ( ! is_multisite() ) {

		update_option( 'wpvulnerability-themes', wp_json_encode( $themes_v ) );
		update_option( 'wpvulnerability-themes-vulnerable', wp_json_encode( number_format( $wpvulnerability_themes_vulnerable, 0, '.', '' ) ) );

	}

	return wp_json_encode( $themes_v );
}

/**
 * Get the cached themes vulnerabilities or update the cache if it's stale or missing.
 *
 * @since 2.0.0
 *
 * @return array Array of installed plugins with their vulnerabilities.
 */
function wpvulnerability_theme_get_vulnerabilities() {

	if( is_multisite() ) {

		// Get the cached theme data and decode it.
		$theme_data_cache = json_decode( get_site_option( 'wpvulnerability-themes-cache' ) );

		// Get the installed theme data and decode it.
		$theme_data = json_decode( get_site_option( 'wpvulnerability-themes' ), true );

	} elseif ( ! is_multisite() ) {

		// Get the cached theme data and decode it.
		$theme_data_cache = json_decode( get_option( 'wpvulnerability-themes-cache' ) );

		// Get the installed theme data and decode it.
		$theme_data = json_decode( get_option( 'wpvulnerability-themes' ), true );

	}

	// If the cache is stale or the theme data is empty, update the cache.
	if ( $theme_data_cache < time() || empty( $theme_data ) ) {

		// Get the installed theme data and update the cache.
		$theme_data = json_decode( wpvulnerability_theme_get_installed(), true );

		if( is_multisite() ) {

			update_site_option( 'wpvulnerability-themes-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

		} elseif ( ! is_multisite() ) {

			update_option( 'wpvulnerability-themes-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

		}

	}

	return $theme_data;
}

/**
 * Update the installed themes cache and remove any old cache data
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_theme_get_vulnerabilities_clean() {

	// Update the installed themes cache.
	wpvulnerability_theme_get_installed();

	if( is_multisite() ) {

		update_site_option( 'wpvulnerability-themes-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

	} elseif ( ! is_multisite() ) {

		update_option( 'wpvulnerability-themes-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

	}

}

/**
 * Admin Head
 * Adds vulnerability information after the theme row and notices on the theme page based on the installed theme cache
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_theme_page() {

	// Check if the current page is the plugins page.
	global $pagenow;
	if (
		'themes.php' === $pagenow &&
		(
			( is_multisite() && is_network_admin() && current_user_can( 'manage_network' ) ) ||
			( ! is_multisite() && is_admin() && current_user_can( 'manage_options' ) )
		)
	) {

		// Get the vulnerabilities for the installed themes.
		$themes = wpvulnerability_theme_get_vulnerabilities();

		// Loop through the plugins and add vulnerability information after the plugin row for vulnerable plugins.
		foreach ( $themes as $theme_file => $theme_data ) {

			if ( isset( $theme_data['wpvulnerability']['vulnerable'] ) && 1 === $theme_data['wpvulnerability']['vulnerable'] ) {

				add_action( 'after_theme_row_' . $theme_file, 'wpvulnerability_theme_info_after', 10, 3 );

			}
		}

	}
}
// Add notices for vulnerable plugins on the plugin page.
add_action( 'admin_head', 'wpvulnerability_theme_page' );
