<?php
/**
 * Process functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Convert vulnerabilities into pretty HTML
 *
 * @version 2.0.0
 *
 * @param string $type Type: core, plugin, theme.
 * @param array  $vulnerabilities Vulnerability data.
 *
 * @return string
 */
function wpvulnerability_html( $type, $vulnerabilities ) {

	$html = '';

	if ( 'plugin' === $type || 'theme' === $type ) {

		foreach ( $vulnerabilities as $vulnerability ) {

			$what = array();
			if ( isset( $vulnerability['impact']['cwe'] ) ) {
				foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
					$what[] = '<div><b>' . wp_kses( $vulnerability_cwe['name'], 'strip' ) . '</b></div><div><i>' . wp_kses_post( $vulnerability_cwe['description'] ) . '</i></div>';
				}
			}
			$sources = array();
			if ( isset( $vulnerability['source'] ) ) {
				foreach ( $vulnerability['source'] as $vulnerability_source ) {
					$sources[] = '<a href="' . esc_url_raw( $vulnerability_source['link'], 'strip' ) . '" target="_blank">[+]</a>&nbsp;' . wp_kses( $vulnerability_source['name'], 'strip' );
				}
			}
			if ( count( $sources ) ) {
				$source = '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>';
			}

			$score = null;
			if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
				$score = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
			}
			$severity = null;
			if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
				$severity = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
			}
			$exploitable = null;
			if ( isset( $vulnerability['impact']['cvss']['exploitable'] ) ) {
				$exploitable = number_format( (float) $vulnerability['impact']['cvss']['exploitable'], 1, '.', '' );
			}

			$html .= '<h4>' . wp_kses( $vulnerability['name'], 'strip' ) . '</h4>';
			if ( (int) $vulnerability['closed'] || (int) $vulnerability['unfixed'] ) {
				$html .= '<div style="padding-bottom: 5px;">';
				if ( (int) $vulnerability['closed'] ) {
					$html .= '<div class="text-red">' . __( 'This plugin is closed. Please replace it with another.', 'wpvulnerability' ) . '</div>';
				}
				if ( (int) $vulnerability['unfixed'] ) {
					$html .= '<div class="text-red">' . __( 'This vulnerability appears to be unpatched. Stay tuned for upcoming plugin updates.', 'wpvulnerability' ) . '</div>';
				}
				$html .= '</div>';
			}
			if ( count( $what ) ) {
				$html .= '<div style="padding-bottom: 5px;">';
				foreach ( $what as $w ) {
					$html .= $w;
				}
				$html .= '</div>';
			}
			if ( ! is_null( $score ) || ! is_null( $severity ) || ! is_null( $exploitable ) ) {
				$html .= '<div style="padding-bottom: 5px;">';
				if ( ! is_null( $score ) ) {
					$html .= '<div>' . __( 'Global score: ', 'wpvulnerability' ) . $score . ' / 10</div>';
				}
				if ( ! is_null( $severity ) ) {
					$html .= '<div>' . __( 'Severity: ', 'wpvulnerability' ) . $severity . '</div>';
				}
				if ( ! is_null( $exploitable ) ) {
					$html .= '<div>' . __( 'Exploitability: ', 'wpvulnerability' ) . $exploitable . ' / 10</div>';
				}
				$html .= '</div>';
			}
			$html .= wp_kses( $source, 'post' );

		}
	} elseif ( 'core' === $type ) {

		foreach ( $vulnerabilities as $vulnerability ) {

			$what = array();
			foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
				$what[] = '<div><b>' . wp_kses( $vulnerability_cwe['name'], 'strip' ) . '</b></div><div><i>' . wp_kses_post( $vulnerability_cwe['description'] ) . '</i></div>';
			}

			$sources = array();
			foreach ( $vulnerability['source'] as $vulnerability_source ) {
				$sources[] = '<a href="' . esc_url_raw( $vulnerability_source['link'], 'strip' ) . '" target="_blank">[+]</a>&nbsp;' . wp_kses( $vulnerability_source['name'], 'strip' );
			}
			$source = '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>';

			$score = null;
			if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
				$score = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
			}
			$severity = null;
			if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
				$severity = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
			}
			$exploitable = null;
			if ( isset( $vulnerability['impact']['cvss']['exploitable'] ) ) {
				$exploitable = number_format( (float) $vulnerability['impact']['cvss']['exploitable'], 1, '.', '' );
			}

			$html .= '<h3> WordPress ' . wp_kses( $vulnerability['name'], 'strip' ) . '</h3>';
			if ( count( $what ) ) {
				$html .= '<div style="padding-bottom: 5px;">';
				foreach ( $what as $w ) {
					$html .= $w;
				}
				$html .= '</div>';
			}
			if ( ! is_null( $score ) || ! is_null( $severity ) || ! is_null( $exploitable ) ) {
				$html .= '<div style="padding-bottom: 5px;">';
				if ( ! is_null( $score ) ) {
					$html .= '<div>' . __( 'Global score: ', 'wpvulnerability' ) . $score . ' / 10</div>';
				}
				if ( ! is_null( $severity ) ) {
					$html .= '<div>' . __( 'Severity: ', 'wpvulnerability' ) . $severity . '</div>';
				}
				if ( ! is_null( $exploitable ) ) {
					$html .= '<div>' . __( 'Exploitability: ', 'wpvulnerability' ) . $exploitable . ' / 10</div>';
				}
				$html .= '</div>';
			}
			$html .= wp_kses( $source, 'post' );

		}
		
	} elseif ( 'php' === $type ) {

		foreach ( $vulnerabilities as $vulnerability ) {

			$sources = array();
			foreach ( $vulnerability['source'] as $vulnerability_source ) {
				$sources[] = '<a href="' . esc_url_raw( $vulnerability_source['link'], 'strip' ) . '" target="_blank">[+]</a>&nbsp;' . wp_kses( $vulnerability_source['id'], 'strip' ) . '<br>' . wp_kses( $vulnerability_source['description'], 'strip' );
			}
			$source = '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>';

			$html .= '<h4> ' . wp_kses( $vulnerability['name'], 'strip' ) . '</h4>';
			$html .= '<div style="padding-bottom: 5px;"></div>';
			$html .= wp_kses( $source, 'post' );

		}

	}

	return $html;
}

/**
 * Convert plugin vulnerabilities into HTML format.
 *
 * @version 2.0.0
 *
 * @return string|false The HTML output if plugin vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_plugins() {

	$html  = '';
	$found = false;

	$plugins = wpvulnerability_plugin_get_vulnerabilities();

	foreach ( $plugins as $file_path => $plugin_data ) {

		if ( isset( $plugin_data['vulnerable'] ) && 1 === $plugin_data['vulnerable'] ) {

			$found = true;

			// Generate HTML markup for the plugin vulnerability.
			$html .= '<h3>' . esc_html__( 'Plugin', 'wpvulnerability' ) . ': ' . wp_kses( $plugin_data['Name'], 'strip' ) . '</h3>';
			$html .= wpvulnerability_html( 'plugin', $plugin_data['vulnerabilities'] );

		}
	}

	if ( $found ) {

		return $html;

	}
	// Return false if no vulnerabilities were found.
	return false;
}

/**
 * Convert PHP vulnerabilities into HTML format.
 *
 * @version 2.0.0
 *
 * @return string|false The HTML output if PHP vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_php() {

	$html  = '';
	$found = false;

	$php = wpvulnerability_php_get_vulnerabilities();

	if ( isset( $php['vulnerable'] ) && 1 === $php['vulnerable'] ) {

		$found = true;

		// Generate HTML markup for the plugin vulnerability.
		$html .= '<h3>' . esc_html__( 'PHP running', 'wpvulnerability' ) . ': ' . wp_kses( phpversion(), 'strip' ) . '</h3>';
		$html .= wpvulnerability_html( 'php', $php['vulnerabilities'] );

	}

	if ( $found ) {

		return $html;

	}
	// Return false if no vulnerabilities were found.
	return false;
}


/**
 * Convert plugin vulnerabilities into list format.
 *
 * @version 2.2.0
 *
 * @return string|false The HTML output if plugin vulnerabilities were found, false otherwise.
 */
function wpvulnerability_list_plugins() {

	$html  = '<ul class="inside">';
	$found = false;

	$plugins = wpvulnerability_plugin_get_vulnerabilities();

	foreach ( $plugins as $file_path => $plugin_data ) {

		if ( isset( $plugin_data['vulnerable'] ) && 1 === $plugin_data['vulnerable'] ) {

			$found = true;

			// Generate HTML markup for the plugin vulnerability.
			$html .= '<li>' . wp_kses( $plugin_data['Name'], 'strip' ) . '</li>';

		}
	}

	$html .= '</ul>';

	if ( $found ) {

		return $html;

	}
	// Return false if no vulnerabilities were found.
	return false;
}

/**
 * Convert theme vulnerabilities into HTML format.
 *
 * @version 2.0.0
 *
 * @return string|false The HTML output if plugin vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_themes() {

	$html  = '';
	$found = false;

	$themes = wpvulnerability_theme_get_vulnerabilities();

	foreach ( $themes as $theme_data ) {

		if ( isset( $theme_data['wpvulnerability']['vulnerable'] ) && 1 === $theme_data['wpvulnerability']['vulnerable'] ) {

				$found = true;

				// Generate HTML markup for the theme vulnerability.
				$html .= '<h3>' . esc_html__( 'Theme', 'wpvulnerability' ) . ': ' . wp_kses( $theme_data['wpvulnerability']['name'], 'strip' ) . '</h3>';
				$html .= wpvulnerability_html( 'theme', $theme_data['wpvulnerability']['vulnerabilities'] );

		}
	}

	if ( $found ) {

		return $html;

	}
	// Return false if no vulnerabilities were found.
	return false;
}

/**
 * Convert theme vulnerabilities into list format.
 *
 * @version 2.2.0
 *
 * @return string|false The HTML output if plugin vulnerabilities were found, false otherwise.
 */
function wpvulnerability_list_themes() {

	$html  = '<ul class="inside">';
	$found = false;

	$themes = wpvulnerability_theme_get_vulnerabilities();

	foreach ( $themes as $theme_data ) {

		if ( isset( $theme_data['wpvulnerability']['vulnerable'] ) && 1 === $theme_data['wpvulnerability']['vulnerable'] ) {

			$found = true;

			// Generate HTML markup for the theme vulnerability.
			$html .= '<li>' . wp_kses( $theme_data['wpvulnerability']['name'], 'strip' ) . '</li>';

		}
	}

	$html .= '</ul>';

	if ( $found ) {

		return $html;

	}
	// Return false if no vulnerabilities were found.
	return false;
}


/**
 * Convert theme vulnerabilities into HTML format.
 *
 * @version 2.0.0
 *
 * @return string|false The HTML output if theme vulnerabilities were found, false otherwise.
 */
function wpvulnerability_html_core() {

	$html  = '';
	$found = false;

	$core = wpvulnerability_core_get_vulnerabilities();

	if ( is_array( $core ) && count( $core ) ) {

		$found = true;

		// Generate HTML markup for the core vulnerability.
		$html .= wpvulnerability_html( 'core', $core );

	}

	if ( $found ) {

		return $html;

	}
	// Return false if no vulnerabilities were found.
	return false;
}
