<?php
/**
 * CLI functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Registers a WP-CLI command for WPVulnerability plugin.
 *
 * @since 2.0.0
 *
 * @return void
 */
if ( defined( 'WP_CLI' ) && WP_CLI ) {

	/**
	 * Core section in WP-CLI command
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 */
	function wpvulnerability_cli_core() {

		// Get core vulnerabilities.
		$core_vulnerabilities = wpvulnerability_core_get_vulnerabilities();

		$vulnerabilities = array();

		if ( $core_vulnerabilities && is_array( $core_vulnerabilities ) ) {

			// Loop through each core vulnerability.
			foreach ( $core_vulnerabilities as $vulnerability ) {

				$severity = null;
				if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
					$severity = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
				}
				$exploitable = null;
				if ( isset( $vulnerability['impact']['cvss']['exploitable'] ) ) {
					$exploitable = number_format( (float) $vulnerability['impact']['cvss']['exploitable'], 1, '.', '' );
				}

				// Add the vulnerability details to the array.
				array_push(
					$vulnerabilities,
					array(
						'Version'                   => trim( html_entity_decode( wp_kses( $vulnerability['name'], 'strip' ) ) ),
						'Vulnerability information' => '[*] WordPress ' . trim( html_entity_decode( wp_kses( $vulnerability['name'], 'strip' ) ) ),
					)
				);

				$what = array();
				if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {

					array_push(
						$vulnerabilities,
						array(
							'Version'                   => ' ',
							'Vulnerability information' => ' ',
						)
					);

					foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {

						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => trim( html_entity_decode( wp_kses( $vulnerability_cwe['name'], 'strip' ) ) ),
							)
						);

						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => trim( html_entity_decode( wp_kses( $vulnerability_cwe['description'], 'strip' ) ) ),
							)
						);

					}
				}

				$score = null;
				if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
					$score = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
				}
				$severity = null;
				if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
					$severity = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
				}
				$exploitable = null;
				if ( isset( $vulnerability['impact']['cvss']['exploitable'] ) ) {
					$exploitable = number_format( (float) $vulnerability['impact']['cvss']['exploitable'], 1, '.', '' );
				}

				if ( ! is_null( $score ) || ! is_null( $severity ) || ! is_null( $exploitable ) ) {

					array_push(
						$vulnerabilities,
						array(
							'Version'                   => ' ',
							'Vulnerability information' => ' ',
						)
					);

					if ( ! is_null( $score ) ) {
						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => __( 'Global score: ', 'wpvulnerability' ) . $score . ' / 10',
							)
						);
					}
					if ( ! is_null( $severity ) ) {
						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => __( 'Severity: ', 'wpvulnerability' ) . $severity,
							)
						);
					}
					if ( ! is_null( $exploitable ) ) {
						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => __( 'Exploitability: ', 'wpvulnerability' ) . $exploitable . ' / 10',
							)
						);

					}
				}

				if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {

					array_push(
						$vulnerabilities,
						array(
							'Version'                   => ' ',
							'Vulnerability information' => ' ',
						)
					);

					foreach ( $vulnerability['source'] as $vulnerability_source ) {

						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => '[+] ' . trim( html_entity_decode( wp_kses( $vulnerability_source['name'], 'strip' ) ) ),
							)
						);
						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => '    ' . esc_url_raw( $vulnerability_source['link'], 'strip' ),
							)
						);

					}
				}

				array_push(
					$vulnerabilities,
					array(
						'Version'                   => ' ',
						'Vulnerability information' => ' ',
					)
				);

			}
		}

		// Format and output the vulnerabilities in a table.
		WP_CLI\Utils\format_items(
			'table',
			$vulnerabilities,
			array( 'Version', 'Vulnerability information' )
		);
	}

	/**
	 * Plugin section in WP-CLI command
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 */
	function wpvulnerability_cli_plugins() {

		// Get plugin vulnerabilities.
		$plugin_vulnerabilities = wpvulnerability_plugin_get_vulnerabilities();

		// Loop through each plugin vulnerability.
		foreach ( $plugin_vulnerabilities as $plugin ) {

			if ( 1 === $plugin['vulnerable'] ) {

				$name = trim( html_entity_decode( wp_kses( $plugin['Name'], 'strip' ) ) );

				// Output the plugin name with red color.
				WP_CLI::line( WP_CLI::colorize( "%r$name%n " ) );

				// Prepare the vulnerabilities array for table format output.
				$vulnerabilities = array();
				foreach ( $plugin['vulnerabilities'] as $vulnerability ) {

					$severity = null;
					if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
						$severity = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
					}
					$exploitable = null;
					if ( isset( $vulnerability['impact']['cvss']['exploitable'] ) ) {
						$exploitable = number_format( (float) $vulnerability['impact']['cvss']['exploitable'], 1, '.', '' );
					}

					// Add the vulnerability details to the array.
					array_push(
						$vulnerabilities,
						array(
							'Version'                   => trim( html_entity_decode( wp_kses( $vulnerability['versions'], 'strip' ) ) ),
							'Vulnerability information' => '[*] ' . trim( html_entity_decode( wp_kses( $vulnerability['name'], 'strip' ) ) ),
						)
					);
					if ( (int) $vulnerability['closed'] || (int) $vulnerability['unfixed'] ) {
						if ( (int) $vulnerability['closed'] ) {

							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => '*** ' . __( 'This plugin is closed. Please replace it with another.', 'wpvulnerability' ) . ' ***',
								)
							);

						}
						if ( (int) $vulnerability['unfixed'] ) {

							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => '*** ' . __( 'This vulnerability appears to be unpatched. Stay tuned for upcoming plugin updates.', 'wpvulnerability' ) . ' ***',
								)
							);

						}
					}

					$what = array();
					if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {

						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => ' ',
							)
						);

						foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {

							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => trim( html_entity_decode( wp_kses( $vulnerability_cwe['name'], 'strip' ) ) ),
								)
							);

							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => trim( html_entity_decode( wp_kses( $vulnerability_cwe['description'], 'strip' ) ) ),
								)
							);

						}
					}

					$score = null;
					if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
						$score = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
					}
					$severity = null;
					if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
						$severity = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
					}
					$exploitable = null;
					if ( isset( $vulnerability['impact']['cvss']['exploitable'] ) ) {
						$exploitable = number_format( (float) $vulnerability['impact']['cvss']['exploitable'], 1, '.', '' );
					}

					if ( ! is_null( $score ) || ! is_null( $severity ) || ! is_null( $exploitable ) ) {

						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => ' ',
							)
						);

						if ( ! is_null( $score ) ) {
							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => __( 'Global score: ', 'wpvulnerability' ) . $score . ' / 10',
								)
							);
						}
						if ( ! is_null( $severity ) ) {
							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => __( 'Severity: ', 'wpvulnerability' ) . $severity,
								)
							);
						}
						if ( ! is_null( $exploitable ) ) {
							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => __( 'Exploitability: ', 'wpvulnerability' ) . $exploitable . ' / 10',
								)
							);

						}
					}

					if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {

						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => ' ',
							)
						);

						foreach ( $vulnerability['source'] as $vulnerability_source ) {

							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => '[+] ' . trim( html_entity_decode( wp_kses( $vulnerability_source['name'], 'strip' ) ) ),
								)
							);
							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => ' ' . esc_url_raw( $vulnerability_source['link'], 'strip' ),
								)
							);

						}
					}

					array_push(
						$vulnerabilities,
						array(
							'Version'                   => ' ',
							'Vulnerability information' => ' ',
						)
					);

				}

				// Format and output the vulnerabilities in a table.
				WP_CLI\Utils\format_items(
					'table',
					$vulnerabilities,
					array( 'Version', 'Vulnerability information' )
				);

			}
		}
	}

	/**
	 * Theme section in WP-CLI command
	 *
	 * @since 2.0.0
	 *
	 * @return void
	 */
	function wpvulnerability_cli_themes() {

		// Get theme vulnerabilities.
		$theme_vulnerabilities = wpvulnerability_theme_get_vulnerabilities();

		// Loop through each theme vulnerability.
		foreach ( $theme_vulnerabilities as $theme ) {

			if ( 1 === $theme['wpvulnerability']['vulnerable'] ) {

				$name = trim( html_entity_decode( wp_kses( $theme['wpvulnerability']['name'], 'strip' ) ) );

				// Output the theme name with red color.
				WP_CLI::line( WP_CLI::colorize( "%r$name%n " ) );

				// Prepare the vulnerabilities array for table format output.
				$vulnerabilities = array();
				foreach ( $theme['wpvulnerability']['vulnerabilities'] as $vulnerability ) {

					$severity = null;
					if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
						$severity = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
					}
					$exploitable = null;
					if ( isset( $vulnerability['impact']['cvss']['exploitable'] ) ) {
						$exploitable = number_format( (float) $vulnerability['impact']['cvss']['exploitable'], 1, '.', '' );
					}

					// Add the vulnerability details to the array.
					array_push(
						$vulnerabilities,
						array(
							'Version'                   => trim( html_entity_decode( wp_kses( $vulnerability['versions'], 'strip' ) ) ),
							'Vulnerability information' => '[*] ' . trim( html_entity_decode( wp_kses( $vulnerability['name'], 'strip' ) ) ),
						)
					);
					if ( (int) $vulnerability['closed'] || (int) $vulnerability['unfixed'] ) {
						if ( (int) $vulnerability['closed'] ) {

							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => '*** ' . __( 'This theme is closed. Please replace it with another.', 'wpvulnerability' ) . ' ***',
								)
							);

						}
						if ( (int) $vulnerability['unfixed'] ) {

							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => '*** ' . __( 'This vulnerability appears to be unpatched. Stay tuned for upcoming theme updates.', 'wpvulnerability' ) . ' ***',
								)
							);

						}
					}

					$what = array();
					if ( isset( $vulnerability['impact']['cwe'] ) && count( $vulnerability['impact']['cwe'] ) ) {

						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => ' ',
							)
						);

						foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {

							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => trim( html_entity_decode( wp_kses( $vulnerability_cwe['name'], 'strip' ) ) ),
								)
							);

							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => trim( html_entity_decode( wp_kses( $vulnerability_cwe['description'], 'strip' ) ) ),
								)
							);

						}
					}

					$score = null;
					if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
						$score = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
					}
					$severity = null;
					if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
						$severity = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
					}
					$exploitable = null;
					if ( isset( $vulnerability['impact']['cvss']['exploitable'] ) ) {
						$exploitable = number_format( (float) $vulnerability['impact']['cvss']['exploitable'], 1, '.', '' );
					}

					if ( ! is_null( $score ) || ! is_null( $severity ) || ! is_null( $exploitable ) ) {

						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => ' ',
							)
						);

						if ( ! is_null( $score ) ) {
							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => __( 'Global score: ', 'wpvulnerability' ) . $score . ' / 10',
								)
							);
						}
						if ( ! is_null( $severity ) ) {
							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => __( 'Severity: ', 'wpvulnerability' ) . $severity,
								)
							);
						}
						if ( ! is_null( $exploitable ) ) {
							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => __( 'Exploitability: ', 'wpvulnerability' ) . $exploitable . ' / 10',
								)
							);

						}
					}

					if ( isset( $vulnerability['source'] ) && count( $vulnerability['source'] ) ) {

						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => ' ',
							)
						);

						foreach ( $vulnerability['source'] as $vulnerability_source ) {

							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => '[+] ' . trim( html_entity_decode( wp_kses( $vulnerability_source['name'], 'strip' ) ) ),
								)
							);
							array_push(
								$vulnerabilities,
								array(
									'Version' => ' ',
									'Vulnerability information' => ' ' . esc_url_raw( $vulnerability_source['link'], 'strip' ),
								)
							);

						}
					}

					array_push(
						$vulnerabilities,
						array(
							'Version'                   => ' ',
							'Vulnerability information' => ' ',
						)
					);

				}

				// Format and output the vulnerabilities in a table.
				WP_CLI\Utils\format_items(
					'table',
					$vulnerabilities,
					array( 'Version', 'Vulnerability information' )
				);

			}
		}
	}

	/**
	 * PHP section in WP-CLI command
	 *
	 * @since 3.0.0
	 *
	 * @return void
	 */
	function wpvulnerability_cli_php() {

		// Get PHP vulnerabilities.
		$php_vulnerabilities = wpvulnerability_php_get_vulnerabilities();

		if ( isset( $php_vulnerabilities['vulnerabilities'] ) ) {

			$name = wp_kses( phpversion(), 'strip' );

			// Output the PHP name with red color.
			WP_CLI::line( WP_CLI::colorize( "%r$name%n " ) );

			// Prepare the vulnerabilities array for table format output.
			$vulnerabilities = array();

			// Loop through each PHP vulnerability.
			foreach ( $php_vulnerabilities['vulnerabilities'] as $php ) {

				// Add the vulnerability details to the array.
				array_push(
					$vulnerabilities,
					array(
						'Version'                   => trim( html_entity_decode( wp_kses( $php['versions'], 'strip' ) ) ),
						'Vulnerability information' => '[*] ' . trim( html_entity_decode( wp_kses( $php['name'], 'strip' ) ) ),
					)
				);
				if ( (int) $php['unfixed'] ) {

					array_push(
						$vulnerabilities,
						array(
							'Version' => ' ',
							'Vulnerability information' => '*** ' . __( 'This vulnerability appears to be unpatched. Stay tuned for upcoming theme updates.', 'wpvulnerability' ) . ' ***',
						)
					);

				}

				if ( isset( $php['source'] ) && count( $php['source'] ) ) {

					array_push(
						$vulnerabilities,
						array(
							'Version' => ' ',
							'Vulnerability information' => ' ',
						)
					);

					foreach ( $php['source'] as $vulnerability_source ) {

						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => '[+] ' . trim( html_entity_decode( wp_kses( $vulnerability_source['id'], 'strip' ) ) ),
							)
						);
						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => trim( html_entity_decode( wp_kses( $vulnerability_source['description'], 'strip' ) ) ),
							)
						);
						array_push(
							$vulnerabilities,
							array(
								'Version' => ' ',
								'Vulnerability information' => ' ' . esc_url_raw( $vulnerability_source['link'], 'strip' ),
							)
						);

					}

					array_push(
						$vulnerabilities,
						array(
							'Version'                   => ' ',
							'Vulnerability information' => ' ',
						)
					);

				}
			}

			// Format and output the vulnerabilities in a table.
			WP_CLI\Utils\format_items(
				'table',
				$vulnerabilities,
				array( 'Version', 'Vulnerability information' )
			);

		}
	}

	/**
	 * Switches the command to show the list of vulnerabilities detected in the site.
	 *
	 * @param array $args The subcommand to execute.
	 * @return void
	 */
	function wpvulnerability_cli_command( $args ) {

		// Selects the correct function to execute based on the subcommand.
		switch ( $args[0] ) {
			case 'core':
				wpvulnerability_cli_core();
				break;
			case 'plugins':
				wpvulnerability_cli_plugins();
				break;
			case 'themes':
				wpvulnerability_cli_themes();
				break;
			case 'php':
				wpvulnerability_cli_php();
				break;
			default:
				// Displays an error message for an invalid subcommand.
				WP_CLI::error( "'$args[0]' is not a registered subcommand of 'wpvulnerability'.\nAvailable subcommands: core, plugins, themes, php" );
				break;
		}
	}

	/**
	 * Adds a WP-CLI command to show the list of vulnerabilities detected in your site.
	 *
	 * EXAMPLES
	 *
	 *  - wp wpvulnerability core
	 *  - wp wpvulnerability plugins
	 *  - wp wpvulnerability themes
	 *
	 * @param object $args Arguments passed from the command line.
	 *
	 * @return void
	 */
	WP_CLI::add_command(
		'wpvulnerability',
		'wpvulnerability_cli_command',
		array(
			'shortdesc' => 'Show the list of vulnerabilities detected in your site.',
			'synopsis'  => array(
				array(
					'type'        => 'positional',
					'name'        => 'subcommand',
					'description' => 'subcommand [core|plugins|themes|php].',
					'optional'    => false,
				),
			),
			'when'      => 'after_wp_load',
			'longdesc'  => "EXAMPLES:\n\n - wp wpvulnerability core\n - wp wpvulnerability plugins\n - wp wpvulnerability themes\n - wp wpvulnerability php",
		)
	);

}
