<?php
/**
 * Process functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Load the settings to be available always.
 *
 * @version 2.0.0
 *
 * @return array|false An array containing the WPVulnerability settings if they exist, or false if they don't.
 */
$wpvulnerability_settings = get_option( 'wpvulnerability-config' );

/**
 * Enqueues the WPVulnerability admin CSS file on all admin pages.
 *
 * @version 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_enqueue_scripts() {

	wp_enqueue_style(
		'wpvulnerability-admin',
		WPVULNERABILITY_PLUGIN_URL . 'assets/admin.css',
		array(),
		WPVULNERABILITY_PLUGIN_VERSION
	);
}
add_action( 'admin_enqueue_scripts', 'wpvulnerability_admin_enqueue_scripts' );

/**
 * Create the WP-Admin options page
 * This function generates the HTML output for the WPVulnerability settings page in the WP-Admin.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_create_admin_page() {

	$wpvulnerability_settings = get_option( 'wpvulnerability-config' );
	?>
	<div class="header-wrap">
		<div class="wrapper">
			<div class="header wpvulnerability-header">
				<div class="logo">
					<img src="<?php echo esc_url( WPVULNERABILITY_PLUGIN_URL ); ?>assets/logo64.png" style="height: 64px; vertical-align: text-top; width: 64px;" alt="" title="WPVulnerability">
					<h2><?php esc_html_e( 'WPVulnerability settings', 'wpvulnerability' ); ?></h2>
				</div>
			</div>
		</div>
	</div>
	<div class="wrap">
		<?php settings_errors(); ?>
		<form method="post" action="options.php">
			<?php
				settings_fields( 'admin_wpvulnerability_settings' );
				do_settings_sections( 'wpvulnerability-config' );
				submit_button();
			?>
		</form>
	</div>
	<?php
}

/**
 * Adds a WP-Admin menu option for the WPVulnerability plugin
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_menu() {

	// Adds a submenu page under the Settings menu.
	add_submenu_page(
		'options-general.php',
		__( 'WPVulnerability', 'wpvulnerability' ),
		__( 'WPVulnerability', 'wpvulnerability' ),
		'manage_options',
		'wpvulnerability-options',
		'wpvulnerability_create_admin_page'
	);
}
add_action( 'admin_menu', 'wpvulnerability_admin_menu' );

/**
 * Print the settings header information for the notifications section.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_section_notifications() {

	// Output the header information for the notifications section.
	esc_html_e( 'Configure and save these settings to receive email notifications.', 'wpvulnerability' );
}

/**
 * Callback function to display the email input field in the admin settings page.
 * This function retrieves the current WPVulnerability plugin settings and displays the email input field
 * for users to enter their email addresses. If no email is saved in the settings, the admin email is displayed.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_emails_callback() {

	// Retrieve the WPVulnerability plugin settings.
	$wpvulnerability_settings = get_option( 'wpvulnerability-config' );

	// Set a default value for the email input field if no email is saved in the settings.
	if ( ! isset( $wpvulnerability_settings['emails'] ) ) {
		$wpvulnerability_settings['emails'] = '';
	}

	// Output the email input field and display the admin email as a hint.
	?>
	<input class="regular-text" type="text" name="wpvulnerability-config[emails]" id="wpvulnerability_emails" placeholder="<?php echo esc_attr( get_bloginfo( 'admin_email' ) ); ?>" value="<?php echo esc_attr( $wpvulnerability_settings['emails'] ); ?>">
	<br><small><?php echo esc_attr( get_bloginfo( 'admin_email' ) ); ?></small>
	<?php
}

/**
 * Print when to send the vulnerability scan emails.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_period_callback() {

	// Get the saved plugin settings.
	$wpvulnerability_settings = get_option( 'wpvulnerability-config' );

	// Set the default period to 'weekly' if not set or empty.
	if ( ! isset( $wpvulnerability_settings['period'] ) || empty( $wpvulnerability_settings['period'] ) ) {
		$wpvulnerability_settings['period'] = 'weekly';
	}

	// Output the period select box.
	?>
	<select name="wpvulnerability-config[period]" id="wpvulnerability_period">
		<option value="weekly" <?php selected( $wpvulnerability_settings['period'], 'weekly' ); ?>><?php esc_html_e( 'Weekly', 'wpvulnerability' ); ?></option>
		<option value="daily" <?php selected( $wpvulnerability_settings['period'], 'daily' ); ?>><?php esc_html_e( 'Daily', 'wpvulnerability' ); ?></option>
	</select>
	<?php
}

/**
 * Sanitize fields before saving into the database
 *
 * @since 2.0.0
 *
 * @param array $input The input fields to sanitize.
 *
 * @return array The sanitized values.
 */
function wpvulnerability_admin_sanitize( $input ) {

	$sanitized_values = array();
	$input_emails     = array();

	if ( isset( $input['emails'] ) ) {

		$input_email_text = explode( ',', $input['emails'] );

		// Loop through each email address in the input field.
		foreach ( $input_email_text as $input_email ) {

			// Sanitize each email address.
			$input_email = sanitize_email( trim( $input_email ) );

			if ( $input_email ) {
				$input_emails[] = $input_email;
			}
		}
	}

	if ( count( $input_emails ) ) {
		$sanitized_values['emails'] = implode( ',', $input_emails );
	} else {
		$sanitized_values['emails'] = null;
	}

	if ( isset( $input['period'] ) ) {

		// Check the value of the period field and sanitize it.
		switch ( $input['period'] ) {
			case 'daily':
				$sanitized_values['period'] = 'daily';
				break;
			case 'weekly':
			default:
				$sanitized_values['period'] = 'weekly';
				break;
		}
	}

	// Clear the scheduled hook for the notification event and set it to run at the sanitized interval.
	wp_clear_scheduled_hook( 'wpvulnerability_notification' );
	wp_schedule_event( time(), $sanitized_values['period'], 'wpvulnerability_notification' );

	return $sanitized_values;
}

/**
 * Initializes the WP-Admin settings page for the WP Vulnerability plugin
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_init() {

	// Register the plugin settings to be saved in the database.
	register_setting(
		'admin_wpvulnerability_settings',
		'wpvulnerability-config',
		'wpvulnerability_admin_sanitize'
	);

	// Add a section to the settings page.
	add_settings_section(
		'admin_wpvulnerability_settings',
		__( 'Receive notifications in your email', 'wpvulnerability' ),
		'wpvulnerability_admin_section_notifications',
		'wpvulnerability-config'
	);

	// Add a field to the settings page for the email addresses.
	add_settings_field(
		'wpvulnerability_emails',
		__( 'eMail addresses to notify (separated by commas)', 'wpvulnerability' ),
		'wpvulnerability_admin_emails_callback',
		'wpvulnerability-config',
		'admin_wpvulnerability_settings'
	);

	// Add a field to the settings page for the notification period.
	add_settings_field(
		'wpvulnerability_period',
		__( 'How often you want to receive notifications', 'wpvulnerability' ),
		'wpvulnerability_admin_period_callback',
		'wpvulnerability-config',
		'admin_wpvulnerability_settings'
	);
}
add_action( 'admin_init', 'wpvulnerability_admin_init' );
