<?php
/**
 * Site Health functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );


/**
 * Tests for vulnerabilities in installed plugins.
 *
 * @version 2.0.0
 *
 * @return array Returns an array with the results of the vulnerability test.
 */
function wpvulnerability_test_plugins() {

	// Define the initial test result values.
	$result = array(
		'label'       => __( 'There aren\'t plugins vulnerabilities', 'wpvulnerability' ),
		'status'      => 'good',
		'badge'       => array(
			'label' => __( 'Security', 'wpvulnerability' ),
			'color' => 'green',
		),
		'description' => sprintf(
			'<p>%s</p>',
			__( 'Shows possible vulnerabilities that exist in installed plugins.', 'wpvulnerability' )
		),
		'actions'     => '',
		'test'        => 'wpvulnerability_plugins',
	);

	// Check if any plugin vulnerabilities were found.
	$wpvulnerability_test_plugins_counter = json_decode( get_option( 'wpvulnerability-plugins-vulnerable' ) );
	if ( $wpvulnerability_test_plugins_counter ) {
		$result['status'] = 'critical';
		/* translators: Site Health message */
		$result['label']          = sprintf( _n( 'There is %d plugin with vulnerabilities', 'There are %d plugins with vulnerabilities', $wpvulnerability_test_plugins_counter, 'wpvulnerability' ), $wpvulnerability_test_plugins_counter );
		$result['badge']['color'] = 'red';
		$result['description']    = sprintf(
			'<p>%1$s</p> %2$s',
			__( 'We\'ve detected potential vulnerabilities in installed plugins. Please check them and keep them updated.', 'wpvulnerability' ),
			wpvulnerability_html_plugins()
		);
		$result['actions']       .= sprintf(
			'<p><a href="%s">%s</a></p>',
			esc_url( admin_url( 'plugins.php' ) ),
			__( 'Update plugins', 'wpvulnerability' )
		);
	}

	return $result;
}

/**
 * Tests for vulnerabilities in installed themes.
 *
 * @version 2.0.0
 *
 * @return array Returns an array with the results of the vulnerability test.
 */
function wpvulnerability_test_themes() {

	// Define the initial test result values.
	$result = array(
		'label'       => __( 'There aren\'t themes vulnerabilities', 'wpvulnerability' ),
		'status'      => 'good',
		'badge'       => array(
			'label' => __( 'Security', 'wpvulnerability' ),
			'color' => 'green',
		),
		'description' => sprintf(
			'<p>%s</p>',
			__( 'Shows possible vulnerabilities that exist in installed themes.', 'wpvulnerability' )
		),
		'actions'     => '',
		'test'        => 'wpvulnerability_themes',
	);

	// Check if any theme vulnerabilities were found.
	$wpvulnerability_test_themes_counter = json_decode( get_option( 'wpvulnerability-themes-vulnerable' ) );
	if ( $wpvulnerability_test_themes_counter ) {
		$result['status'] = 'critical';
		/* translators: Site Health message */
		$result['label']          = sprintf( _n( 'There is %d theme with vulnerabilities', 'There are %d themes with vulnerabilities', $wpvulnerability_test_themes_counter, 'wpvulnerability' ), $wpvulnerability_test_themes_counter );
		$result['badge']['color'] = 'red';
		$result['description']    = sprintf(
			'<p>%1$s</p> %2$s',
			__( 'We\'ve detected potential vulnerabilities in installed themes. Please check them and keep them updated.', 'wpvulnerability' ),
			wpvulnerability_html_themes()
		);
		$result['actions']       .= sprintf(
			'<p><a href="%s">%s</a></p>',
			esc_url( admin_url( 'themes.php' ) ),
			__( 'Update themes', 'wpvulnerability' )
		);
	}

	return $result;
}

/**
 * Tests for vulnerabilities in core.
 *
 * @version 2.0.0
 *
 * @return array Returns an array with the results of the vulnerability test.
 */
function wpvulnerability_test_core() {

	// Define the initial test result values.
	$result = array(
		'label'       => __( 'There aren\'t WordPress vulnerabilities', 'wpvulnerability' ),
		'status'      => 'good',
		'badge'       => array(
			'label' => __( 'Security', 'wpvulnerability' ),
			'color' => 'green',
		),
		'description' => sprintf(
			'<p>%s</p>',
			__( 'Shows possible vulnerabilities existing in the WordPress core.', 'wpvulnerability' )
		),
		'actions'     => '',
		'test'        => 'wpvulnerability_core',
	);

	// Check if any core vulnerabilities were found.
	$wpvulnerability_test_core_counter = json_decode( get_option( 'wpvulnerability-core-vulnerable' ) );
	if ( $wpvulnerability_test_core_counter ) {
		$result['status'] = 'critical';
		/* translators: Site Health message */
		$result['label']          = sprintf( _n( 'There is %d core vulnerability', 'There are %d core vulnerabilities', $wpvulnerability_test_core_counter, 'wpvulnerability' ), $wpvulnerability_test_core_counter );
		$result['badge']['color'] = 'red';
		$result['description']    = sprintf(
			'<p>%1$s</p> %2$s',
			__( 'We\'ve detected potential vulnerabilities in this WordPress installation. Please check them and keep your installation updated.', 'wpvulnerability' ),
			wpvulnerability_html_core()
		);
		$result['actions']       .= sprintf(
			'<p><a href="%s">%s</a></p>',
			esc_url( admin_url( 'update-core.php' ) ),
			__( 'Update WordPress', 'wpvulnerability' )
		);
	}

	return $result;
}

/**
 * Adds vulnerability tests to Health Check & Troubleshooting page.
 *
 * @since 2.0.0
 *
 * @param array $tests Array of current site status tests.
 *
 * @return array The updated array of site status tests.
 */
function wpvulnerability_tests( $tests ) {

	// Adds test for Core WordPress vulnerabilities.
	$tests['direct']['wpvulnerability_core'] = array(
		'label' => __( 'WPVulnerability Core', 'wpvulnerability' ),
		'test'  => 'wpvulnerability_test_core',
	);

	// Adds test for Theme vulnerabilities.
	$tests['direct']['wpvulnerability_themes'] = array(
		'label' => __( 'WPVulnerability Themes', 'wpvulnerability' ),
		'test'  => 'wpvulnerability_test_themes',
	);

	// Adds test for Plugin vulnerabilities.
	$tests['direct']['wpvulnerability_plugins'] = array(
		'label' => __( 'WPVulnerability Plugins', 'wpvulnerability' ),
		'test'  => 'wpvulnerability_test_plugins',
	);

	return $tests;
}
// Adds the vulnerability tests to the site status tests.
add_filter( 'site_status_tests', 'wpvulnerability_tests' );
