<?php
/**
 * Core functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Adds a vulnerability notice under vulnerable core.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_core_info_after() {

	// Retrieve the vulnerabilities for core from the options table and decode the JSON.
	$core_vulnerabilities = json_decode( get_option( 'wpvulnerability-core' ), true );

	// Generate the vulnerability notice message.
	$message = sprintf(
		/* translators: 1: core version */
		__( 'WordPress %1$s has a known vulnerability that may be affecting this version.', 'wpvulnerability' ),
		get_bloginfo( 'version' )
	);

	$information = '<p class="text-red"><img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/logo16.png" style="height: 16px; vertical-align: text-top; width: 16px;" alt="" title="WPVulnerability"> <strong>' . $message . '</strong></p>';

	$information .= '<table class="wp-list-table widefat wpvulnerability">';

	// Loop through all vulnerabilities for the current version and add their details to the table row HTML markup.
	foreach ( $core_vulnerabilities as $vulnerability ) {

		$what = array();
		if ( isset( $vulnerability['impact']['cwe'] ) ) {
			foreach ( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
				$what[] = '<div><b>' . wp_kses( $vulnerability_cwe['name'], 'strip' ) . '</b></div><div><i>' . wp_kses_post( $vulnerability_cwe['description'] ) . '</i></div>';
			}
		}

		$sources = array();
		if ( isset( $vulnerability['source'] ) ) {
			foreach ( $vulnerability['source'] as $vulnerability_source ) {
				$sources[] = '<a href="' . esc_url_raw( $vulnerability_source['link'], 'strip' ) . '" target="_blank" rel="external nofollow noopener noreferrer">[+]</a>&nbsp;' . wp_kses( $vulnerability_source['name'], 'strip' );
			}
		}
		if ( count( $sources ) ) {
			$source = '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>';
		}

		$score = null;
		if ( isset( $vulnerability['impact']['cvss']['score'] ) ) {
			$score = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
		}
		$severity = null;
		if ( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
			$severity = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
		}
		$exploitable = null;
		if ( isset( $vulnerability['impact']['cvss']['exploitable'] ) ) {
			$exploitable = number_format( (float) $vulnerability['impact']['cvss']['exploitable'], 1, '.', '' );
		}

		$information .= '<tr>';
		$information .= '<td style="max-width: 256px; min-width: 96px;">WordPress <b>' . wp_kses( $vulnerability['name'], 'strip' ) . '</b></td>';
		$information .= '<td>';
		if ( count( $what ) ) {
			$information .= '<div style="padding-bottom: 5px;">';
			foreach ( $what as $w ) {
				$information .= $w;
			}
			$information .= '</div>';
		}
		if ( ! is_null( $score ) || ! is_null( $severity ) || ! is_null( $exploitable ) ) {
			$information .= '<div style="padding-bottom: 5px;">';
			if ( ! is_null( $score ) ) {
				$information .= '<div>' . __( 'Global score: ', 'wpvulnerability' ) . $score . ' / 10</div>';
			}
			if ( ! is_null( $severity ) ) {
				$information .= '<div>' . __( 'Severity: ', 'wpvulnerability' ) . $severity . '</div>';
			}
			if ( ! is_null( $exploitable ) ) {
				$information .= '<div>' . __( 'Exploitability: ', 'wpvulnerability' ) . $exploitable . ' / 10</div>';
			}
			$information .= '</div>';
		}
		$information .= wp_kses( $source, 'post' );
		$information .= '</td>';
		$information .= '</tr>';

	}

	$information .= '</table>';

	echo $information; // phpcs:ignore
}

/**
 * Retrieves vulnerabilities for a given version and updates its data.
 *
 * @since 2.0.0
 *
 * @return array The updated theme data array.
 */
function get_fresh_core_vulnerabilities() {

	// Get the core version from the data.
	$version = wpvulnerability_sanitize_version( get_bloginfo( 'version' ) );

	// Retrieve vulnerabilities for the theme using its slug and version.
	$response = wpvulnerability_get( 'core', $version );

	$core_data = array();

	if ( empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// If vulnerabilities are found, update the data accordingly.
	foreach ( $response['data']['vulnerability'] as $v ) {

		$core_data[] = array(
			'name'   => wp_kses( $v['name'], 'strip' ),
			'source' => $v['source'],
			'impact' => $v['impact'],
		);

	}
	return $core_data;
}

/**
 * Get Vulnerabilities
 *
 * @since 2.0.0
 *
 * @return string JSON-encoded array of core data with vulnerabilities and vulnerable status
 */
function wpvulnerability_core_get_installed() {

	$wpvulnerability_core_vulnerable = 0;

	$core = get_fresh_core_vulnerabilities();

	if ( is_array( $core ) && count( $core ) ) {

		$wpvulnerability_core_vulnerable = count( $core );

	}

	update_option( 'wpvulnerability-core', wp_json_encode( $core ) );
	update_option( 'wpvulnerability-core-vulnerable', wp_json_encode( number_format( $wpvulnerability_core_vulnerable, 0, '.', '' ) ) );

	return wp_json_encode( $core );
}

/**
 * Get the cached vulnerabilities or update the cache if it's stale or missing.
 *
 * @since 2.0.0
 *
 * @return array Array of core with their vulnerabilities.
 */
function wpvulnerability_core_get_vulnerabilities() {

	// Get the cached core data and decode it.
	$core_data_cache = json_decode( get_option( 'wpvulnerability-core-cache' ) );

	// Get the core data and decode it.
	$core_data = json_decode( get_option( 'wpvulnerability-core' ), true );

	// If the cache is stale or the theme data is empty, update the cache.
	if ( $core_data_cache < time() || empty( $core_data ) ) {

		// Get the core data and update the cache.
		$core_data = json_decode( wpvulnerability_core_get_installed(), true );
		update_option( 'wpvulnerability-core-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

	}

	return $core_data;
}

/**
 * Update the core cache and remove any old cache data
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_core_get_vulnerabilities_clean() {

	// Update the core cache.
	wpvulnerability_core_get_installed();
	update_option( 'wpvulnerability-core-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );
}

/**
 * Admin Head
 * Adds vulnerability information after the core and notices
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_core_page() {

	// Check if the current page is the update core page.
	global $pagenow;
	if ( 'update-core.php' === $pagenow ) {

		// Get the vulnerabilities for the core.
		$core = wpvulnerability_core_get_vulnerabilities();

		// Loop through and add vulnerability information after the core version for vulnerabilities.
		if ( is_array( $core ) && count( $core ) ) {

			add_action( 'after_core_auto_updates_settings', 'wpvulnerability_core_info_after' );

		}
	}
}
// Add notices for vulnerable core on the core page.
add_action( 'admin_head', 'wpvulnerability_core_page' );
