<?php
/**
 * Running functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */
defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Adds a settings link to the plugin row in the plugins list.
 *
 * @param array $links The links that appear in the plugin row.
 *
 * @return array The modified array of links.
 *
 * @version 2.0.0
 */
function wpvulnerability_add_settings_link( $links ) {

	$links[] = '<a href="' . get_admin_url( null, 'options-general.php?page=wpvulnerability-options' ) . '">' . __( 'Settings', 'wpvulnerability' ) . '</a>';
	return $links;

}
add_filter( 'plugin_action_links_' . WPVULNERABILITY_PLUGIN_BASE, 'wpvulnerability_add_settings_link' );

/**
 * Updates the plugin's data on scheduled wp-cron job.
 * 
 * @since 2.0.0
 * 
 * @return void
 */
function wpvulnerability_update_database_data() {

	// Update core vulnerabilities.
	wpvulnerability_core_get_vulnerabilities_clean();

	// Update plugin vulnerabilities.
	wpvulnerability_plugin_get_vulnerabilities_clean();

	// Update theme vulnerabilities.
	wpvulnerability_theme_get_vulnerabilities_clean();

}

/**
 * Callback function for when the plugin is activated.
 * Adds plugin data options if they are not already created.
 * 
 * @since 2.0.0
 * 
 * @return void
 */
function wpvulnerability_activation() {

	// Add wpvulnerability-config option if it does not exist.
	if ( ! get_option( 'wpvulnerability-config' ) ) {
		add_option( 'wpvulnerability-config', array(
			'emails' => get_bloginfo( 'admin_email' ),
			'period' => 'weekly'
		) );
	}

	// Add wpvulnerability-plugins option if it does not exist.
	if ( ! get_option( 'wpvulnerability-plugins' ) ) {
		add_option( 'wpvulnerability-plugins', '' );
	}
	
	// Add wpvulnerability-plugins-cache option if it does not exist.
	if ( ! get_option( 'wpvulnerability-plugins-cache' ) ) {
		add_option( 'wpvulnerability-plugins-cache', time() );
	}

	// Add wpvulnerability-plugins-vulnerable option if it does not exist.
	if ( ! get_option( 'wpvulnerability-plugins-vulnerable' ) ) {
		add_option( 'wpvulnerability-plugins-vulnerable', 0 );
	}

	// Add wpvulnerability-themes option if it does not exist.
	if ( ! get_option( 'wpvulnerability-themes' ) ) {
		add_option( 'wpvulnerability-themes', '' );
	}
	
	// Add wpvulnerability-themes-cache option if it does not exist.
	if ( ! get_option( 'wpvulnerability-themes-cache' ) ) {
		add_option( 'wpvulnerability-themes-cache', time() );
	}

	// Add wpvulnerability-themes-vulnerable option if it does not exist.
	if ( ! get_option( 'wpvulnerability-themes-vulnerable' ) ) {
		add_option( 'wpvulnerability-themes-vulnerable', 0 );
	}

	// Add wpvulnerability-core option if it does not exist.
	if ( ! get_option( 'wpvulnerability-core' ) ) {
		add_option( 'wpvulnerability-core', '' );
	}
	
	// Add wpvulnerability-core-cache option if it does not exist.
	if ( ! get_option( 'wpvulnerability-core-cache' ) ) {
		add_option( 'wpvulnerability-core-cache', time() );
	}

	// Add wpvulnerability-core-vulnerable option if it does not exist.
	if ( ! get_option( 'wpvulnerability-core-vulnerable' ) ) {
		add_option( 'wpvulnerability-core-vulnerable', 0 );
	}

	wpvulnerability_update_database_data();

}
register_activation_hook( WPVULNERABILITY_PLUGIN_FILE, 'wpvulnerability_activation' );

/**
 * On Deactivation
 * Callback function to run when the plugin is deactivated
 * Deletes options and removes scheduled wp-cron jobs.
 *
 * @since 2.0.0
 * 
 * @return void
 */
function wpvulnerability_deactivation() {

	// Delete all plugin options.
	delete_option( 'wpvulnerability_settings' );
	delete_option( 'wpvulnerability-data' );

	delete_option( 'wpvulnerability-themes' );
	delete_option( 'wpvulnerability-themes-cache' );
	delete_option( 'wpvulnerability-themes-vulnerable' );
	delete_option( 'wpvulnerability-plugins' );
	delete_option( 'wpvulnerability-plugins-cache' );
	delete_option( 'wpvulnerability-plugins-vulnerable' );
	delete_option( 'wpvulnerability-core' );
	delete_option( 'wpvulnerability-core-cache' );
	delete_option( 'wpvulnerability-core-vulnerable' );

	// Unschedule and remove scheduled wp-cron jobs.
	wp_unschedule_event( wp_next_scheduled( 'wpvulnerability_notification' ), 'wpvulnerability_notification' );
	wp_clear_scheduled_hook( 'wpvulnerability_notification' );

	wp_unschedule_event( wp_next_scheduled( 'wpvulnerability_update_database' ), 'wpvulnerability_update_database' );
	wp_clear_scheduled_hook( 'wpvulnerability_update_database' );

}
register_deactivation_hook( WPVULNERABILITY_PLUGIN_FILE, 'wpvulnerability_deactivation' );

/**
 * Schedule Automatic Vulnerability Database Update
 * If the 'wpvulnerability_update_database' event is not already scheduled, schedule it to run twice daily.
 * 
 * @since 2.0.0
 * 
 * @return void
 */
if ( ! wp_next_scheduled( 'wpvulnerability_update_database' ) ) {
	wp_schedule_event( time(), 'twicedaily', 'wpvulnerability_update_database' );
}
add_action( 'wpvulnerability_update_database', 'wpvulnerability_update_database_data' );

/**
 * Clean the cache after an update
 *
 * @since 2.0.0
 * 
 * @return void
 */
add_action( 'upgrader_process_complete', 'wpvulnerability_update_database_data' );
