<?php
/**
 * Plugin functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */
defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Adds a vulnerability notice under vulnerable plugins.
 *
 * @since 2.0.0
 *
 * @param  string $plugin_file Main plugin folder/file name.
 * @param  array  $plugin_data Plugin data.
 * @param  array  $status Status.
 *
 * @return string $information HTML.
 */
function wpvulnerability_plugin_info_after( $plugin_file, $plugin_data, $status ) {

	// Retrieve the vulnerabilities for all plugins from the options table and decode the JSON.
	$plugin_vulnerabilities = json_decode( get_option( 'wpvulnerability-plugins' ), true );

	// Determine whether the plugin is active and add an appropriate CSS class to the table row.
	$tr_class = '';
	if ( is_plugin_active( $plugin_file ) ) {
		$tr_class .= 'active';
	}

	// Generate the vulnerability notice message with the plugin name.
	$message = sprintf(
		/* translators: 1: Plugin name */
		__( '%1$s has a known vulnerability that may be affecting this version.', 'wpvulnerability' ),
		wp_kses( $plugin_data['Name'], 'strip' )
	);

	// Begin generating the table row HTML markup with appropriate CSS classes and the vulnerability notice message.
	$information  = '<tr class="wpvulnerability ' . $tr_class . '">';
	$information .= '<td colspan="4">';
	$information .= '<p class="text-red"><img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/logo16.png" style="height: 16px; vertical-align: text-top; width: 16px;" alt="" title="WPVulnerability"> <strong>' . $message . '</strong>';
	$information .= '</p>';
	$information .= '<table>';

	// Loop through all vulnerabilities for the current plugin and add their details to the table row HTML markup.
	$vulnerabilities = $plugin_vulnerabilities[ $plugin_file ]['vulnerabilities'];

	foreach ( $vulnerabilities as $vulnerability ) {

		$what = array();
		if( isset( $vulnerability['impact']['cwe'] ) ) {
			foreach( $vulnerability['impact']['cwe'] as $vulnerability_cwe ) {
				$what[] = '<div><b>' . wp_kses( $vulnerability_cwe['name'], 'strip' ) . '</b></div><div><i>' . wp_kses_post( $vulnerability_cwe['description'] ) . '</i></div>';
			}
		}

		$sources = array();
		if( isset( $vulnerability['source'] ) ) {
			foreach( $vulnerability['source'] as $vulnerability_source ) {
				$sources[] = '<a href="' . esc_url_raw( $vulnerability_source['link'], 'strip') . '" target="_blank" rel="external nofollow noopener noreferrer">[+]</a>&nbsp;' . wp_kses( $vulnerability_source['name'], 'strip' );
			}
		}
		if( count( $sources ) ) {
			$source = '<div style="padding-bottom: 5px;">' . implode( '<br>', $sources ) . '</div>';
		}

		$score = null;
		if( isset( $vulnerability['impact']['cvss']['score'] ) ) {
			$score = number_format( (float) $vulnerability['impact']['cvss']['score'], 1, '.', '' );
		}
		$severity = null;
		if( isset( $vulnerability['impact']['cvss']['severity'] ) ) {
			$severity = wpvulnerability_severity( $vulnerability['impact']['cvss']['severity'] );
		}
		$exploitable = null;
		if( isset( $vulnerability['impact']['cvss']['exploitable'] ) ) {
			$exploitable = number_format( (float) $vulnerability['impact']['cvss']['exploitable'], 1, '.', '' );
		}

		$information .= '<tr>';
		$information .= '<td style="max-width: 256px; min-width: 96px;"><b>' . wp_kses( $vulnerability['versions'], 'strip' ) . '</b></td>';
		$information .= '<td>';
		if( (int)$vulnerability['closed'] || (int)$vulnerability['unfixed'] ) {
			$information .= '<div style="padding-bottom: 5px;">';
			if( (int)$vulnerability['closed'] ) {
				$information .= '<div class="text-red">' . __( 'This plugin is closed. Please replace it with another.', 'wpvulnerability' ) . '</div>';
			}
			if( (int)$vulnerability['unfixed'] ) {
				$information .= '<div class="text-red">' . __( 'This vulnerability appears to be unpatched. Stay tuned for upcoming plugin updates.', 'wpvulnerability' ) . '</div>';
			}
			$information .= '</div>';
		}
		if( count( $what ) ) {
			$information .= '<div style="padding-bottom: 5px;">';
			foreach( $what as $w ) {
				$information .= $w;
			}
			$information .= '</div>';
		}
		if( !is_null( $score ) || !is_null( $severity ) || !is_null( $exploitable ) ) {
			$information .= '<div style="padding-bottom: 5px;">';
			if( !is_null( $score ) ) {
				$information .= '<div>' . __( 'Global score: ', 'wpvulnerability' ) . $score . ' / 10</div>';
			}
			if( !is_null( $severity ) ) {
				$information .= '<div>' . __( 'Severity: ', 'wpvulnerability' ) . $severity . '</div>';
			}
			if( !is_null( $exploitable ) ) {
				$information .= '<div>' . __( 'Exploitability: ', 'wpvulnerability' ) . $exploitable . ' / 10</div>';
			}
			$information .= '</div>';
		}
		$information .= wp_kses( $source, 'post' );
		$information .= '</td>';
		$information .= '</tr>';

	}

	$information .= '</table>';
	$information .= '</td>';
	$information .= '</tr>';

	echo $information; // phpcs:ignore
}

/**
 * Retrieves vulnerabilities for a given plugin and updates its data.
 *
 * @since 2.0.0
 * 
 * @param array $plugin_data The plugin data array.
 * @param string $file_path The path to the plugin file.
 * 
 * @return array The updated plugin data array.
 * 
 */
function get_fresh_plugin_vulnerabilities( $plugin_data, $file_path ) {

  // If the TextDomain key is empty, extract it from the file path.
	if( empty( $plugin_data['TextDomain'] ) ) {
		
		$folder_name = explode( '/', $file_path );
		
		if( isset( $folder_name[0] ) ) {
			$plugin_data['TextDomain'] = wp_kses( $folder_name[0], 'strip' );
		}

	}

	// Get the plugin slug and version from the plugin data.
	$plugin_slug = wp_kses( $plugin_data['TextDomain'], 'strip' );
	$plugin_version = wp_kses( $plugin_data['Version'], 'strip' );

	// Initialize vulnerability related fields.
	$plugin_data['vulnerabilities'] = null;
	$plugin_data['vulnerable'] = 0;

	// Retrieve vulnerabilities for the plugin using its slug and version.
	if( $plugin_slug ) {

		$plugin_api_response = wpvulnerability_get_plugin( $plugin_slug, $plugin_version );

		// If vulnerabilities are found, update the plugin data accordingly.
		if ( !empty( $plugin_api_response ) ) {

			$plugin_data['vulnerabilities'] = $plugin_api_response;
			$plugin_data['vulnerable'] = 1;

		}

	}

	return $plugin_data;
}

/**
 * Get Installed Plugins
 * Retrieves the list of installed plugins, checks for vulnerabilities in each of them, caches the data, and sends an email notification if vulnerabilities are detected.
 *
 * @since 2.0.0
 *
 * @return string JSON-encoded array of plugin data with vulnerabilities and vulnerable status
 */
function wpvulnerability_plugin_get_installed() {

	if ( ! function_exists( 'get_plugins' ) ) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}

	$plugins = get_plugins();

	foreach ( $plugins as $file_path => $plugin_data ) {

		$plugins[$file_path] = get_fresh_plugin_vulnerabilities( $plugin_data, $file_path );

	}

	update_option( 'wpvulnerability-plugins', wp_json_encode( $plugins ) );

	return wp_json_encode( $plugins );
}

/**
 * Get the cached plugin vulnerabilities or update the cache if it's stale or missing.
 *
 * @since 2.0.0
 *
 * @return array Array of installed plugins with their vulnerabilities.
 */
function wpvulnerability_plugin_get_vulnerabilities( ) {

	// Get the cached plugin data and decode it.
	$plugin_data_cache = json_decode( get_option( 'wpvulnerability-plugins-cache' ) );

	// Get the installed plugin data and decode it.
	$plugin_data = json_decode( get_option( 'wpvulnerability-plugins' ), true );

	// If the cache is stale or the plugin data is empty, update the cache.
	if( $plugin_data_cache < time() || empty( $plugin_data ) ) {

		// Get the installed plugin data and update the cache.
		$plugin_data = json_decode( wpvulnerability_plugin_get_installed( ), true );
		update_option( 'wpvulnerability-plugins-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );

	}

	return $plugin_data;

}

/**
 * Update the installed plugins cache and remove any old cache data
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_plugin_get_vulnerabilities_clean( ) {

	// Update the installed plugins cache
	wpvulnerability_plugin_get_installed( );

}

/**
 * Admin Head
 * Adds vulnerability information after the plugin row and notices on the plugin page based on the installed plugins cache
 * 
 * @since 2.0.0
 * 
 * @return void
 */
function wpvulnerability_plugin_page() {
	
	// Check if the current page is the plugins page
	global $pagenow;
	if( 'plugins.php' == $pagenow ) {

		// Get the vulnerabilities for the installed plugins
		$plugins = wpvulnerability_plugin_get_vulnerabilities();

		// Loop through the plugins and add vulnerability information after the plugin row for vulnerable plugins
		foreach ( $plugins as $file_path => $plugin_data ) {

			if ( isset( $plugin_data['vulnerable'] ) && 1 === $plugin_data['vulnerable'] ) {

				add_action( 'after_plugin_row_' . $file_path, 'wpvulnerability_plugin_info_after', 10, 3 );

			}

		}

	}
}
// Add notices for vulnerable plugins on the plugin page
add_action( 'admin_head', 'wpvulnerability_plugin_page' );
