<?php
/**
 * General functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */
defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Sanitize a version string.
 *
 * @version 2.0.0
 *
 * @param string $version The version string to sanitize.
 *
 * @return string The sanitized version string.
 */
function wpvulnerability_sanitize_version( $version ) {

	// Remove any leading/trailing whitespace
	// Strip out any non-alphanumeric characters except for hyphens, underscores, and dots
	$version = trim( preg_replace( '/[^a-zA-Z0-9_\-.]+/', '', $version ) );

	return $version;

}

/**
 * Returns a human-readable HTML entity for the given comparison operator.
 *
 * @version 2.0.0
 *
 * @param string $op The operator string to prettify.
 *
 * @return string The pretty operator HTML string.
 */
function wpvulnerability_pretty_operator( $op ) {

  switch( trim( strtolower( $op ) ) ) {
		// Less than
    case 'lt':
      return '&lt;&nbsp;';
      break;
		// Less than or equal to
    case 'le':
      return '&le;&nbsp;';
      break;
		// Greater than
    case 'gt':
      return '&gt;&nbsp;';
      break;
		// Greater than or equal to
    case 'ge':
      return '&ge;&nbsp;';
      break;
		// Equal to
    case 'eq':
      return '&equals;&nbsp;';
      break;
		// Not equal to
    case 'ne':
      return '&ne;&nbsp;';
      break;
		// Return the original operator if it's not recognized
    default:
      return $op;
      break;
  }

}

/**
 * Returns a human-readable HTML entity for the given comparison operator.
 *
 * @version 2.0.0
 *
 * @param string $op The operator string to prettify.
 *
 * @return string The pretty operator HTML string.
 */
function wpvulnerability_pretty_operator_cli( $op ) {

  switch( trim( strtolower( $op ) ) ) {
		// Less than
    case 'lt':
      return '< ';
      break;
		// Less than or equal to
    case 'le':
      return '<= ';
      break;
		// Greater than
    case 'gt':
      return '> ';
      break;
		// Greater than or equal to
    case 'ge':
      return '>= ';
      break;
		// Equal to
    case 'eq':
      return '= ';
      break;
		// Not equal to
    case 'ne':
      return '!= ';
      break;
		// Return the original operator if it's not recognized
    default:
      return $op;
      break;
  }

}


/**
 * Returns a Severity
 *
 * @version 2.0.0
 *
 * @param string $severity The severity string to prettify.
 *
 * @return string The severity string.
 */
function wpvulnerability_severity( $severity ) {

  switch( trim( strtolower( $severity ) ) ) {
    case 'n':
			/* translators: Severity: None */
      return __( 'None', 'wpvulnerability' );
      break;
    case 'l':
			/* translators: Severity: Low */
      return __( 'Low', 'wpvulnerability' );
      break;
    case 'm':
			/* translators: Severity: Medium */
      return __( 'Medium', 'wpvulnerability' );
      break;
    case 'm':
			/* translators: Severity: High */
      return __( 'High', 'wpvulnerability' );
      break;
    case 'c':
			/* translators: Severity: Critical */
      return __( 'Critical', 'wpvulnerability' );
      break;
		// Return the original severity if it's not recognized
    default:
      return $severity;
      break;
  }

}

/**
 * Retrieves vulnerabilities information from the API.
 *
 * @version 2.0.0
 *
 * @param string $type The type of vulnerability. Can be 'core', 'plugin' or 'theme'.
 * @param string $slug The slug of the plugin or theme. For core vulnerabilities, it is the version string.
 *
 * @return array|bool An array with the vulnerability information or false if there's an error.
 */
function wpvulnerability_get( $type, $slug = '' ) {

	$args = array(
		'timeout'   => 10000,
		'sslverify' => false,
	);

	// Validate vulnerability type.
	switch( trim( strtolower( $type ) ) ) {
		case 'core':
			$type = 'core';
			break;
		case 'plugin':
			$type = 'plugin';
			break;
		case 'theme':
			$type = 'theme';
			break;
		default: 
			wp_die( 'Unknown vulnerability type sent.' );
			break;
	}

	// Validate slug.
	if( 'plugin' == $type || 'theme' == $type ) {

		if( empty( sanitize_title( $slug ) ) ) {
			return false;
		}

	} elseif( $type == 'core' ) {

		if( !wpvulnerability_sanitize_version( $slug ) ) {
			return false;
		}

	}

	// Cache key.
	$key = 'wpvulnerability_' . $type . '_' . $slug;
	$vulnerability = get_transient( $key );

	// If not cached, get the updated data
	if ( empty( $vulnerability ) ) {

		$url = WPVULNERABILITY_API_HOST . $type . '/' . $slug . '/';
		$response = wp_remote_get( $url, $args );

		if ( !is_wp_error( $response ) ) {

			$body = wp_remote_retrieve_body( $response );
			set_transient( $key, $body, HOUR_IN_SECONDS * WPVULNERABILITY_CACHE_HOURS );

		}

	}

	return json_decode( $vulnerability, true );

}

/**
 * Retrieve vulnerabilities for a specific version of WordPress Core.
 *
 * @since 2.0.0
 *
 * @param string|null $version The version number of WordPress Core. If null, retrieves for the installed version.
 *
 * @return array|false Array of vulnerabilities, or false on error.
 */
function wpvulnerability_get_core( $version = null ) {

	// Sanitize the version number.
	if( !wpvulnerability_sanitize_version( $version ) ) {
		$version = null;
	}

	// If version number is null, retrieve for the installed version.
	if ( is_null( $version ) ) {
		$version = get_bloginfo( 'version' );
	}

	// Get vulnerabilities from API.
	$response = wpvulnerability_get( 'core', $version );

	// Check for errors.
	if ( empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Process vulnerabilities and return as an array.
	$vulnerability = array();
	foreach ( $response['data']['vulnerability'] as $v ) {

		$vulnerability[] = array(
			'name' => wp_kses( $v['name'], 'strip' ),
			'link' => esc_url_raw( $v['link'] ),
			'source' => $v['source'],
			'impact' => $v['impact'],
		);

	}
	return $vulnerability;
}

/**
 * Get vulnerabilities for a specific plugin.
 *
 * @since 2.0.0
 *
 * @param string $slug    Slug of the plugin.
 * @param string $version Version of the plugin.
 *
 * @return array|false Returns an array of vulnerabilities, or false if there are none.
 */
function wpvulnerability_get_plugin( $slug, $version ) {

	// Sanitize the plugin slug.
	$slug = sanitize_title( $slug );

	// If the version number is invalid, return false.
	if( !wpvulnerability_sanitize_version( $version ) ) {
		return false;
	}

	// Get the response from the vulnerability API.
	$response = wpvulnerability_get( 'plugin', $slug );

	// Create an empty array to store the vulnerabilities.
	$vulnerability = array();

	// If there are no vulnerabilities, return false.
	if ( empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	// Loop through each vulnerability and check if it affects the specified version of the plugin.
	foreach ( $response['data']['vulnerability'] as $v ) {

		// If the vulnerability has minimum and maximum versions, check if the specified version falls within that range.
		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {
			
			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {
				
				// Add the vulnerability to the array.
				$vulnerability[] = array(
					'name' => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version' => wp_kses( $v['operator']['min_version'], 'strip' ),
					'unfixed' => (int)$v['operator']['unfixed'],
					'closed' => (int)$v['operator']['closed'],
					'source' => $v['source'],
					'impact' => $v['impact'],
				);

			}

		// If the vulnerability has only a maximum version, check if the specified version is below that version.
		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name' => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version' => wp_kses( $v['operator']['max_version'], 'strip' ),
					'unfixed' => (int)$v['operator']['unfixed'],
					'closed' => (int)$v['operator']['closed'],
					'source' => $v['source'],
					'impact' => $v['impact'],
				);

			}

		// If the vulnerability has a minimum version and maximum version, check if the specified version is within that range.
		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name' => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'], 'strip' ),
					'version' => wp_kses( $v['operator']['min_version'], 'strip' ),
					'unfixed' => (int)$v['operator']['unfixed'],
					'closed' => (int)$v['operator']['closed'],
					'source' => $v['source'],
					'impact' => $v['impact'],
				);

			}

		}

	}

	return $vulnerability;

}

/**
 * Get vulnerabilities for a specific theme.
 *
 * @since 2.0.0
 *
 * @param string $slug    Slug of the theme.
 * @param string $version Version of the theme.
 *
 * @return array|false Returns an array of vulnerabilities, or false if there are none.
 */
function wpvulnerability_get_theme( $slug, $version ) {

	// Sanitize the plugin slug.
	$slug = sanitize_title( $slug );

	// Get the response from the vulnerability API.
	if( !wpvulnerability_sanitize_version( $version ) ) {
		return false;
	}

	// Get the response from the vulnerability API.
	$response = wpvulnerability_get( 'theme', $slug );

	// Create an empty array to store the vulnerabilities.
	$vulnerability = array();

	// If there are no vulnerabilities, return false.
	if ( empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	foreach ( $response['data']['vulnerability'] as $v ) {

		// If the vulnerability has minimum and maximum versions, check if the specified version falls within that range.
		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name' => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_version'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version' => wp_kses( $v['operator']['min_version'], 'strip' ),
					'unfixed' => (int)$v['operator']['unfixed'],
					'closed' => (int)$v['operator']['closed'],
					'source' => $v['source'],
					'impact' => $v['impact'],
				);

			}

		// If the vulnerability has only a maximum version, check if the specified version is below that version.
		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name' => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version' => wp_kses( $v['operator']['max_version'], 'strip' ),
					'unfixed' => (int)$v['operator']['unfixed'],
					'closed' => (int)$v['operator']['closed'],
					'source' => $v['source'],
					'impact' => $v['impact'],
				);

			}

		// If the vulnerability has only a maximum version, check if the specified version is below that version.
		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

				// Add the vulnerability to the list.
				$vulnerability[] = array(
					'name' => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_version'] ) . $v['operator']['min_version'], 'strip' ),
					'version' => wp_kses( $v['operator']['min_version'], 'strip' ),
					'unfixed' => (int)$v['operator']['unfixed'],
					'closed' => (int)$v['operator']['closed'],
					'source' => $v['source'],
					'impact' => $v['impact'],
				);

			}

		}

	}

	return $vulnerability;

}

/**
 * Get statistics
 *
 * Returns an array with statistical information about vulnerabilities and their respective products.
 *
 * @since 2.0.0
 *
 * @return array|false Returns an array with the statistical information if successful, false otherwise.
 */
function wpvulnerability_get_statistics() {

	$key = 'wpvulnerability_stats';

	// Get cached statistics if available
	$vulnerability = get_transient( $key );

	// If cached statistics are not available, retrieve them from the API and store them in cache
	if ( empty( $vulnerability ) ) {

		$url = WPVULNERABILITY_API_HOST;
		// Parse the JSON response into an associative array
		$response = wp_remote_get( $url );

		if ( !is_wp_error( $response ) ) {

			$body = wp_remote_retrieve_body( $response );
			set_transient( $key, $body, HOUR_IN_SECONDS * 12 );

		}

	}

	// If the response does not contain statistics, return false
	$response = json_decode( $vulnerability, true );

	if ( ! isset( $response['stats'] ) ) {
		return false;
	}

	// Return an array with statistical information
	return array(
		'core' => array(
			'versions' => (int)$response['stats']['products']['core']
		),
		'plugins' => array(
			'products' => (int)$response['stats']['products']['plugins'],
			'vulnerabilities' => (int)$response['stats']['plugins']
		),
		'themes' => array(
			'products' => (int)$response['stats']['products']['themes'],
			'vulnerabilities' => (int)$response['stats']['themes']
		),
		'updated' => array(
			'unixepoch' => (int)$response['updated'],
			'datetime'  => gmdate( 'Y-m-d H:i:s', (int)$response['updated'] ),
			'iso8601'   => gmdate( 'c', (int)$response['updated'] ),
			'rfc2822'   => gmdate( 'r', (int)$response['updated'] ),
		)
	);

}
