<?php
/**
 * API Class
 *
 * @package    WordPress
 * @author     David Perez <david@closemarketing.es>
 * @copyright  2022 Closemarketing
 * @version    1.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Sanitize a version string.
 *
 * @param string $version The version string to sanitize.
 * @return string The sanitized version string.
 */
function wpvulnerability_sanitize_version( $version ) {

	// Remove any leading/trailing whitespace
	// Strip out any non-alphanumeric characters except for hyphens, underscores, and dots
	$version = trim( preg_replace( '/[^a-zA-Z0-9_\-.]+/', '', $version ) );

	return $version;

}

/**
 * Returns a Pretty Operator
 *
 * @param string $version The operator string to prettify.
 * @return string The pretty operator html string.
 */
function wpvulnerability_pretty_operator( $op ) {
  switch( trim( $op ) ) {
    case 'lt':
      return '&lt;&nbsp;';
      break;
    case 'le':
      return '&le;&nbsp;';
      break;
    case 'gt':
      return '&gt;&nbsp;';
      break;
    case 'ge':
      return '&ge;&nbsp;';
      break;
    case 'eq':
      return '&equals;&nbsp;';
      break;
    case 'ne':
      return '&ne;&nbsp;';
      break;
    default:
      return $op;
      break;
  }
}

/**
 * Gets vulnerabilities info from API
 *
 * @param string $type Type of query.
 * @param string $slug Slug.
 * @return array
 */
function wpvulnerability_get( $type, $slug = '' ) {

	$args = array(
		'timeout'   => 3000,
		'sslverify' => false,
	);

	switch( $type ) {
		case 'core':
			$type = 'core';
			break;
		case 'plugin':
			$type = 'plugin';
			break;
		case 'theme':
			$type = 'theme';
			break;
		default: 
			wp_die( 'Unknown vulnerability type sent.' );
			break;
	}

	if( 'plugin' == $type || 'theme' == $type ) {

		if( empty( sanitize_title( $slug ) ) ) {
			return false;
		}

	} elseif( $type == 'core' ) {

		if( !wpvulnerability_sanitize_version( $slug ) ) {
			return false;
		}

	}

	$key = 'wpvulnerability_' . $type . '_' . $slug;

	$vulnerability = get_transient( $key );

	if ( empty( $vulnerability ) ) {

		$url = 'https://www.wpvulnerability.net/' . $type . '/' . $slug . '/';
		$response = wp_remote_get( $url, $args );

		if ( !is_wp_error( $response ) ) {

			$body = wp_remote_retrieve_body( $response );
			set_transient( $key, $body, HOUR_IN_SECONDS * 12 );

		}

	}

	return json_decode( $vulnerability, true );

}

/**
 * Get vulnerabilities from Core
 *
 * @param string $version Version of core.
 * @return array
 */
function wpvulnerability_get_core( $version = null ) {

	if( !wpvulnerability_sanitize_version( $version ) ) {
		$version = null;
	}

	if ( is_null( $version ) ) {
		$version = get_bloginfo( 'version' );
	}

	$response = wpvulnerability_get( 'core', $version );

	$vulnerability = array();

	if ( empty( $response['data']['vulnerability'] ) ) {
		return false;
	}
	
	foreach ( $response['data']['vulnerability'] as $v ) {

		$vulnerability[] = array(
			'name' => wp_kses( $v['name'], 'strip' ),
			'link' => sanitize_url( $v['link'] ),
			'source' => $v['source'],
			'impact' => $v['impact'],
		);

	}
	return $vulnerability;
}

/**
 * Get vulnerabilities from Plugin
 *
 * @param string $slug Slug of plugin.
 * @param string $version Version of plugin.
 * @return array
 */
function wpvulnerability_get_plugin( $slug, $version ) {

	$slug = sanitize_title( $slug );

	if( !wpvulnerability_sanitize_version( $version ) ) {
		return false;
	}

	$response = wpvulnerability_get( 'plugin', $slug );

	$vulnerability = array();

	if ( empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	foreach ( $response['data']['vulnerability'] as $v ) {

		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {
			
			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {
				
				$vulnerability[] = array(
					'name' => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version' => wp_kses( $v['operator']['min_version'], 'strip' ),
					'unfixed' => (int)$v['operator']['unfixed'],
					'closed' => (int)$v['operator']['closed'],
					'source' => $v['source'],
					'impact' => $v['impact'],
				);

			}

		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				$vulnerability[] = array(
					'name' => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version' => wp_kses( $v['operator']['max_version'], 'strip' ),
					'unfixed' => (int)$v['operator']['unfixed'],
					'closed' => (int)$v['operator']['closed'],
					'source' => $v['source'],
					'impact' => $v['impact'],
				);

			}

		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

				$vulnerability[] = array(
					'name' => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_operator'] ) . $v['operator']['min_version'], 'strip' ),
					'version' => wp_kses( $v['operator']['min_version'], 'strip' ),
					'unfixed' => (int)$v['operator']['unfixed'],
					'closed' => (int)$v['operator']['closed'],
					'source' => $v['source'],
					'impact' => $v['impact'],
				);

			}

		}

	}

	return $vulnerability;

}

/**
 * Get vulnerabilities from Theme
 *
 * @param string $slug Slug of theme.
 * @param string $version Version of theme.
 * @return array
 */
function wpvulnerability_get_theme( $slug, $version ) {

	$slug = sanitize_title( $slug );

	if( !wpvulnerability_sanitize_version( $version ) ) {
		return false;
	}

	$response = wpvulnerability_get( 'theme', $slug );

	$vulnerability = array();

	if ( empty( $response['data']['vulnerability'] ) ) {
		return false;
	}

	foreach ( $response['data']['vulnerability'] as $v ) {

		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				$vulnerability[] = array(
					'name' => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_version'] ) . $v['operator']['min_version'] . ' - ' . wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version' => wp_kses( $v['operator']['min_version'], 'strip' ),
					'unfixed' => (int)$v['operator']['unfixed'],
					'closed' => (int)$v['operator']['closed'],
					'source' => $v['source'],
					'impact' => $v['impact'],
				);

			}

		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {

			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {

				$vulnerability[] = array(
					'name' => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['max_operator'] ) . $v['operator']['max_version'], 'strip' ),
					'version' => wp_kses( $v['operator']['max_version'], 'strip' ),
					'unfixed' => (int)$v['operator']['unfixed'],
					'closed' => (int)$v['operator']['closed'],
					'source' => $v['source'],
					'impact' => $v['impact'],
				);

			}

		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {

			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {

				$vulnerability[] = array(
					'name' => wp_kses( $v['name'], 'strip' ),
					'description' => wp_kses_post( $v['description'] ),
					'versions' => wp_kses( wpvulnerability_pretty_operator( $v['operator']['min_version'] ) . $v['operator']['min_version'], 'strip' ),
					'version' => wp_kses( $v['operator']['min_version'], 'strip' ),
					'unfixed' => (int)$v['operator']['unfixed'],
					'closed' => (int)$v['operator']['closed'],
					'source' => $v['source'],
					'impact' => $v['impact'],
				);

			}

		}

	}

	return $vulnerability;

}

/**
 * Get statistics
 *
 * @return array
 */
function wpvulnerability_get_statistics() {

	$key = 'wpvulnerability_stats';

	$vulnerability = get_transient( $key );

	if ( empty( $vulnerability ) ) {

		$url = 'https://www.wpvulnerability.net/';
		$response = wp_remote_get( $url );

		if ( !is_wp_error( $response ) ) {

			$body = wp_remote_retrieve_body( $response );
			set_transient( $key, $body, HOUR_IN_SECONDS * 12 );

		}

	}

	$response = json_decode( $vulnerability, true );

	if ( ! isset( $response['data']['stats'] ) ) {
		return false;
	}

	return array(
		'core'    => (int)$response['data']['stats']['core'],
		'plugins' => (int)$response['data']['stats']['plugins'],
		'themes'  => (int)$response['data']['stats']['themes'],
		'updated' => array(
			'unixepoch' => (int)$response['updated'],
			'datetime'  => gmdate( 'Y-m-d H:i:s', (int)$response['updated'] ),
			'iso8601'   => gmdate( 'c', (int)$response['updated'] ),
			'rfc2822'   => gmdate( 'r', (int)$response['updated'] ),
		)
	);

}

/**
 * Convert vulnerabilities in html
 *
 * @param array $vulnerabilities Vulnerabilites.
 * @return html
 */
function wpvulnerability_get_html_vulnerabilities( $vulnerabilities, $type = 'columns' ) {

	$html = '<ul>';

	foreach ( $vulnerabilities as $vulnerability ) {

		foreach ( $vulnerability['source'] as $source ) {

			$html .= '<li><b>' . wp_kses( $vulnerability['name'], 'strip' ) . '</b><br>' . wp_kses( $source['name'], 'strip' );
			//if ( str_contains( $type, 'no-columns' ) ) {
			//	$html .= '</tr><tr>';
			//}
			if ( ! str_contains( $type, 'no-desc' ) ) {

				$html .= '<br><br><i>' . wp_kses_post( $source['description'] ) . '</i>';

			}

			$html .= '</li>';

		}

	}

	$html .= '</ul>';

	return $html;

}

/**
 * Convert Theme Vulnerabilities in html
 *
 * @param string $type Type of markup.
 * @return html
 */
function wpvulnerability_get_html_theme_vulnerabilities( $type = 'columns' ) {

	$html_vuln = '';
	$found_vulnerabilities = false;
	$themes = wp_get_themes();

	foreach ( $themes as $slug => $theme ) {

		$slug = sanitize_title( $slug );

		$theme_vulnerability = wpvulnerability_get_theme( $slug, $theme->get( 'Version' ) );

		if ( ! empty( $theme_vulnerability ) ) {

			$html_vuln .= '<h3>Theme: ' . $theme->get( 'Name' ) . '</h3>';
			$html_vuln .= wpvulnerability_get_html_vulnerabilities( $theme_vulnerability, $type );

			$found_vulnerabilities = true;

		}

	}

	if ( $found_vulnerabilities ) {
		return $html_vuln;
	} else {
		return false;
	}

}

/**
 * Convert Plugin vulnerabilities in html
 *
 * @param string $type Type of markup.
 * @return html
 */
function wpvulnerability_get_html_plugin_vulnerabilities( $type = 'columns' ) {

	$html_vuln = '';
	$found_vulnerabilities = false;
	$plugins = get_plugins();

	foreach ( $plugins as $key => $plugin ) {

		if ( empty( $plugin['TextDomain'] ) && isset( $plugin['file_path'] ) ) {

			$folder_name = explode( '/', $plugin['file_path'] );
			$slug = sanitize_title( $folder_name[0] );

		} else {

			$slug = sanitize_title( $plugin['TextDomain'] );

		}

		$plugin_vulnerability = wpvulnerability_get_plugin( $slug, $plugin['Version'] );

		if ( ! empty( $plugin_vulnerability ) ) {

			$html_vuln .= '<h3>Plugin: ' . $plugin['Name'] . '</h3>';
			$html_vuln .= wpvulnerability_get_html_vulnerabilities( $plugin_vulnerability, $type );

			$found_vulnerabilities = true;

		}

	}

	if ( $found_vulnerabilities ) {
		return $html_vuln;
	} else {
		return false;
	}

}
