<?php
/**
 * Primary class file for the Health Check plugin.
 *
 * @package Health Check
 */

// Make sure the file is not directly accessible.
if ( ! defined( 'ABSPATH' ) ) {
	die( 'We\'re sorry, but you can not directly access this file.' );
}

/**
 * Class HealthCheck
 */
class Health_WPVulnerability {
	/**
	 * Construct Class for Health Kit
	 */
	public function __construct() {
		add_filter( 'site_status_tests', array( $this, 'add_vulnerability_tests' ) );
	}

	/**
	 * Add vulnerability tests in Health
	 *
	 * @param array $tests Actual tests.
	 * @return array
	 */
	public function add_vulnerability_tests( $tests ) {
		$tests['direct']['wpvulnerability_core'] = array(
			'label' => __( 'WPVulnerability Core', 'wpvulnerability' ),
			'test'  => array( $this, 'test_vulnerability_core' ),
		);

		$tests['direct']['wpvulnerability_themes'] = array(
			'label' => __( 'WPVulnerability Themes', 'wpvulnerability' ),
			'test'  => array( $this, 'test_vulnerability_themes' ),
		);

		$tests['direct']['wpvulnerability_plugins'] = array(
			'label' => __( 'WPVulnerability Plugins', 'wpvulnerability' ),
			'test'  => array( $this, 'test_vulnerability_plugins' ),
		);

		return $tests;
	}

	/**
	 * Tests for Vulnerability in Core
	 *
	 * @return array
	 */
	public function test_vulnerability_core() {
		$result = array(
			'label'       => __( 'There aren\'t WordPress vulnerabilities', 'wpvulnerability' ),
			'status'      => 'good',
			'badge'       => array(
				'label' => __( 'Security', 'wpvulnerability' ),
				'color' => 'green',
			),
			'description' => sprintf(
				'<p>%s</p>',
				__( 'Shows possible vulnerabilities existing in the WordPress core.', 'wpvulnerability' )
			),
			'actions'     => '',
			'test'        => 'wpvulnerability_core',
		);

		$core_vulnerabilities = wpvulnerability_get_core();
		if ( ! empty( $core_vulnerabilities ) ) {
			$result['status']      = 'critical';
			$result['label']       = __( 'There are WordPress Core vulnerabilities', 'wpvulnerability' );
			$result['badge']['color'] = 'red';
			$result['description'] = sprintf(
				'<p>%1$s</p> %2$s',
				__( 'We\'ve detected potential vulnerabilities in this WordPress installation. Please check them and keep your installation updated.', 'wpvulnerability' ),
				wpvulnerability_get_html_vulnerabilities( $core_vulnerabilities )
			);
			$result['actions']    .= sprintf(
				'<p><a href="%s">%s</a></p>',
				esc_url( admin_url( 'update-core.php' ) ),
				__( 'Update WordPress', 'wpvulnerability' )
			);
		}

		return $result;
	}

	/**
	 * Tests for Vulnerability in Themes
	 *
	 * @return array
	 */
	public function test_vulnerability_themes() {
		$result = array(
			'label'       => __( 'There aren\'t themes vulnerabilities', 'wpvulnerability' ),
			'status'      => 'good',
			'badge'       => array(
				'label' => __( 'Security', 'wpvulnerability' ),
				'color' => 'green',
			),
			'description' => sprintf(
				'<p>%s</p>',
				__( 'Shows possible vulnerabilities that exist in installed themes.', 'wpvulnerability' )
			),
			'actions'     => '',
			'test'        => 'wpvulnerability_themes',
		);

		$html_vuln = wpvulnerability_get_html_theme_vulnerabilities();

		if ( ! empty( $html_vuln ) ) {
			$result['status']      = 'critical';
			$result['label']       = __( 'There are themes vulnerabilities', 'wpvulnerability' );
			$result['badge']['color'] = 'red';
			$result['description'] = sprintf(
				'<p>%1$s</p> %2$s',
				__( 'We\'ve detected potential vulnerabilities in installed themes. Please check them and keep them updated.', 'wpvulnerability' ),
				$html_vuln
			);
			$result['actions']    .= sprintf(
				'<p><a href="%s">%s</a></p>',
				esc_url( admin_url( 'themes.php' ) ),
				__( 'Update themes', 'wpvulnerability' )
			);
		}

		return $result;
	}

	/**
	 * Tests for Vulnerability in Plugins
	 *
	 * @return array
	 */
	public function test_vulnerability_plugins() {
		$result = array(
			'label'       => __( 'There aren\'t plugins vulnerabilities', 'wpvulnerability' ),
			'status'      => 'good',
			'badge'       => array(
				'label' => __( 'Security', 'wpvulnerability' ),
				'color' => 'green',
			),
			'description' => sprintf(
				'<p>%s</p>',
				__( 'Shows possible vulnerabilities that exist in installed plugins.', 'wpvulnerability' )
			),
			'actions'     => '',
			'test'        => 'wpvulnerability_plugins',
		);

		$html_vuln = wpvulnerability_get_html_plugin_vulnerabilities();

		if ( ! empty( $html_vuln ) ) {
			$result['status']      = 'critical';
			$result['label']       = __( 'There are plugins vulnerabilities', 'wpvulnerability' );
			$result['badge']['color'] = 'red';
			$result['description'] = sprintf(
				'<p>%1$s</p> %2$s',
				__( 'We\'ve detected potential vulnerabilities in installed plugins. Please check them and keep them updated.', 'wpvulnerability' ),
				$html_vuln
			);
			$result['actions']    .= sprintf(
				'<p><a href="%s">%s</a></p>',
				esc_url( admin_url( 'plugins.php' ) ),
				__( 'Update plugins', 'wpvulnerability' )
			);
		}

		return $result;
	}
}

new Health_WPVulnerability();
