<?php
/**
 * API Class
 *
 * @package    WordPress
 * @author     David Perez <david@closemarketing.es>
 * @copyright  2022 Closemarketing
 * @version    1.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Gets vulnerabilities info from API
 *
 * @param string $type Type of query.
 * @param string $slug Slug.
 * @return array
 */
function wpvulnerability_get( $type, $slug = '' ) {
	$args = array(
		'timeout'   => 3000,
		'sslverify' => false,
	);
	$key = 'wpvulnerability_' . $type;
	if ( ! empty( $slug ) ) {
		$key .= '_' . $slug;
	}
	$vulnerability = get_transient( $key );
	if ( ! $vulnerability ) {
		$url = 'https://www.wpvulnerability.net/' . $type . '/';
		if ( ! empty( $slug ) ) {
			$url .= $slug . '/';
		}
		$response = wp_remote_get( $url, $args );
		if ( 200 === $response->get_error_code() ) {
			$body = wp_remote_retrieve_body( $response );
			set_transient( $key, $body, HOUR_IN_SECONDS * 4 );
		}
	}
	return json_decode( $vulnerability, true );
}

/**
 * Get vulnerabilities from Core
 *
 * @param string $version Version of core.
 * @return array
 */
function wpvulnerability_get_core( $version = null ) {
	if ( empty( $version ) ) {
		$version = get_bloginfo( 'version' );
	}
	$response      = wpvulnerability_get( 'core', $version );
	$vulnerability = array();
	if ( ! isset( $response['data']['vulnerability'] ) ) {
		return false;
	}
	foreach ( $response['data']['vulnerability'] as $v ) {
		$vulnerability[] = array(
			'name'   => $v['name'],
			'link'   => $v['link'],
			'source' => $v['source'],
		);
	}
	return $vulnerability;
}

/**
 * Get vulnerabilities from Plugin
 *
 * @param string $slug Slug of plugin.
 * @param string $version Version of plugin.
 * @return array
 */
function wpvulnerability_get_plugin( $slug, $version ) {
	$response      = wpvulnerability_get( 'plugin', $slug );
	$vulnerability = array();
	if ( ! isset( $response['data']['vulnerability'] ) ) {
		return false;
	}
	foreach ( $response['data']['vulnerability'] as $v ) {
		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {
			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {
				$vulnerability[] = array(
					'name'        => $v['name'],
					'description' => $v['description'],
					'versions'    => $v['operator']['min_version'] . $v['operator']['min_version'] . ' - ' . $v['operator']['max_operator'] . $v['operator']['max_version'],
					'unfixed'     => $v['operator']['unfixed'],
					'closed'      => $v['operator']['closed'],
					'source'      => $v['source'],
				);
				unset( $msg );
			}
		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {
			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {
				$vulnerability[] = array(
					'name'        => $v['name'],
					'description' => $v['description'],
					'versions'    => $v['operator']['max_operator'] . $v['operator']['max_version'],
					'unfixed'     => $v['operator']['unfixed'],
					'closed'      => $v['operator']['closed'],
					'source'      => $v['source'],
				);
				unset( $msg );
			}
		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {
			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {
				$vulnerability[] = array(
					'name'        => $v['name'],
					'description' => $v['description'],
					'versions'    => $v['operator']['min_version'] . $v['operator']['min_version'],
					'unfixed'     => $v['operator']['unfixed'],
					'closed'      => $v['operator']['closed'],
					'source'      => $v['source'],
				);
				unset( $msg );
			}
		}
	}
	return $vulnerability;
}

/**
 * Get vulnerabilities from Theme
 *
 * @param string $slug Slug of theme.
 * @param string $version Version of theme.
 * @return array
 */
function wpvulnerability_get_theme( $slug, $version ) {
	$response      = wpvulnerability_get( 'theme', $slug );
	$vulnerability = array();
	if ( ! isset( $response['data']['vulnerability'] ) ) {
		return false;
	}
	foreach ( $response['data']['vulnerability'] as $v ) {
		if ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] && isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {
			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) && version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {
				$vulnerability[] = array(
					'name'        => $v['name'],
					'description' => $v['description'],
					'versions'    => $v['operator']['min_version'] . $v['operator']['min_version'] . ' - ' . $v['operator']['max_operator'] . $v['operator']['max_version'],
					'unfixed'     => $v['operator']['unfixed'],
					'closed'      => $v['operator']['closed'],
					'source'      => $v['source'],
				);
				unset( $msg );
			}
		} elseif ( isset( $v['operator']['max_operator'] ) && $v['operator']['max_operator'] ) {
			if ( version_compare( $version, $v['operator']['max_version'], $v['operator']['max_operator'] ) ) {
				$vulnerability[] = array(
					'name'        => $v['name'],
					'description' => $v['description'],
					'versions'    => $v['operator']['max_operator'] . $v['operator']['max_version'],
					'unfixed'     => $v['operator']['unfixed'],
					'closed'      => $v['operator']['closed'],
					'source'      => $v['source'],
				);
				unset( $msg );
			}
		} elseif ( isset( $v['operator']['min_operator'] ) && $v['operator']['min_operator'] ) {
			if ( version_compare( $version, $v['operator']['min_version'], $v['operator']['min_operator'] ) ) {
				$vulnerability[] = array(
					'name'        => $v['name'],
					'description' => $v['description'],
					'versions'    => $v['operator']['min_version'] . $v['operator']['min_version'],
					'unfixed'     => $v['operator']['unfixed'],
					'closed'      => $v['operator']['closed'],
					'source'      => $v['source'],
				);
				unset( $msg );
			}
		}
	}
	return $vulnerability;
}

/**
 * Get statistics
 *
 * @return array
 */
function wpvulnerability_get_statistics() {
	$key           = 'wpvulnerability_stats';
	$vulnerability = get_transient( $key );
	if ( ! $vulnerability ) {
		$url      = 'https://www.wpvulnerability.net/';
		$response = wp_remote_get( $url );
		if ( 200 === $response->get_error_code() ) {
			$body = wp_remote_retrieve_body( $response );
			set_transient( $key, $body, HOUR_IN_SECONDS * 4 );
		}
	}
	$response = json_decode( $vulnerability, true );
	if ( ! isset( $response['data']['stats'] ) ) {
		return false;
	}
	return array(
		'core'    => $response['data']['stats']['core'],
		'plugins' => $response['data']['stats']['plugins'],
		'themes'  => $response['data']['stats']['themes'],
		'updated' => array(
			'unixepoch' => $response['updated'],
			'datetime'  => gmdate( 'Y-m-d H:i:s', $response['updated'] ),
			'iso8601'   => gmdate( 'c', $response['updated'] ),
			'rfc2822'   => gmdate( 'r', $response['updated'] ),
		),
	);
}

/**
 * Convert vulnerabilities in html
 *
 * @param array $vulnerabilities Vulnerabilites.
 * @return html
 */
function wpvulnerability_get_html_vulnerabilities( $vulnerabilities, $type = 'columns' ) {
	$html = '<ul>';
	foreach ( $vulnerabilities as $vulnerability ) {
		foreach ( $vulnerability['source'] as $source ) {
			$html .= '<li><b>' . esc_html( $vulnerability['name'] ) . '</b><br>' . esc_html( $source['name'] );
			//if ( str_contains( $type, 'no-columns' ) ) {
			//	$html .= '</tr><tr>';
			//}
			if ( ! str_contains( $type, 'no-desc' ) ) {
				$html .= '<br><i>' . esc_html( $source['description'] ) . '</i>';
			}
			$html .= '</li>';
		}
	}
	$html .= '</ul>';
	return $html;
}

/**
 * Convert Theme Vulnerabilities in html
 *
 * @param string $type Type of markup.
 * @return html
 */
function wpvulnerability_get_html_theme_vulnerabilities( $type = 'columns' ) {
	$html_vuln             = '';
	$found_vulnerabilities = false;
	$themes                = wp_get_themes();
	foreach ( $themes as $slug => $theme ) {
		$slug                = sanitize_title( $slug );
		$theme_vulnerability = wpvulnerability_get_theme( $slug, $theme->get( 'Version' ) );
		if ( ! empty( $theme_vulnerability ) ) {
			$html_vuln .= '<h3>Theme: ' . $theme->get( 'Name' ) . '</h3>';
			$html_vuln .= wpvulnerability_get_html_vulnerabilities( $theme_vulnerability, $type );

			$found_vulnerabilities = true;
		}
	}
	if ( $found_vulnerabilities ) {
		return $html_vuln;
	} else {
		return false;
	}
}

/**
 * Convert Plugin vulnerabilities in html
 *
 * @param string $type Type of markup.
 * @return html
 */
function wpvulnerability_get_html_plugin_vulnerabilities( $type = 'columns' ) {
	$found_vulnerabilities = false;
	$plugins   = get_plugins();
	$html_vuln = '';
	foreach ( $plugins as $key => $plugin ) {
		if ( empty( $plugin['TextDomain'] ) && isset( $plugin['file_path'] ) ) {
			$folder_name = explode( '/', $plugin['file_path'] );
			$slug        = $folder_name[0];
		} else {
			$slug = $plugin['TextDomain'];
		}
		$plugin_vulnerability = wpvulnerability_get_plugin( $slug, $plugin['Version'] );
		if ( ! empty( $plugin_vulnerability ) ) {
			$html_vuln .= '<h3>Plugin: ' . $plugin['Name'] . '</h3>';
			$html_vuln .= wpvulnerability_get_html_vulnerabilities( $plugin_vulnerability, $type );

			$found_vulnerabilities = true;
		}
	}
	if ( $found_vulnerabilities ) {
		return $html_vuln;
	} else {
		return false;
	}
}
