<?php
/**
 * Notification options
 *
 * @package    WordPress
 * @author     David Perez <david@close.technology>
 * @copyright  2022 Closemarketing
 * @version    1.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Notification.
 *
 * Notificate to the user.
 *
 * @since 1.2
 */
class WPVUL_Notification {
	/**
	 * Construct of Class
	 */
	public function __construct() {
		add_filter( 'cron_schedules', array( $this, 'add_every_week' ) );
		add_filter( 'cron_schedules', array( $this, 'add_every_day' ) );
		$this->settings = get_option( 'wpvulnerability_settings' );
		$period         = isset( $this->settings['period'] ) ? $this->settings['period'] : '';
		if ( $period ) {
			if ( ! wp_next_scheduled( 'wpvulnerability_notification' ) ) {
				wp_schedule_event( time(), $period, 'wpvulnerability_notification' );
			}
			add_action( 'wpvulnerability_notification', array( $this, 'execute_notification' ) );
		}
	}

	/**
	 * Adds schedule every week
	 *
	 * @param array $schedules System schedules.
	 * @return array
	 */
	public function add_every_week( $schedules ) {
		$schedules['weekly'] = array(
			'interval' => 604800,
			'display'  => __( 'Every week', 'wpvulnerability' ),
		);
		return $schedules;
	}
	
	/**
	 * Adds schedule every week
	 *
	 * @param array $schedules System schedules.
	 * @return array
	 */
	public function add_every_day( $schedules ) {
		$schedules['daily'] = array(
			'interval' => 86400,
			'display'  => __( 'Every day', 'wpvulnerability' ),
		);
		return $schedules;
	}

	/**
	 * Executes notification
	 *
	 * @return void
	 */
	public function execute_notification() {
		$email  = isset( $this->settings['emails'] ) ? $this->settings['emails'] : get_bloginfo( 'admin_email' );
		$period = isset( $this->settings['period'] ) ? $this->settings['period'] : '';
		if ( empty( $period ) ) {
			return;
		}
		// Calc vulnerabilities.
		$core_vulnerabilities        = wpvulnerability_get_core();
		$html_plugin_vulnerabilities = wpvulnerability_get_html_plugin_vulnerabilities( 'no-columns,no-desc' );
		$html_theme_vulnerabilities  = wpvulnerability_get_html_theme_vulnerabilities( 'no-columns,no-desc' );
		if ( empty( $core_vulnerabilities ) && false === $html_plugin_vulnerabilities && false === $html_plugin_vulnerabilities ) {
			return; // don't send email.
		}
		if ( ! empty( $core_vulnerabilities ) ) {
			$email_content  = '<h2>' . esc_html__( 'Core vulnerabilities', 'wpvulnerability' ) . '</h2>';
			$email_content .= wpvulnerability_get_html_vulnerabilities( $core_vulnerabilities, 'no-columns,no-desc' );
		}
		// Plugins vulnerabilities.
		if ( $html_plugin_vulnerabilities ) {
			$email_content .= '<h2>' . esc_html__( 'Plugins vulnerabilities', 'wpvulnerability' ) . '</h2>';
			$email_content .= $html_plugin_vulnerabilities;
		}
		// Themes vulnerabilities.
		if ( $html_theme_vulnerabilities ) {
			$email_content .= '<h2>' . esc_html__( 'Themes vulnerabilities', 'wpvulnerability' ) . '</h2>';
			$email_content .= $html_theme_vulnerabilities;
		}
		$email_content_hml = $this->html_email( esc_html__( 'Vulnerability found', 'wpvulnerability' ), $email_content );
		// Prepare email.
		$email_headers = array();
		$email_subject = sprintf(
			// translators: site name.
			__( 'Vulnerability found: %s', 'wpvulnerability' ),
			get_bloginfo( 'name' )
		);
		$email_headers[] = 'From: WPVulnerability <' . get_bloginfo( 'admin_email' ) . '>';
		$email_headers[] = 'Content-Type: text/html; charset=UTF-8';
		wp_mail( $email, $email_subject, $email_content_hml, $email_headers );
	}

	/**
	 * Creates the HTML Email
	 *
	 * @param string $title   Title of Email.
	 * @param string $content Content of the email.
	 * @return html
	 */
	private function html_email( $title, $content ) {
		$message  = '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
<html xmlns="http://www.w3.org/1999/xhtml" style="font-family: Helvetica Neue, Helvetica, Arial, sans-serif; box-sizing: border-box; font-size:18px; margin: 0;">
<head>
	<meta name="viewport" content="width=device-width" />
	<meta http-equiv="Content-Type" content="text/html; charset=UTF-8" />
	<title>WPVulnerability</title>
	<style type="text/css">
	img { max-width: 100%; }
  body { -webkit-font-smoothing: antialiased; -webkit-text-size-adjust: none; width: 100% !important; height: 100%; line-height: 1.6em; }
  body { background-color: #f6f6f6; }
  @media only screen and (max-width: 640px) {
		body { padding: 0 !important; }
		h1 { font-weight: 800 !important; margin: 20px 0 5px !important; }
		h2 { font-weight: 800 !important; margin: 20px 0 5px !important; }
		h3 { font-weight: 800 !important; margin: 20px 0 5px !important; }
		h4 { font-weight: 800 !important; margin: 20px 0 5px !important; }
		h1 { font-size: 22px !important; } 
		h2 { font-size: 18px !important; }
		h3 { font-size: 16px !important; }
		.container { padding: 0 !important; width: 100% !important; }
		.content { padding: 0 !important; }
		.content-wrap { padding: 10px !important; }
		.invoice { width: 100% !important; }
  }
  </style>
</head>
<body itemscope itemtype="http://schema.org/EmailMessage" style="font-family: sans-serif; box-sizing: border-box; font-size:18px; -webkit-font-smoothing: antialiased; -webkit-text-size-adjust: none; width: 100% !important; height: 100%; line-height: 1.6em; background-color: #f6f6f6; margin: 0;" bgcolor="#f6f6f6">
	<table class="body-wrap" style="font-family: sans-serif; box-sizing: border-box; font-size:18px; width: 100%; background-color: #f6f6f6; margin: 0;" bgcolor="#f6f6f6">
		<tr style="font-family: sans-serif; box-sizing: border-box; font-size:18px; margin: 0;">
			<td style="font-family: sans-serif; box-sizing: border-box; font-size:18px; vertical-align: top; margin: 0;" valign="top"></td>
			<td class="container" width="600" style="font-family: sans-serif; box-sizing: border-box; font-size:18px; vertical-align: top; display: block !important; max-width: 600px !important; clear: both !important; margin: 0 auto;" valign="top">
				<div class="content" style="font-family: sans-serif; box-sizing: border-box; font-size:18px; max-width: 600px; display: block; margin: 0 auto; padding: 20px;">
					<table class="main" width="100%" cellpadding="0" cellspacing="0" style="font-family: sans-serif; box-sizing: border-box; font-size:18px; border-radius: 3px; background-color: #fff; margin: 0; border: 1px solid #e9e9e9;" bgcolor="#fff">
						<tr style="font-family: sans-serif; box-sizing: border-box; font-size:18px; margin: 0;">
							<td class="content-wrap aligncenter" style="font-family: sans-serif; box-sizing: border-box; font-size:18px; vertical-align: top; text-align: center; margin: 0; padding: 20px;" align="center" valign="top">
								<table width="100%" cellpadding="0" cellspacing="0" style="font-family: sans-serif; box-sizing: border-box; font-size:18px; margin: 0;">
									<tr style="font-family: Helvetica Neue,Helvetica,Arial,sans-serif; box-sizing: border-box; font-size:18px; margin: 0;">
										<td class="content-block" style="font-family: sans-serif; box-sizing: border-box; font-size:18px; vertical-align: top; margin: 0; padding: 0 0 20px; text-align: center;" valign="top">
											<img class="aligncenter" src="' . WPVUL_PLUGIN_URL . 'includes/assets/logo.png" width="100">
											<h1 class="aligncenter" style="font-family: sans-serif; box-sizing: border-box; font-size: 32px; color: #000; line-height: 1.2em; font-weight: 500; text-align: center; margin: 40px 0 0;" align="center">' . $title . '</h1>
										</td>
									</tr>
									<tr style="font-family: sans-serif; box-sizing: border-box; font-size:18px; margin: 0;">
										<td class="content-block alignlef" style="font-family: sans-serif; box-sizing: border-box; font-size:18px; vertical-align: top; text-align: left; margin: 0; padding: 0 0 20px;" valign="top">';
		$message .= $content;
		$message .= '									</td>
									</tr>
								</table>
								<div class="footer" style="font-family: sans-serif; box-sizing: border-box; font-size:18px; width: 100%; clear: both; color: #999; margin: 0; padding: 20px;">
									<table width="100%" style="font-family: sans-serif; box-sizing: border-box; font-size:18px; margin: 0;">
										<tr style="font-family: sans-serif; box-sizing: border-box; font-size:18px; margin: 0;">
											<td class="aligncenter content-block" style="font-family: sans-serif; box-sizing: border-box; font-size: 12px; vertical-align: top; color: #999; text-align: center; margin: 0; padding: 0 0 20px;" align="center" valign="top">';
		$message .= sprintf(
			// translators: %s the website of Databaase.
			'%s <a href="%s">%s</a>',
			__( 'Learn more about the WordPress Vulnerability Database API at', 'wpvulnerability' ),
			'https://vulnerability.wpsysadmin.com/',
			'WPVulnerability'
		);
		$message .= '											</td>
										</tr>
									</table>
								</div>
							</td>
							<td style="font-family: sans-serif; box-sizing: border-box; font-size:18px; vertical-align: top; margin: 0;" valign="top"></td>
						</tr>
					</table>
				</div>
			</td>
		</tr>
	</table>
</body>
</html>';
		return $message;
	}
}
new WPVUL_Notification();
