<?php
/**
 * Library for admin settings
 *
 * @package    WordPress
 * @author     David Perez <david@closemarketing.es>
 * @copyright  2019 Closemarketing
 * @version    1.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Admin settings
 */
class WPVul_Admin_Settings {
	/**
	 * Settings
	 *
	 * @var array
	 */
	private $wpvulnerability_settings;

	/**
	 * Construct of class
	 */
	public function __construct() {
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
		add_action( 'admin_menu', array( $this, 'add_plugin_page' ) );
		add_action( 'admin_init', array( $this, 'page_init' ) );
		add_action( 'admin_notices', array( $this, 'admin_notices_action' ) );
	}

	/**
	 * Admin Scripts
	 *
	 * @return void
	 */
	public function admin_scripts() {
		wp_enqueue_style(
			'wpvulnerability-admin',
			WPVUL_PLUGIN_URL . 'includes/assets/admin.css',
			array(),
			WPVUL_PLUGIN_VERSION
		);
	}

	/**
	 * Adds plugin page.
	 *
	 * @return void
	 */
	public function add_plugin_page() {
		add_submenu_page(
			'options-general.php',
			__( 'WPVulnerability', 'wpvulnerability' ),
			__( 'WPVulnerability', 'wpvulnerability' ),
			'manage_options',
			'wpvulnerability-options',
			array( $this, 'create_admin_page' )
		);
	}

	/**
	 * Admin notices
	 *
	 * @return void
	 */
	public function admin_notices_action() {
		settings_errors( 'wpvulnerability_notification_error' );
	}

	/**
	 * Create admin page.
	 *
	 * @return void
	 */
	public function create_admin_page() {
		$this->wpvulnerability_settings = get_option( 'wpvulnerability_settings' );
		?>
		<div class="header-wrap">
			<div class="wrapper">
				<h2 style="display: none;"></h2>
				<div id="nag-container"></div>
				<div class="header wpvulnerability-header">
					<div class="logo">
						<img src="<?php echo esc_url( WPVUL_PLUGIN_URL ) . 'includes/assets/logo.png'; ?>" width="50" height="47" />
						<h2><?php esc_html_e( 'WPVulnerability settings', 'wpvulnerability' ); ?></h2>
					</div>
				</div>
			</div>
		</div>

		<div class="wrap">
			<p></p>
			<?php settings_errors(); ?>

			<form method="post" action="options.php">
				<?php
					settings_fields( 'admin_wpvulnerability_settings' );
					do_settings_sections( 'wpvulnerability_settings' );
					submit_button();
				?>
			</form>
		</div>
		<?php
	}

	/**
	 * Init for page
	 *
	 * @return void
	 */
	public function page_init() {

		/**
		 * ## Notification settings
		 * --------------------------- */
		register_setting(
			'admin_wpvulnerability_settings',
			'wpvulnerability_settings',
			array( $this, 'sanitize_fields_api' )
		);

		add_settings_section(
			'admin_wpvulnerability_settings',
			__( 'Receive notifications in your email', 'wpvulnerability' ),
			array( $this, 'admin_section_api_info' ),
			'wpvulnerability_settings'
		);

		add_settings_field(
			'wpvulnerability_emails',
			__( 'Email address to notify (separated by commas)', 'wpvulnerability' ),
			array( $this, 'emails_callback' ),
			'wpvulnerability_settings',
			'admin_wpvulnerability_settings'
		);

		add_settings_field(
			'wpvulnerability_period',
			__( 'How often you want to receive notifications', 'wpvulnerability' ),
			array( $this, 'period_callback' ),
			'wpvulnerability_settings',
			'admin_wpvulnerability_settings'
		);
	}

	/**
	 * # Notification settings
	 * ---------------------------------------------------------------------------------------------------- */

	/**
	 * Sanitize fiels before saves in DB
	 *
	 * @param array $input Input fields.
	 * @return array
	 */
	public function sanitize_fields_api( $input ) {
		$sanitary_values = array();

		if ( isset( $input['emails'] ) ) {
			$sanitary_values['emails'] = sanitize_text_field( $input['emails'] );
		}

		if ( isset( $input['period'] ) ) {
			$sanitary_values['period'] = sanitize_text_field( $input['period'] );
		}

		// Reschedules.
		wp_clear_scheduled_hook( 'wpvulnerability_notification' );
		wp_schedule_event( time(), $sanitary_values['period'], 'wpvulnerability_notification' );

		return $sanitary_values;
	}

	/**
	 * Info for neo automate section.
	 *
	 * @return void
	 */
	public function admin_section_api_info() {
		esc_html_e( 'Configure and save these settings to receive email notifications.', 'wpvulnerability' );
	}

	/**
	 * Username callback
	 *
	 * @return void
	 */
	public function emails_callback() {
		printf(
			'<input class="regular-text" type="text" name="wpvulnerability_settings[emails]" id="wpvulnerability_emails" value="%s"><br><small>%s</small>',
			isset( $this->wpvulnerability_settings['emails'] ) ? esc_attr( $this->wpvulnerability_settings['emails'] ) : '',
			get_bloginfo( 'admin_email' )
		);
	}

	/**
	 * Period of time
	 *
	 * @return void
	 */
	public function period_callback() {
		if ( empty( $this->wpvulnerability_settings['period'] ) ) {
			$this->wpvulnerability_settings['period'] = 'weekly';
		}
		?>
		<select name="wpvulnerability_settings[period]" id="wpvulnerability_period">
			<option value="weekly" <?php selected( $this->wpvulnerability_settings['period'], 'weekly' ); ?>><?php esc_html_e( 'Weekly', 'wpvulnerability' ); ?></option>
			<option value="daily" <?php selected( $this->wpvulnerability_settings['period'], 'daily' ); ?>><?php esc_html_e( 'Daily', 'wpvulnerability' ); ?></option>
		</select>
		<?php
	}
}

new WPVul_Admin_Settings();
