<?php
/**
 * Primary class file for the Health Check plugin.
 *
 * @package Health Check
 */

// Make sure the file is not directly accessible.
if ( ! defined( 'ABSPATH' ) ) {
	die( 'We\'re sorry, but you can not directly access this file.' );
}

/**
 * Class HealthCheck
 */
class Plugins_WPVulnerability {

	/**
	 * Notices to show at the head of the admin screen.
	 *
	 * @access public
	 *
	 * @var array
	 */
	public $admin_notices = array();

	/**
	 * HealthCheck constructor.
	 *
	 * @uses Health_Vulnerability::init()
	 *
	 * @return void
	 */
	public function __construct() {
		add_action( 'admin_head', array( $this, 'admin_head' ) );

		register_activation_hook( WPVUL_PLUGIN_FILE, array( $this, 'on_activation' ) );
		add_action( 'wpvulnerability_pull_db_data_event', array( $this, 'get_installed_plugins' ) );

		register_deactivation_hook( WPVUL_PLUGIN_FILE, array( $this, 'on_deactivation' ) );
		add_filter( 'plugin_action_links_' . WPVUL_PLUGIN_BASE, array( $this, 'add_settings_link' ) );

		add_action( 'activated_plugin', array( $this, 'get_installed_plugins' ), 10, 2 );
		add_action( 'upgrader_process_complete', array( $this, 'get_installed_plugins' ), 10, 2 );
	}

	/**
	 * Admin Head
	 * gets installed plugins cache, adds after row text and notices based on the results for the plugin page
	 *
	 * @return void
	 */
	public function admin_head() {
		global $pagenow;

		$plugins = $this->get_installed_plugins_cache();

		// add after plugin row text.
		foreach ( $plugins as $plugin ) {
			$path         = $plugin['file_path'];
			$added_notice = false;

			if ( isset( $plugin['vulnerable'] ) && 'true' === $plugin['vulnerable'] ) {
				add_action( 'after_plugin_row_' . $path, array( $this, 'after_row_text' ), 10, 3 );

				if ( ! $added_notice ) {
					add_action( 'admin_notices', array( $this, 'vulnerable_admin_notice' ) );
					$added_notice = true;
				}
			}
		}
	}

	/**
	 * Get Installed plugins cache
	 *
	 * @return array
	 */
	public function get_installed_plugins_cache() {

		$plugin_data = json_decode( get_option( 'wpvulnerability-data' ) );
		if ( ! empty( $plugin_data ) ) {

			if ( ! function_exists( 'get_plugins' ) ) {
					require_once ABSPATH . 'wp-admin/includes/plugin.php';
			}

			$plugins = json_decode( get_option( 'wpvulnerability-data' ), true );

			foreach ( $plugins as $key => $plugin ) {
				$plugin          = $this->get_cached_plugin_vulnerabilities( $plugin, $key );
				$plugins[ $key ] = $plugin;
			}
			return $plugins;
		} else {
			$this->get_installed_plugins();
		}
	}

	/**
	 * On Activation
	 * callback function for when the plugin is activated
	 * add plugin data option if it isn't created already, schedule wp-cron job
	 */
	public function on_activation() {

		if ( ! get_option( 'wpvulnerability-data' ) ) {
			add_option( 'wpvulnerability-data', '' );
		}

		wp_schedule_event( time(), 'twicedaily', 'wpvulnerability_pull_db_data_event' );

		$this->get_installed_plugins( false );
	}

	/**
	 * On Deactivation
	 * callback function when a plugin is deactivated
	 * delete, option and remove wp-cron job
	 */
	public function on_deactivation() {
		delete_option( 'wpvulnerability-data' );
		wp_clear_scheduled_hook( 'wpvulnerability_pull_db_data_event' );
	}

	/**
	 * Get Cached Plugin Vulnerabilities
	 * pulls installed plugins, compares version to cached vulnerabilities, adds vulnerable key to plugin.
	 *
	 * @param array  $plugin Plugin name.
	 * @param string $file_path plugin file path.
	 * @return array updated plugin array.
	 */
	public function get_cached_plugin_vulnerabilities( $plugin, $file_path ) {

		global $installed_plugins;

		if ( ! is_array( $installed_plugins ) ) {
			if ( ! function_exists( 'get_plugins' ) ) {
				require_once ABSPATH . 'wp-admin/includes/plugin.php';
			}
			$installed_plugins = get_plugins();
		}

		$plugin = $this->set_text_domain( $plugin );

		if ( isset( $installed_plugins[ $file_path ]['Version'] ) ) {
			$plugin['Version']    = $installed_plugins[ $file_path ]['Version'];
			$plugin['vulnerable'] = 'false';

			if ( ! empty( $plugin['vulnerabilities'] ) ) {
				$plugin['vulnerable'] = 'true';
			}
		}
		$plugin['file_path'] = $file_path;
		return $plugin;
	}

	/**
	 * Get Fresh Plugin Vulnerabilities
	 * pull vulnerabilities through API, compare version to vulnerabilities, add is_know_vulnerable key
	 *
	 * @param  array  $plugin_data Plugin data.
	 * @param  string $file_path plugin file path.
	 * @return array  updated plugin
	 */
	public function get_fresh_plugin_vulnerabilities( $plugin_data, $file_path ) {
		$plugin_data['file_path']  = $file_path;
		$plugin_data               = $this->set_text_domain( $plugin_data );
		$plugin_slug               = $plugin_data['TextDomain'];
		$plugin_data['vulnerable'] = 'false';

		$plugin_vuls = wpvulnerability_get_plugin( $plugin_slug, $plugin_data['Version'] );
		if ( ! empty( $plugin_vuls ) ) {
			$plugin_data['vulnerabilities'] = $plugin_vuls;
			$plugin_data['vulnerable']      = 'true';
		}

		return $plugin_data;
	}

	/**
	 * Set Text Domain
	 * sets the text domain to the TextDomain key if it is not set
	 *
	 * @param  array $plugin Plugin name.
	 * @return array updated plugin
	 */
	public function set_text_domain( $plugin ) {

		// get text domain from folder if it isn't listed.
		if ( isset( $plugin['file_path'] ) ) {
			$folder_name          = explode( '/', $plugin['file_path'] );
			$plugin['TextDomain'] = $folder_name[0];
		}

		return $plugin;
	}

	/**
	 * Get Installed Plugins
	 * gets the installed plugins, checks for vulnerabilities in them, caches the data, sends email if vulnerabilities detected
	 *
	 * @return array
	 */
	public function get_installed_plugins() {

		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$plugins      = get_plugins();
		$vuln_plugins = array();
		foreach ( $plugins as $key => $plugin ) {
			$plugin          = $this->get_fresh_plugin_vulnerabilities( $plugin, $key );
			$plugins[ $key ] = $plugin;

			if ( isset( $plugin['vulnerable'] ) && 'true' === $plugin['vulnerable'] ) {
				$vuln_plugins[] = $plugin['Name'];
			}
		}

		update_option( 'wpvulnerability-data', wp_json_encode( $plugins ) );

		return $plugins;
	}

	/**
	 * Add Settings Link
	 *
	 * @param array $links links that appear in the plugin row.
	 */
	public function add_settings_link( $links ) {
		// $links[] = '<a href="' . get_admin_url( null, 'options-general.php?page=wpvulnerability-settings' ) . '">' . __( 'Settings', 'wpvulnerability' ) . '</a>';
		return $links;
	}

	/**
	 * After Row Text
	 * callback function for adding vulnerability notice under vulnerable plugins
	 *
	 * @param  string $plugin_file main plugin folder/file_name.
	 * @param  array  $plugin_data Plugin data.
	 * @param  array  $status Status.
	 * @return void
	 */
	public function after_row_text( $plugin_file, $plugin_data, $status ) {

		global $wpvulnerability_data;

		$tr_class = '';
		if ( is_plugin_active( $plugin_data['plugin'] ) ) {
			$tr_class .= 'active';
		}

		if ( ! is_array( $wpvulnerability_data ) ) {
			$wpvulnerability_data = json_decode( get_option( 'wpvulnerability-data' ), true );
		}

		$message = sprintf(
			/* translators: 1: plugin name */
			__( '%1$s has a known vulnerability that may be affecting this version.', 'wpvulnerability' ),
			$plugin_data['Name']
		);

		$string  = '<tr class="wpvulnerability ' . $tr_class . '">';
		$string .= '<td colspan="4">';
		$string .= '<p class="text-red"><strong>' . $message . '</strong>';
		$string .= '</p>';
		$string .= '<table>';

		$vulnerabilities = $this->get_cached_plugin_vulnerabilities( $wpvulnerability_data[ $plugin_file ], $plugin_file );
		foreach ( $vulnerabilities['vulnerabilities'] as $vulnerability ) {
			$source  = implode( '<br>', array_column( $vulnerability['source'], 'name' ) );
			$string .= '<tr>';
			$string .= '<td><strong>' . esc_html( $vulnerability['name'] ) . '</strong></td>';

			$string .= '<td>';
			if( $vulnerability['closed'] ) {
				$string .= '<span class="text-red">' . __( 'This plugin is closed. Please replace it with another.', 'wpvulnerability' ) . '</span><br>';
			}
			if( $vulnerability['unfixed'] ) {
				$string .= '<span class="text-red">' . __( 'This vulnerability appears to be unpatched. Stay tuned for upcoming plugin updates.', 'wpvulnerability' ) . '</span><br>';
			}
			$string .= esc_html( $source ) . '</td>';
			$string .= '</tr>';
		}
		$string .= '</table>';
		$string .= '</td>';
		$string .= '</tr>';

		echo $string; // phpcs:ignore
	}

	/**
	 * Vulnerable Admin Notice
	 * prints out error message if plugin(s) is/are vulnerable
	 */
	public function vulnerable_admin_notice() {
		sprintf(
			// translators: Dismissible message.
			'<div class="notice notice-error is-dismissible"><strong>%s</strong>: %s</div>',
			'WPVulnerability',
			__('There are possible vulnerabilities in your installation. Please, check your WordPress, plugins, and themes.', 'wpvulnerability' )
		);
	}
}
